import re
from os import listdir
from os.path import isdir, join
from pathlib import Path
from typing import Any, Dict, Union, cast

import axipy

from .parser import MANIFEST_FILE_NAME, IParser

# noinspection PyUnresolvedReferences
langs = "|".join(elem.value for elem in axipy.AxiomaLanguage)
re_langs = re.compile(rf".*_({langs})\.qch$", re.RegexFlag.IGNORECASE)


class FolderParser(IParser):
    def parse(self, location_path: Union[str, Path]) -> dict:
        if isinstance(location_path, Path):
            location_path = str(location_path)

        manifest_filepath = join(location_path, MANIFEST_FILE_NAME)
        result: Dict[str, Any]
        with open(manifest_filepath, encoding="utf-8") as manifest_file:
            result = self.read_config(manifest_file)
        doc_dir = join(location_path, "documentation")
        if isdir(doc_dir):

            def is_doc_entry(filename: str) -> bool:
                return filename.startswith("index") and filename.endswith("html")

            doc_entry_points = filter(is_doc_entry, listdir(doc_dir))
            result["documentation"] = [join(doc_dir, entry) for entry in doc_entry_points]

        def is_qthelp(filename: str) -> bool:
            return filename.endswith(".qch")

        def detect_lang(filename: str) -> str:
            match = re_langs.match(filename)
            if match:
                return match.group(1)
            return "default"

        def get_qthelp_dict_from_dirs(*dirs_to_search: str) -> Dict[str, str]:
            qthelp_dict = {}
            for dir_arg in filter(isdir, dirs_to_search):
                for entry in filter(is_qthelp, sorted(listdir(dir_arg))):
                    k = detect_lang(entry)
                    v = join(cast(str, dir_arg), cast(str, entry))
                    if k in qthelp_dict:
                        # If there are multiple qch files for the same language, load none of them, to indicate
                        # user, of possible undefined behaviour.
                        return {}
                    else:
                        qthelp_dict[k] = v
            return qthelp_dict

        result["qthelp"] = get_qthelp_dict_from_dirs(location_path, doc_dir)

        return result

    @staticmethod
    def supports(location_path: Union[str, Path]) -> bool:
        return True
