import logging
import os
import platform
import subprocess
from pathlib import Path
from typing import Any, Dict, List, Optional, Union, cast

from .dependency_installer import Dependencies, DependencyInstaller, InstallError

__all__: List[str] = [
    "PipDependencyInstaller",
]


def python_interpreter() -> Optional[Path]:
    result = Path(os.environ["AXIOMA_PYTHON_INTERPRETER"])
    return result if result.is_file() else None


class PipDependencyInstaller(DependencyInstaller):

    @property
    def __flags(self) -> Dict[str, Any]:
        flags = {}
        if platform.system() == "Windows":
            flags["creationflags"] = subprocess.CREATE_NO_WINDOW  # type: ignore[attr-defined]

        return flags

    @property
    def __executable(self) -> Path:
        executable = python_interpreter()
        if not executable:
            raise InstallError("Python interpreter was not found.")
        return executable

    def install_by_name(self, package: Union[str, Path], dependencies_destination: Path) -> None:
        package_str: str = str(package)
        # check for empty strings
        if not package_str or len(package_str.split()) == 0:
            return None

        p = dependencies_destination.absolute().as_posix()
        logging.info(f"Dependency catalog: '{p}'.")
        logging.info(f"--- Package deps: '{package_str}' ---")
        args = [str(self.__executable), "-m", "pip", "install", f"--target={p}"]
        if Path(package).suffix == ".whl":
            args.append("--no-deps")
            args.append("--no-index")
            args.append("--force-reinstall")
        args.append(package_str)

        try:
            exit_data = subprocess.run(
                args,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                **self.__flags,
                check=True,
            )
            exit_string = exit_data.stdout if exit_data.returncode == 0 else exit_data.stderr
            exit_string = cast(bytes, exit_string)
            for m in Dependencies._to_list(exit_string.decode()):
                if m:
                    logging.info(m)
            if exit_data.returncode != 0:
                raise InstallError("Error while installing dependent packages.")
        except subprocess.CalledProcessError as e:
            err_msg = f"Error while installing dependent package '{package}':\n{e.stderr.decode('utf8')}"
            logging.error(err_msg)
            raise InstallError(err_msg)
        except Exception as e:
            raise e

    def install(self, dependency: Dependencies, dependencies_destination: Path) -> None:
        for package in dependency.packages():
            self.install_by_name(package, dependencies_destination)

    def download(self, requirements_file: Union[str, Path], dest_folder: Union[str, Path]) -> None:
        requirements_file = str(requirements_file)
        dest_folder = str(dest_folder)
        logging.info(f"Download catalog: '{dest_folder}'.")
        try:
            args = [str(self.__executable), "-m", "pip", "download"]
            if requirements_file.lower().endswith(".whl"):
                args.extend([requirements_file])
            else:
                args.extend(["-r", requirements_file])
            args.extend(["-d", dest_folder])
            exit_data = subprocess.run(
                args,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                **self.__flags,
                check=True,
            )
            exit_string = exit_data.stdout if exit_data.returncode == 0 else exit_data.stderr
            exit_string = cast(bytes, exit_string)
            for m in Dependencies._to_list(exit_string.decode()):
                if m:
                    logging.info(m)
            if exit_data.returncode != 0:
                raise InstallError("Error while downloading dependent packages.")
        except subprocess.CalledProcessError as e:
            err_msg = f"Error while downloading dependencies '{requirements_file}'\n{e.stderr.decode('utf8')}"
            logging.error(err_msg)
            raise InstallError(err_msg)
        except Exception as e:
            raise e
