from __future__ import annotations

from collections.abc import Iterable, Collection
from enum import auto, Enum
from pathlib import Path
from typing import Iterator, TYPE_CHECKING, Optional, Generator, Any

import axipy
from PySide2.QtCore import Qt, Slot, QLocale
from PySide2.QtGui import QIcon, QKeyEvent
from PySide2.QtWidgets import (QFileDialog, QDialog, QDialogButtonBox, QWidget, QPushButton,
                               QAbstractItemView)
from axipy._internal._decorator import _try_except_silent
from .bounds import _TabFilesResultBoundsCalculator
from .dialog_choose_opened_tables import DialogChooseOpenedTables
from .ui import Ui_ExportToFileDialog
from .utils import is_tab_table
from .worker import ExportToFileWorker

if TYPE_CHECKING:
    from .__init__ import ExportToFilePlugin


class _CalcTask(axipy.DialogTask):
    def __init__(self) -> None:
        super().__init__(self._calculate, cancelable=True)

    def _paths_callback(self, tab_files: Collection[Path]) -> Generator[Path]:
        self.title = "Расчет оптимального охвата"
        self.message = self.title

        self.max = len(tab_files)

        for path in tab_files:
            self.raise_if_canceled()
            self.value += 1
            yield path

    def _calculate(
            self,
            tab_files: Collection[Path],
            inp_cs: Optional[axipy.CoordSystem] = None,
            out_cs: Optional[axipy.CoordSystem] = None,
    ) -> axipy.Rect:
        tab_files = self._paths_callback(tab_files)
        calc = _TabFilesResultBoundsCalculator(tab_files, inp_cs, out_cs)
        return calc.calculate()

    def run_and_get(
            self,
            tab_files: Collection[Path],
            inp_cs: Optional[axipy.CoordSystem] = None,
            out_cs: Optional[axipy.CoordSystem] = None,
    ) -> axipy.Rect:
        return super().run_and_get(tab_files, inp_cs, out_cs)


class ExportToFileDialog(QDialog, Ui_ExportToFileDialog):
    class InitialCsState(Enum):
        UNKNOWN = auto()
        DIFFERENT = auto()

    def __init__(self, plugin: 'ExportToFilePlugin') -> None:
        super().__init__(axipy.view_manager.global_parent)
        self.plugin: 'ExportToFilePlugin' = plugin
        self.setupUi(self)
        self.setAttribute(Qt.WA_DeleteOnClose, True)

        self.__inp_cs_initial: axipy.CoordSystem | ExportToFileDialog.InitialCsState = self.InitialCsState.UNKNOWN
        self.__inp_cs_override: axipy.CoordSystem | None = None
        self.__out_cs: axipy.CoordSystem | None = None
        self.__out_cs_rect: axipy.Rect | None = None
        self.__task: ExportToFileWorker | None = None
        self.__dialog_choose_opened_tables_result = DialogChooseOpenedTables.Result()

        current_cs = axipy.CoordSystem.current()
        self.inp_cs_override = current_cs
        self.out_cs = current_cs
        self.out_cs_rect = current_cs.rect

        self.pbar_features.hide()

        self.tb_select_files.setIcon(QIcon.fromTheme("open"))
        self.tb_select_files.clicked.connect(self.slot_on_open_files_clicked)

        self.tb_choose_opened.setIcon(axipy.action_manager.icons["sql_tables"])
        self.tb_choose_opened.clicked.connect(self.slot_on_tb_choose_opened_clicked)
        en = any(filter(is_tab_table, axipy.data_manager.tables))
        self.tb_choose_opened.setEnabled(en)

        self.tb_delete_selected.setIcon(QIcon.fromTheme("delete"))
        self.tb_delete_selected.clicked.connect(self.slot_on_delete_selected_clicked)

        self.list_source_tables.setSelectionMode(QAbstractItemView.SelectionMode.ExtendedSelection)
        self.list_source_tables.itemSelectionChanged.connect(self.slot_list_tables_selection_changed)
        self.list_source_tables.paths_dropped.connect(self.slot_on_paths_dropped)

        if self.plugin.last_save_path:
            last_save_path = self.plugin.last_save_path
        else:
            last_save_path = self._get_existing_last_save_path()
        self.le_out_folder.setText(str(last_save_path))

        self.tb_choose_out_folder.clicked.connect(self.slot_choose_out_folder)

        self.pb_inp_cs.clicked.connect(self.slot_inp_cs_clicked)
        self.pb_out_cs.clicked.connect(self.slot_out_cs_clicked)

        self.pb_out_bounds.clicked.connect(self.slot_on_pb_out_bounds_clicked)

        self.gb_inp_cs.clicked.connect(self.slot_gb_inp_cs_clicked)
        self.cb_preserve_geometry.clicked.connect(self.slot_cb_preserve_geometry_clicked)

        self.pb_calculate_out_rect.clicked.connect(self.slot_pb_calculate_out_rect_clicked)
        self.pb_calculate_out_rect.setEnabled(False)
        self.pb_restore_out_rect.clicked.connect(self.slot_pb_restore_out_rect_clicked)

        self.__restore_settings()

        self._ui_ok.setText(self.plugin.tr("Экспорт"))
        self._ui_ok.setEnabled(False)

        self._ui_cancel.setText(self.plugin.tr("Закрыть"))
        self._ui_cancel.clicked.connect(self.slot_cancel_active_task)

    @property
    def _ui_ok(self) -> QPushButton:
        return self.buttonBox.button(QDialogButtonBox.Ok)

    @property
    def _ui_cancel(self) -> QPushButton:
        return self.buttonBox.button(QDialogButtonBox.Cancel)

    def keyPressEvent(self, event: QKeyEvent) -> None:
        if event.key() == Qt.Key_Delete:
            self.slot_on_delete_selected_clicked()
        super().keyPressEvent(event)

    def accept(self) -> None:
        self._export_process_prepare()
        self.__save_settings()

    def reject(self) -> None:
        # Try cancel.
        if (
                self.__task is not None and
                not self.__task.finished and
                not self.__task.is_canceled
        ):
            self.__task.cancel()
            return None

        # If no active tasks, reject.
        if self.__task is None or self.__task.finished:
            super().reject()

    def _set_enabled_pb_calculate_out_rect(self, is_enabled: bool) -> None:
        if self.list_source_tables.count() == 0 or self.cb_preserve_geometry.isChecked():
            is_enabled = False
        self.pb_calculate_out_rect.setEnabled(is_enabled)

    @Slot()
    def slot_cancel_active_task(self) -> None:
        if self.__task is not None:
            self.__task.cancel()

    @Slot()
    def slot_cb_preserve_geometry_clicked(self, checked: bool) -> None:
        if checked:
            self.gb_inp_cs.setChecked(False)
        self._check_pb_bounds_state()

    @Slot(bool)
    def slot_gb_inp_cs_clicked(self, checked: bool) -> None:
        if checked:
            self.cb_preserve_geometry.setChecked(False)
        self._check_pb_bounds_state()

    @Slot()
    def slot_list_tables_selection_changed(self) -> None:
        self.tb_delete_selected.setEnabled(bool(len(self.list_source_tables.selectedItems())))

    @Slot()
    def slot_inp_cs_clicked(self) -> None:
        self.__cs_clicked(is_input=True)

    @Slot()
    def slot_out_cs_clicked(self) -> None:
        self.__cs_clicked(is_input=False)

    def __cs_clicked(self, is_input: bool) -> None:
        if is_input:
            cs = self.inp_cs_override
        else:
            cs = self.out_cs

        result_cs = axipy.prompt_coordsystem(cs)
        if not result_cs:
            return None

        if is_input:
            self.inp_cs_override = result_cs
        else:
            self.out_cs = result_cs
            if not self.cb_preserve_geometry.isChecked() and result_cs.non_earth:
                self.choose_out_cs_rect_dialog()

    @property
    def inp_cs_initial(self) -> axipy.CoordSystem | InitialCsState:
        return self.__inp_cs_initial

    @inp_cs_initial.setter
    def inp_cs_initial(self, value: axipy.CoordSystem | InitialCsState) -> None:
        self.__inp_cs_initial = value
        if isinstance(value, axipy.CoordSystem):
            text = value.name
        elif value == self.InitialCsState.DIFFERENT:
            text = self.plugin.tr("Проекции различаются")
        else:
            text = ""
        self.le_inp_files_cs.setText(text)

    @property
    def inp_cs_override(self) -> axipy.CoordSystem:
        return self.__inp_cs_override

    @inp_cs_override.setter
    def inp_cs_override(self, cs: axipy.CoordSystem) -> None:
        self.__inp_cs_override = cs
        self.pb_inp_cs.setText(cs.name)

    @property
    def out_cs(self) -> axipy.CoordSystem:
        return self.__out_cs

    @out_cs.setter
    def out_cs(self, cs: axipy.CoordSystem) -> None:
        self.__out_cs = cs
        self.out_cs_rect = cs.rect
        self.pb_out_cs.setText(cs.name)

    @property
    def out_cs_rect(self) -> axipy.Rect:
        return self.__out_cs_rect.clone()

    @out_cs_rect.setter
    def out_cs_rect(self, rect: axipy.Rect) -> None:
        self.__out_cs_rect = rect.clone()
        self.pb_out_bounds.setText(self._format_rect_as_string(self.__out_cs_rect))

    def _format_rect_as_string(self, rect: axipy.Rect) -> str:
        params = {"precision": 6, "locale": QLocale.English}
        x_min = axipy.FloatCoord(rect.xmin).as_string(**params)
        y_min = axipy.FloatCoord(rect.ymin).as_string(**params)
        x_max = axipy.FloatCoord(rect.xmax).as_string(**params)
        y_max = axipy.FloatCoord(rect.ymax).as_string(**params)
        return f"({x_min}, {y_min}, {x_max}, {y_max})"

    def _check_pb_bounds_state(self) -> None:
        is_checked = self.cb_preserve_geometry.isChecked()
        is_enabled = not is_checked
        for pb in (self.pb_out_bounds, self.pb_calculate_out_rect, self.pb_restore_out_rect):
            if pb == self.pb_calculate_out_rect:
                self._set_enabled_pb_calculate_out_rect(is_enabled)
            else:
                pb.setEnabled(is_enabled)

    def _check_enabled_export(self) -> None:
        if (
                self.list_source_tables.count() > 0 and
                self.le_out_folder.text()
        ):
            enabled = True
        else:
            enabled = False

        self._set_enabled_pb_calculate_out_rect(self.list_source_tables.count() > 0)
        self._ui_ok.setEnabled(enabled)

    def __update_initial_cs(self) -> None:
        paths = self._read_inp_paths()
        self.list_source_tables.clear()
        self.inp_cs_initial = self.InitialCsState.UNKNOWN
        self._add_file_names(paths)

    def _read_inp_paths(self) -> list[Path]:
        list_widget = self.list_source_tables
        return list(Path(list_widget.item(i).text()) for i in range(list_widget.count()))

    @Slot()
    def slot_on_open_files_clicked(self) -> None:
        result = axipy.open_files_dialog(
            title="Выбор файлов для экспорта",
            filter_arg="Файлы TAB (*.tab *.TAB *.Tab)",
        )
        if not result:
            return None

        file_names: tuple[Path, ...] = result
        self._add_file_names(file_names)

    def _add_file_names(self, file_names: Iterable[Path]) -> None:
        path_list = self._read_inp_paths()
        target_cs = self.inp_cs_initial
        for fn in file_names:
            if fn not in path_list:

                if target_cs != self.InitialCsState.DIFFERENT:
                    src = axipy.provider_manager.tab.get_source(str(fn))
                    table, close_needed = axipy.provider_manager._open_hidden_check_if_close_needed(src)
                    try:
                        table_cs: axipy.CoordSystem = table.coordsystem
                        if target_cs == self.InitialCsState.UNKNOWN:
                            target_cs = table_cs
                        elif not target_cs.is_equal(table_cs, ignore_rect=True):
                            target_cs = self.InitialCsState.DIFFERENT
                    finally:
                        if close_needed:
                            table.close()

                self.list_source_tables.addItem(str(fn))

        self.inp_cs_initial = target_cs
        self._check_enabled_export()

    @Slot()
    def slot_on_delete_selected_clicked(self) -> None:
        selected_items = self.list_source_tables.selectedItems()
        for item in selected_items:
            self.list_source_tables.takeItem(self.list_source_tables.row(item))

        if self.list_source_tables.count() == 0:
            self.inp_cs_initial = self.InitialCsState.UNKNOWN
        else:
            self.__update_initial_cs()

        self._check_enabled_export()

    def _get_existing_last_save_path(self) -> Path:
        last_save_path = axipy.CurrentSettings.LastSavePath
        if not Path(last_save_path).exists():
            last_save_path = axipy.DefaultSettings.LastSavePath
        return last_save_path

    @Slot()
    def slot_choose_out_folder(self) -> None:
        last_save_path = self._get_existing_last_save_path()
        out_folder: str = QFileDialog.getExistingDirectory(
            self,
            caption=axipy.tr("Выбор выходной папки"),
            dir=str(last_save_path),
            options=QFileDialog.ShowDirsOnly,
        )
        if not out_folder:
            return None
        self.le_out_folder.setText(out_folder)
        self.plugin.last_save_path = Path(out_folder)
        axipy.CurrentSettings.LastSavePath = Path(out_folder)

        self._check_enabled_export()

    def _toggle_ui(self, state: bool) -> None:
        objects = self.children()
        widgets: Iterator[QWidget] = filter(lambda x: isinstance(x, QWidget), objects)
        for widget in widgets:
            if widget in (self.buttonBox, self.pbar_features):
                continue
            widget.setEnabled(state)

        self._ui_ok.setEnabled(state)

    def _export_process_prepare(self) -> None:
        out_folder: Path = Path(self.le_out_folder.text())
        if not out_folder.exists():
            axipy.show_message(
                title=self.plugin.tr("Ошибка"),
                text=self.plugin.tr("Выбранной выходной папки не существует, пожалуйста, выберите другую папку."),
                icon=axipy.DlgIcon.ERROR,
            )
            return None

        self._toggle_ui(False)
        self._ui_cancel.setText(self.plugin.tr("Отмена"))

        self.pbar_features.show()

        override_inp_cs = self.gb_inp_cs.isChecked()
        preserve_geometry = self.cb_preserve_geometry.isChecked()
        inp_cs_override: axipy.CoordSystem | None = self.__inp_cs_override if override_inp_cs else None

        # Clone cs with new rect as tmp cs for export process
        out_cs = self.out_cs.clone()
        out_cs.rect = self.out_cs_rect
        self.__task = ExportToFileWorker(
            self.plugin,
            paths=self._read_inp_paths(),
            out_path=out_folder,
            preserve_geometry=preserve_geometry,
            inp_cs_override=inp_cs_override,
            out_cs=out_cs
        )
        self.__task.range_changed.connect(self.slot_pbar_set_range)
        self.__task.value_changed.connect(self.slot_pbar_set_value)
        self.__task.finished.connect(self.slot_on_finished)

        self.__task.start()

    @Slot(axipy.Task.Range)
    def slot_pbar_set_range(self, task_range: axipy.Task.Range) -> None:
        self.pbar_features.setRange(task_range.min, task_range.max)

    @Slot(float)
    def slot_pbar_set_value(self, value: float) -> None:
        self.pbar_features.setValue(value)

    @Slot()
    def slot_on_finished(self) -> None:
        self.__task = None

        self.pbar_features.hide()
        self._ui_cancel.setText(self.plugin.tr("Закрыть"))
        self._toggle_ui(True)
        self._check_pb_bounds_state()
        self._check_enabled_export()

    @Slot(list)
    def slot_on_paths_dropped(self, paths: list[Path]) -> None:
        self._add_file_names(paths)

    @Slot()
    def slot_on_pb_out_bounds_clicked(self) -> None:
        self.choose_out_cs_rect_dialog()

    @Slot()
    def slot_on_tb_choose_opened_clicked(self) -> None:
        dialog = DialogChooseOpenedTables(self.plugin, self.__dialog_choose_opened_tables_result)
        dialog.finished.connect(self.slot_on_dialog_choose_opened_tables_finished)
        dialog.open()

    @Slot()
    def slot_on_dialog_choose_opened_tables_finished(self) -> None:
        tab_files: list[Path] | None = self.__dialog_choose_opened_tables_result.tab_files
        if tab_files:
            self._add_file_names(tab_files)

    def choose_out_cs_rect_dialog(self) -> None:
        result = axipy.prompt_rect(self.out_cs_rect, self.out_cs.unit)
        if result:
            self.out_cs_rect = result

    @Slot()
    def slot_pb_calculate_out_rect_clicked(self) -> None:
        t = _CalcTask()
        inp_cs = self.__inp_cs_override if self.gb_inp_cs.isChecked() else None
        self.out_cs_rect = t.run_and_get(self._read_inp_paths(), inp_cs, self.out_cs)

    @Slot()
    def slot_pb_restore_out_rect_clicked(self) -> None:
        self.out_cs_rect = self.out_cs._default_rect

    @_try_except_silent
    def __save_settings(self) -> None:
        s = self.plugin.settings
        s.setValue("gb_inp_cs_isChecked", self.gb_inp_cs.isChecked())
        s.setValue("inp_cs_override_to_string", self.inp_cs_override.to_string())
        s.setValue("cb_preserve_geometry_isChecked", self.cb_preserve_geometry.isChecked())
        s.setValue("out_cs_to_string", self.out_cs.to_string())
        s.setValue("out_cs_rect_to_qt", self.out_cs_rect.to_qt())

    def __restore_single(self, func, settings_key: str, value_type: Optional[type] = None) -> Any | None:
        s = self.plugin.settings
        if value_type:
            v = s.value(settings_key, None, value_type)
        else:
            v = s.value(settings_key, None)
        if v is not None:
            return func(v)

    @_try_except_silent
    def __restore_settings(self) -> None:
        self.__restore_single(self.gb_inp_cs.setChecked, "gb_inp_cs_isChecked", bool)
        self.slot_gb_inp_cs_clicked(self.gb_inp_cs.isChecked())

        inp_cs_override = self.__restore_single(axipy.CoordSystem.from_string, "inp_cs_override_to_string")
        if isinstance(inp_cs_override, axipy.CoordSystem):
            self.inp_cs_override = inp_cs_override

        self.__restore_single(self.cb_preserve_geometry.setChecked, "cb_preserve_geometry_isChecked", bool)
        self.slot_cb_preserve_geometry_clicked(self.cb_preserve_geometry.isChecked())

        out_cs = self.__restore_single(axipy.CoordSystem.from_string, "out_cs_to_string")
        if isinstance(out_cs, axipy.CoordSystem):
            self.out_cs = out_cs

        out_cs_rect = self.__restore_single(axipy.Rect.from_qt, "out_cs_rect_to_qt")
        if out_cs_rect:
            self.out_cs_rect = out_cs_rect
