from __future__ import annotations

import collections
from collections.abc import Iterator
from typing import Iterable, TYPE_CHECKING, Generator, Final, cast, overload

import axipy
from PySide2.QtCore import Qt, Slot, QModelIndex, QPoint, QLine, QSize, QItemSelectionModel, QRect
from PySide2.QtGui import QPaintEvent, QPainter, QPen, QMouseEvent, QKeyEvent, QIcon
from PySide2.QtWidgets import QWidget, QToolButton
from axipy.gui.map_tool import SelectToolHelpers
from .observer import CollectionOrderEditorObserver
from .tree_model import TreeModel, TreeItem
from .ui import Ui_Form
from .utils import ICON_PATH

if TYPE_CHECKING:
    from .__init__ import CollectionOrderEditor

HALF_LINE_SIZE: Final[int] = 5
LINE_SIZE: Final[int] = HALF_LINE_SIZE * 2

SMALL_CROSS_SIZE: Final[float] = 3.0
BIG_CROSS_SIZE: Final[float] = 5.0


def _fix_cs_from_selection_table_and_map_view(scene: axipy.Pnt | axipy.Rect) -> axipy.Pnt | axipy.Rect:
    selection = axipy.data_manager.selection
    if selection is not None:
        map_view = CollectionOrderEditorObserver.active_map_view()
        cs1 = map_view.coordsystem
        cs2 = selection.coordsystem
        if cs1 != cs2:
            transformer = axipy.CoordTransformer(cs2, cs1)
            try:
                scene = transformer.transform(scene)
            except Exception:
                pass
    return scene


class CollectionOrderEditorWidget(QWidget, Ui_Form):
    def __init__(self, plugin: 'CollectionOrderEditor') -> None:
        super().__init__()
        self.setupUi(self)

        self._plugin = plugin

        self.tree_view.setDragEnabled(True)
        self.tree_view.setAcceptDrops(True)
        self.tree_view.setDropIndicatorShown(True)

        self.__set_icon_from_name(self.tb_down, "arrow_down")
        self.__set_icon_from_name(self.tb_up, "arrow_up")
        self.tb_down_all.setIcon(self.__create_icon("down_a.png"))
        self.tb_up_all.setIcon(self.__create_icon("up_a.png"))
        self.__set_icon_from_name(self.tb_show_all, "search")

        self.tb_down.clicked.connect(self.slot_down)
        self.tb_up.clicked.connect(self.slot_up)
        self.tb_down_all.clicked.connect(self.slot_down_all)
        self.tb_up_all.clicked.connect(self.slot_up_all)
        self.tb_show_all.clicked.connect(self.slot_show_all)

        self.check_buttons_enabled()

    def __create_icon(self, file_name: str) -> QIcon:
        icon_path = ICON_PATH / "16px" / file_name
        icon = QIcon()
        icon.addFile(str(icon_path), QSize(16, 16))
        return icon

    def __set_icon_from_name(self, tb: QToolButton, name: str) -> None:
        icon = axipy.action_manager.icons.get(name, None)
        if icon:
            tb.setIcon(icon)
        else:
            tb.setText(name)

    def __check_buttons_enabled_if_tree_item(self, tree_item: TreeItem) -> None:
        down = up = down_all = up_all = False

        if tree_item.data() in (
                CollectionOrderEditorMapTool.GEOMETRY_COLLECTION_POLYGONS_NAME,
                CollectionOrderEditorMapTool.GEOMETRY_COLLECTION_LINE_STRINGS_NAME,
                CollectionOrderEditorMapTool.GEOMETRY_COLLECTION_POINTS_NAME,
        ):
            down = up = down_all = up_all = show_all = False
            self.__set_buttons_enabled(down, up, down_all, up_all, show_all)
            return None

        if not axipy.ObserverManager.SelectionEditableIsSame.value:
            down = up = down_all = up_all = False
            show_all = True
            self.__set_buttons_enabled(down, up, down_all, up_all, show_all)
            return None

        show_all = True

        count: int = tree_item.parent().childCount()

        if count > 1:
            row: int = tree_item.childNumber()
            if row != 0:
                up = up_all = True
            if row != count - 1:
                down = down_all = True

        self.__set_buttons_enabled(down, up, down_all, up_all, show_all)

    @Slot()
    def check_buttons_enabled(self) -> None:
        if self.selection_model:
            tree_item = self.selection_model.currentIndex().internalPointer()
            if isinstance(tree_item, TreeItem):
                self.__check_buttons_enabled_if_tree_item(tree_item)
                return None

        down = up = down_all = up_all = show_all = False
        self.__set_buttons_enabled(down, up, down_all, up_all, show_all)

    def __set_buttons_enabled(self, down, up, down_all, up_all, show_all) -> None:
        self.tb_down.setEnabled(down)
        self.tb_up.setEnabled(up)
        self.tb_down_all.setEnabled(down_all)
        self.tb_up_all.setEnabled(up_all)
        self.tb_show_all.setEnabled(show_all)

    @property
    def selection_model(self) -> QItemSelectionModel | None:
        return self.tree_view.selectionModel()

    def __get_parent_and_row(self) -> tuple[QModelIndex, int]:
        index = self.selection_model.currentIndex()
        return index.parent(), index.row()

    def __notify_failed_to_move(self) -> None:
        axipy.Notifications.push(
            self._plugin.title,
            self._plugin.tr("Не удалось переместить элемент."),
            axipy.Notifications.Critical
        )

    @Slot()
    def slot_down(self) -> None:
        parent, row = self.__get_parent_and_row()
        result = self.tree_view.model().moveRow(
            parent,
            row,
            parent,
            row + 1,
        )
        if not result:
            self.__notify_failed_to_move()

    @Slot()
    def slot_up(self) -> None:
        parent, row = self.__get_parent_and_row()
        result = self.tree_view.model().moveRow(
            parent,
            row,
            parent,
            row - 1,
        )
        if not result:
            self.__notify_failed_to_move()

    @Slot()
    def slot_down_all(self) -> None:
        parent, row = self.__get_parent_and_row()
        count = self.tree_view.model().rowCount(parent)
        last_row = count - 1

        result = self.tree_view.model().moveRow(
            parent,
            row,
            parent,
            last_row,
        )
        if not result:
            self.__notify_failed_to_move()

    @Slot()
    def slot_up_all(self) -> None:
        parent, row = self.__get_parent_and_row()
        result = self.tree_view.model().moveRow(
            parent,
            row,
            parent,
            0,
        )
        if not result:
            self.__notify_failed_to_move()

    def __show_g(self, g: axipy.Geometry) -> None:
        map_view = CollectionOrderEditorObserver.active_map_view()

        bounds: axipy.Rect = g.bounds
        center: axipy.Pnt = bounds.center

        bounds = _fix_cs_from_selection_table_and_map_view(bounds)
        center = _fix_cs_from_selection_table_and_map_view(center)

        width: float = bounds.width
        if isinstance(g, axipy.Point):
            map_view.center = center
            return None

        cs = g.coordsystem
        unit = cs.unit if cs else None

        if cs != map_view.coordsystem:
            unit = map_view.coordsystem.unit

        map_view.set_zoom_and_center(width * 2.0, center, unit)

    @Slot()
    def slot_show_all(self) -> None:
        tree_item = self.selection_model.currentIndex().internalPointer()
        if isinstance(tree_item, TreeItem):
            g = tree_item.geometry()
            if g:
                self.__show_g(g)


class CollectionOrderEditorMapTool(axipy.MapTool):
    POLYGON_NAME = "Полигон"
    HOLE_NAME = "Внутренний контур"
    LINE_STRING_NAME = "Полилиния"
    POINT_NAME = "Точка"
    GEOMETRY_COLLECTION_POLYGONS_NAME = "Полигоны"
    GEOMETRY_COLLECTION_LINE_STRINGS_NAME = "Полилинии"
    GEOMETRY_COLLECTION_POINTS_NAME = "Точки"

    def __init__(self, plugin: 'CollectionOrderEditor') -> None:
        super().__init__()
        self._plugin: 'CollectionOrderEditor' = plugin

        self.__current_geom_to_draw: axipy.Geometry | None = None

        self.__already_asked_unsaved_changes: bool = False

        self.big_white_pen: QPen = QPen(Qt.white, BIG_CROSS_SIZE)
        self.small_blue_pen: QPen = QPen(Qt.blue, SMALL_CROSS_SIZE)

        # init ui
        self.__widget: CollectionOrderEditorWidget = CollectionOrderEditorWidget(self._plugin)
        view = self.__widget.tree_view

        self._root_item = TreeItem("root")
        self._tree_model = TreeModel(self._root_item)
        self._tree_model.rowsMoved.connect(self.slot_on_model_rows_moved)

        view.setModel(self._tree_model)

        selection_model = view.selectionModel()
        self.selection_model = selection_model
        selection_model.currentChanged.connect(self.slot_tree_view_selection_current_changed)

        self._tool_panel = axipy.ActiveToolPanel().make_acceptable(
            title=self._plugin.title,
            observer_id=self._plugin._observer,
            widget=self.__widget,
        )
        self._tree_model.rowsMoved.connect(self.slot_check_panel_accepted_enabled)
        self._tree_model.dataChanged.connect(self.slot_check_panel_accepted_enabled)
        self.slot_check_panel_accepted_enabled()

    @Slot()
    def slot_check_panel_accepted_enabled(self) -> None:
        enable: bool = self._tree_model.check_has_unsaved_changes()
        if enable:
            self._tool_panel.try_enable()
        else:
            self._tool_panel.disable()

    @Slot()
    def slot_on_model_rows_moved(self) -> None:
        self.__widget.check_buttons_enabled()

    @Slot(QModelIndex, QModelIndex)
    def slot_tree_view_selection_current_changed(self, _old: QModelIndex, _new: QModelIndex) -> None:
        tree_item = self.selection_model.currentIndex().internalPointer()
        if tree_item:
            self.__widget.check_buttons_enabled()

            self.__current_geom_to_draw = tree_item.geometry()
            self.redraw()

    def _enumerate_items(
            self,
            items: Iterable[axipy.Geometry],
            child_item_name: str,
            parent_item: TreeItem,
    ) -> Generator[TreeItem]:
        for i, item in enumerate(items, 1):
            yield TreeItem(f"{child_item_name} {i}", parentItem=parent_item, geometry=item)

    def _consume_iter(self, iterator: Iterator) -> None:
        collections.deque(iterator, maxlen=0)

    def __holes_to_polygon_gen(self, holes, cs: axipy.CoordSystem) -> Generator[axipy.Polygon]:
        for hole in holes:
            yield axipy.Polygon(*hole, cs=cs)

    def __consume_polygons(self, polygons: Iterable[axipy.Polygon], root_item: TreeItem) -> None:
        for poly_item in self._enumerate_items(polygons, self.POLYGON_NAME, root_item):
            polygon = cast(axipy.Polygon, poly_item.geometry())
            holes = polygon.holes
            cs = polygon.coordsystem
            self._consume_iter(self._enumerate_items(self.__holes_to_polygon_gen(holes, cs), self.HOLE_NAME, poly_item))

    def analyze_single_geometry_collection(self, g: axipy.Geometry | None) -> None:
        self._root_item.clearChildren()

        if isinstance(g, axipy.MultiPolygon):
            self.__consume_polygons(g, self._root_item)
        elif isinstance(g, axipy.MultiLineString):
            self._consume_iter(self._enumerate_items(g, self.LINE_STRING_NAME, self._root_item))
        elif isinstance(g, axipy.MultiPoint):
            self._consume_iter(self._enumerate_items(g, self.POINT_NAME, self._root_item))
        elif isinstance(g, axipy.GeometryCollection):
            # axipy.Polygon
            polygons = tuple(elem for elem in g if isinstance(elem, axipy.Polygon))
            if polygons:
                polygons_root = TreeItem(self.GEOMETRY_COLLECTION_POLYGONS_NAME, parentItem=self._root_item)
                self.__consume_polygons(polygons, polygons_root)
            # axipy.LineString
            line_strings = tuple(elem for elem in g if isinstance(elem, axipy.LineString))
            if line_strings:
                line_strings_root = TreeItem(self.GEOMETRY_COLLECTION_LINE_STRINGS_NAME, parentItem=self._root_item)
                self._consume_iter(self._enumerate_items(line_strings, self.LINE_STRING_NAME, line_strings_root))
            # axipy.Point
            points = tuple(elem for elem in g if isinstance(elem, axipy.Point))
            if points:
                points_root = TreeItem(self.GEOMETRY_COLLECTION_POINTS_NAME, parentItem=self._root_item)
                self._consume_iter(self._enumerate_items(points, self.POINT_NAME, points_root))

    @Slot()
    def slot_tool_panel_accepted(self) -> None:
        self.commit_changes()
        self.slot_check_panel_accepted_enabled()

    def load(self) -> None:
        self.__current_geom_to_draw = None
        self.__already_asked_unsaved_changes = False

        axipy.ObserverManager.SelectionEditableIsSame.changed.connect(self.__widget.check_buttons_enabled)

        axipy.selection_manager.changed.connect(self.slot_selection_manager_changed)
        axipy.view_manager.active_changed.connect(self.reset)
        self._tool_panel.accepted.connect(self.slot_tool_panel_accepted)

        self._tool_panel.activate()
        self.redraw()

    def unload(self) -> None:
        self.__ask_save_changes_on_unload()

        axipy.ObserverManager.SelectionEditableIsSame.changed.disconnect(self.__widget.check_buttons_enabled)

        axipy.selection_manager.changed.disconnect(self.slot_selection_manager_changed)
        axipy.view_manager.active_changed.disconnect(self.reset)
        self._tool_panel.accepted.disconnect(self.slot_tool_panel_accepted)

        self._tool_panel.deactivate()

        self.redraw()

    def __ask_save_changes_on_unload(self) -> None:
        if self.__already_asked_unsaved_changes:
            return None

        if not self._tree_model.check_has_unsaved_changes():
            return None

        result = axipy.show_dialog(
            title=self._plugin.title,
            text=self._plugin.tr("Работа инструмента завершена. Применить изменения?"),
            icon=axipy.DlgIcon.QUESTION,
            buttons=axipy.DlgButtons.YES_NO,
            default_button=axipy.DlgButtons.YES,
        )
        if result == axipy.DlgButtons.YES:
            self.commit_changes()

    @Slot()
    def slot_selection_manager_changed(self) -> None:
        self.__current_geom_to_draw = None
        self.redraw()

    def __draw_points(self, painter: QPainter, points: list[QPoint]) -> None:
        painter.setPen(self.big_white_pen)
        painter.drawPoints(points)
        painter.setPen(self.small_blue_pen)
        painter.drawPoints(points)

    def __draw_lines(self, painter: QPainter, line: list[QLine]) -> None:
        painter.setRenderHint(QPainter.Antialiasing, True)
        painter.setPen(self.big_white_pen)
        painter.drawLines(line)
        painter.setPen(self.small_blue_pen)
        painter.drawLines(line)

    def __draw_polygon(self, painter: QPainter, polygon: list[QPoint]) -> None:
        painter.setRenderHint(QPainter.Antialiasing, True)
        painter.setPen(self.big_white_pen)
        painter.drawPolyline(polygon)
        painter.setPen(self.small_blue_pen)
        painter.drawPolygon(polygon)

    def __draw_current_geom(self, painter: QPainter, g: axipy.Geometry) -> None:
        if isinstance(g, axipy.Point):
            p = self.to_device(axipy.Pnt(g.x, g.y))
            self.__draw_points(painter, [p])
        elif isinstance(g, axipy.LineString):
            self.__draw_line_string(painter, g)
        elif isinstance(g, axipy.Polygon):
            self.__draw_list_points_as_polygon(painter, g.points)

    def __draw_line_string(self, painter: QPainter, ls: axipy.LineString) -> None:
        points = ls.points
        self.__draw_list_points_as_lines(painter, points)

    def __convert_list_points_to_lines(self, list_points: axipy.ListPoint | list[axipy.Pnt]) -> list[QLine]:
        lines: list[QLine] = []
        for i in range(len(list_points) - 1):
            line = QLine(self.to_device(list_points[i]), self.to_device(list_points[i + 1]))
            lines.append(line)
        return lines

    def __draw_list_points_as_lines(self, painter: QPainter, list_points: axipy.ListPoint | list[axipy.Pnt]) -> None:
        lines: list[QLine] = self.__convert_list_points_to_lines(list_points)
        self.__draw_lines(painter, lines)

    def __draw_list_points_as_polygon(self, painter: QPainter, list_points: axipy.ListPoint | list[axipy.Pnt]) -> None:
        polygon: list[QPoint] = [self.to_device(pnt) for pnt in list_points]
        self.__draw_polygon(painter, polygon)

    @overload
    def to_device(self, scene: axipy.Pnt) -> QPoint:
        ...

    @overload
    def to_device(self, scene: axipy.Rect) -> QRect:
        ...

    def to_device(self, scene):
        scene = _fix_cs_from_selection_table_and_map_view(scene)
        return super().to_device(scene)

    def paintEvent(self, event: QPaintEvent, painter: QPainter) -> None:
        painter.save()

        self.__draw_current_geom(painter, self.__current_geom_to_draw)

        painter.restore()
        super().paintEvent(event, painter)

    def keyPressEvent(self, event: QKeyEvent) -> bool | None:
        if event.key() == Qt.Key_Escape:
            self.reset()
            return None
        return super().keyPressEvent(event)

    def commit_changes(self) -> None:
        if not self._tree_model.check_has_unsaved_changes():
            return None

        selection = axipy.data_manager.selection
        if selection is not None:
            f = next(selection.items(), None)
            f.geometry = self._tree_model.get_geometry_collection()
            selection.update(f)

        self.rebuild_from_selection()

    def rebuild_from_selection(self) -> None:
        self._tree_model.beginResetModel()

        g: axipy.Geometry | None = None
        selection = axipy.data_manager.selection
        if selection is not None:
            f = next(selection.items(), None)
            if f:
                g = f.geometry
        self.analyze_single_geometry_collection(g)

        self._tree_model.endResetModel()

        self.redraw()

    def mouseReleaseEvent(self, event: QMouseEvent) -> bool:
        if event.button() != Qt.LeftButton:
            return self.PassEvent

        modifiers = event.modifiers() & ~Qt.ShiftModifier
        no_modifiers_event = QMouseEvent(
            event.type(),
            event.localPos(),
            event.button(),
            event.buttons(),
            modifiers,
        )
        SelectToolHelpers.select_by_mouse(self.view, no_modifiers_event)

        self.rebuild_from_selection()

        self.redraw()
        return self.BlockEvent

    @Slot()
    def reset(self) -> None:
        super().reset()

    def canUnload(self, reason: axipy.DeactivationReason) -> bool:
        if not self._tree_model.check_has_unsaved_changes():
            return True

        if reason in (
                axipy.DeactivationReason.Unknown,
                axipy.DeactivationReason.ObjectClose,
                axipy.DeactivationReason.ActionClick,
                axipy.DeactivationReason.ActionShortcut,
                axipy.DeactivationReason.WindowClose,
                axipy.DeactivationReason.LayerClick,
        ):

            result = axipy.show_dialog(
                title=self._plugin.title,
                text=self._plugin.tr("Работа инструмента будет завершена. Применить изменения?"),
                icon=axipy.DlgIcon.QUESTION,
                buttons=axipy.DlgButtons.YES_NO_CANCEL,
                default_button=axipy.DlgButtons.YES,
            )

            if result == axipy.DlgButtons.CANCEL:
                return False
            elif result == axipy.DlgButtons.NO:
                self.__already_asked_unsaved_changes = True
            elif result == axipy.DlgButtons.YES:
                self.commit_changes()
                self.__already_asked_unsaved_changes = True

        return True
