from typing import Dict

from axipy.da import GeometryType
from PySide2.QtCore import Qt
from PySide2.QtUiTools import QUiLoader
from PySide2.QtWidgets import QCheckBox, QGridLayout, QLayout, QVBoxLayout, QWidget

from ..data_types import NamedSearchPatterns, SearchGroupPatterns, SearchSettings
from .style_widget_interface import SelectByStyleWidgetInterface


class SelectByCollectionStyleWidget(SelectByStyleWidgetInterface):

    def __init__(self, parent: QWidget, ui_file: str) -> None:
        super().__init__(parent)
        self.__ui = self.load_ui(ui_file)
        self.__patterns_comboboxes = {}  # type: Dict[str, Dict[str, QCheckBox]]
        self.__geom_comboboxes = {}  # type: Dict[GeometryType, QCheckBox]
        self.__ui.symbolGB.setVisible(False)
        self.__ui.regionGB.setVisible(False)
        self.__ui.lineGB.setVisible(False)

    def result(self) -> SearchSettings:
        return SearchSettings(self.search_patterns, {GeometryType.GeometryCollection: True})

    def set_point_patterns(self, patterns: NamedSearchPatterns):
        self.__ui.symbolGB.setVisible(True)
        self.add_search_patterns(patterns, self.__ui.symbolGB)

    def set_region_patterns(self, border_patterns: NamedSearchPatterns, brush_patterns: NamedSearchPatterns):
        self.__ui.regionGB.setVisible(True)
        self.fill_layout(border_patterns, self.__ui.borderVerticalLayout)
        self.fill_layout(brush_patterns, self.__ui.brushVerticalLayout)

    def set_line_patterns(self, patterns: NamedSearchPatterns):
        self.__ui.lineGB.setVisible(True)
        self.add_search_patterns(patterns, self.__ui.lineGB)

    def add_search_patterns(self, patterns: NamedSearchPatterns, widget: QWidget):
        vlayout = QVBoxLayout(widget)
        self.fill_layout(patterns, vlayout)
        widget.setLayout(vlayout)

    def fill_layout(self, patterns: NamedSearchPatterns, layout: QLayout):
        if patterns.name not in self.__patterns_comboboxes:
            self.__patterns_comboboxes[patterns.name] = {}
        grouped_comboboxes = self.__patterns_comboboxes[patterns.name]
        for pattern_name, is_checked in patterns.patterns.items():
            cb = QCheckBox(pattern_name)
            cb.setCheckState(Qt.Checked if is_checked else Qt.Unchecked)
            layout.addWidget(cb)
            grouped_comboboxes[pattern_name] = cb

    def load_ui(self, ui_file: str) -> QWidget:
        ui = QUiLoader().load(ui_file, self)
        grid_layout = QGridLayout()
        grid_layout.setHorizontalSpacing(0)
        grid_layout.setVerticalSpacing(0)
        grid_layout.addWidget(ui)
        self.setLayout(grid_layout)
        return ui

    @property
    def search_patterns(self) -> SearchGroupPatterns:
        result = {}  # type: SearchGroupPatterns
        for group_name, cb_search_patterns in self.__patterns_comboboxes.items():
            patterns = {}
            for pattern_key, cb in cb_search_patterns.items():
                patterns[pattern_key] = cb.isChecked()
            result[group_name] = patterns
        return result
