from typing import Iterable, Union

from axipy import (
    ActiveToolPanel,
    Arc,
    CoordSystem,
    CoordTransformer,
    CosmeticLayer,
    CurrentSettings,
    Ellipse,
    Feature,
    Geometry,
    GeometryCollection,
    Line,
    LineString,
    ListLayers,
    MapTool,
    MultiLineString,
    MultiPolygon,
    ObserverManager,
    Polygon,
    Rect,
    Rectangle,
    RoundRectangle,
    Table,
    VectorLayer,
    view_manager,
)
from PySide2.QtCore import QLineF, QPoint, Qt, QTimer
from PySide2.QtGui import QBrush, QMouseEvent, QPainter, QPaintEvent, QPen

from .AngleBetween import AngleBetweenWidget
from .Arrow import ArrowLine


class AngleBetweenTool(MapTool):
    first_activation = True
    panel_closed = False
    first_switch = True
    timer_ended = False

    def __init__(self) -> None:
        super().__init__()
        self.az = 0
        self._lines = []
        self._lines_draw = []
        self.draw_layers = []
        self.widget = AngleBetweenWidget()
        self.widget.pushButton.clicked.connect(self.clear_data)
        self.widget.lineEdit.setReadOnly(True)
        self.widget.lineEdit.setAlignment(Qt.AlignRight)
        self.widget.lineEdit_2.setReadOnly(True)
        self.widget.lineEdit_2.setAlignment(Qt.AlignRight)
        self.widget.lineEdit_3.setReadOnly(True)
        self.widget.lineEdit_3.setAlignment(Qt.AlignRight)
        stylesheet = 'QLineEdit[readOnly="true"] {' "color: black;" "background-color: white;}"
        self.widget.lineEdit.setStyleSheet(stylesheet)
        self.widget.lineEdit_2.setStyleSheet(stylesheet)
        self.widget.lineEdit_3.setStyleSheet(stylesheet)
        self.widget.radioButton_plain.toggled.connect(self.radiobutton_plain_switched_action)
        self.painter = QPainter()
        self.timer = QTimer(interval=200, timeout=self.timer_end)
        self.timer.start()
        service = ActiveToolPanel()
        self.tool_panel = service.make_custom("Рассчет угла", observer_id=ObserverManager.Editable, widget=self.widget)
        self.tool_panel.panel_was_closed.connect(self.panel_closed_action)

    def load(self):
        view_manager.active_changed.connect(self.reset)
        self.redraw()

    def unload(self):
        view_manager.active_changed.disconnect(self.reset)
        if not self.first_activation:
            self.first_activation = True
        self.redraw()

    def panel_closed_action(self):
        self.panel_closed = True

    def radiobutton_plain_switched_action(self):
        if len(self._lines_draw) > 0:
            azs = []
            if len(self._lines) == 1 and self.first_switch and len(self._lines_draw) > 1:
                self._lines_draw[0], self._lines_draw[1] = self._lines_draw[1], self._lines_draw[0]
                self.first_switch = False
            for line in self._lines_draw:
                azs.append(self.count_az(line))
            if len(self._lines_draw) == 1:
                self.widget.lineEdit.setText(str(azs[0]))
            elif len(self._lines_draw) == 2:
                self.widget.lineEdit.setText(str(azs[0]))
                self.widget.lineEdit_2.setText(str(azs[1]))
                self.widget.lineEdit_3.setText(str(self.count_angle_az(azs[0], azs[1])))

    def mouseReleaseEvent(self, event: QMouseEvent):
        if event.button() == Qt.LeftButton:
            if self.first_activation:
                self.tool_panel.activate()
                self.first_activation = False
            if self.panel_closed:
                self.tool_panel.activate()
                self.panel_closed = False
            if len(self._lines) <= 2:
                if self.is_snapped():
                    device_point = self.snap_device()
                else:
                    device_point = event.pos()
                if len(self._lines) == 2:
                    self.first_switch = False
                if not self.first_switch and not len(self._lines) == 2:
                    self.first_switch = True
                mv = self.view
                if mv.coordsystem.lat_lon:
                    self.widget.radioButton_ellipsoid.setChecked(True)
                    self.widget.radioButton_plain.setDisabled(True)
                elif mv.coordsystem.non_earth:
                    self.widget.radioButton_plain.setChecked(True)
                    self.widget.radioButton_ellipsoid.setDisabled(True)
                else:
                    self.widget.radioButton_plain.setEnabled(True)
                    self.widget.radioButton_ellipsoid.setEnabled(True)
                layers = mv.map.layers
                self.layer_action(layers, device_point)
            else:
                return self.BlockEvent

    def layer_action(self, layers: ListLayers, device_point: QPoint):
        cos_layer = self.view.map.cosmetic
        if cos_layer.selectable and cos_layer.visible:
            do = cos_layer.data_object
            bbox = self.get_select_rect(device_point, CurrentSettings.DistancePrecision)
            coord_transformer = CoordTransformer(self.view.coordsystem, cos_layer.coordsystem)
            bbox = coord_transformer.transform(bbox)
            if isinstance(do, Table):
                features = do.items(bbox=bbox)
                if self.features_action(cos_layer, features, bbox, do.coordsystem):
                    pass
                else:
                    for layer in layers:
                        if isinstance(layer, VectorLayer) and layer.selectable and layer.visible:
                            do = layer.data_object
                            bbox = self.get_select_rect(device_point, CurrentSettings.DistancePrecision)
                            if isinstance(do, Table):
                                if not self.view.coordsystem == do.coordsystem:
                                    coord_transformer = CoordTransformer(self.view.coordsystem, do.coordsystem)
                                    bbox = coord_transformer.transform(bbox)
                                features = do.itemsInObject(Rectangle(bbox, cs=do.coordsystem))
                                if self.features_action(layer, features, bbox, do.coordsystem):
                                    break
                        elif isinstance(layer, ListLayers):
                            self.layer_action(layer, device_point)
        self.redraw()

    def features_action(self, layer: VectorLayer, features: Iterable[Feature], bbox: Rect, cs: CoordSystem) -> bool:
        for f in features:
            geom = f.geometry.reproject(cs)
            if self.geometry_check(layer, geom, bbox):
                return True
        return False

    def geometry_check(self, layer: VectorLayer, geom: Geometry, bbox: Rect) -> bool:
        if not isinstance(geom, (Arc, RoundRectangle, Ellipse)):
            if isinstance(geom, (Line, LineString)):
                self.draw_layers.append(layer)
                if len(self.draw_layers) > 2:
                    self.draw_layers.pop(0)
                self.geometry_action(geom, bbox)
                return True
            elif isinstance(geom, (Rectangle, Polygon)):
                geom = geom.to_linestring()
                if geom.intersects(Rectangle(bbox, cs=geom.coordsystem)):
                    self.draw_layers.append(layer)
                    if len(self.draw_layers) > 2:
                        self.draw_layers.pop(0)
                    self.geometry_action(geom, bbox)
                    return True
            elif isinstance(geom, (MultiPolygon, MultiLineString, GeometryCollection)):
                for g in geom:
                    if not g.coordsystem == geom.coordsystem:
                        g = g.reproject(geom.coordsystem)
                    if g.intersects(Rectangle(bbox, cs=geom.coordsystem)):
                        return self.geometry_check(layer, g, bbox)
        return False

    def linestring_to_segments(self, ls: LineString):
        pts = ls.points
        cs = ls.coordsystem
        segments = []
        if len(pts) >= 2:
            for i in range(len(pts) - 1):
                segments.append(Line(pts[i], pts[i + 1], cs=cs))
            return segments

    def geometry_action(self, geom: Geometry, bbox: Rect):
        if isinstance(geom, Line):
            self.line_action(geom)
        elif isinstance(geom, LineString):
            self.linestring_action(geom, bbox)
        elif isinstance(geom, MultiLineString):
            self.multilinestring_action(geom, bbox)
        elif isinstance(geom, Polygon):
            self.polygon_action(geom, bbox)
        elif isinstance(geom, MultiPolygon):
            self.multipolygon_action(geom, bbox)
        elif isinstance(geom, Rectangle):
            self.polygon_action(geom, bbox)
        elif isinstance(geom, GeometryCollection):
            self.geometrycollection_action(geom, bbox)

    def line_action(self, geom: Line):
        if len(self._lines) == 2:
            self._lines.clear()
        self._lines.append(geom)
        if len(self._lines) == 1:
            if len(self._lines_draw) == 2:
                self.az = self.count_az(self._lines[0])
                az2 = self.count_az(self._lines_draw[1])
                self.widget.lineEdit.setText(str(az2))
                self.widget.lineEdit_2.setText(str(self.az))
                self.widget.lineEdit_3.setText(str(self.count_angle_az(self.az, az2)))
            else:
                self.az = self.count_az(self._lines[0])
                self.widget.lineEdit.setText(str(self.az))
            for line in self._lines:
                self._lines_draw.append(line)
            while len(self._lines_draw) > 2:
                self._lines_draw.pop(0)
            return self.PassEvent
        elif len(self._lines) == 2:
            az2 = self.count_az(self._lines[1])
            self.widget.lineEdit.setText(str(self.az))
            self.widget.lineEdit_2.setText(str(az2))
            self.widget.lineEdit_3.setText(str(self.count_angle_az(self.az, az2)))
            for line in self._lines:
                self._lines_draw.append(line)
            while len(self._lines_draw) > 2:
                self._lines_draw.pop(0)
            return self.BlockEvent

    def linestring_action(self, geom: LineString, bbox: Rect):
        segments = self.linestring_to_segments(geom)
        for segment in segments:
            if segment.intersects(Rectangle(bbox, cs=segment.coordsystem)):
                self.geometry_action(segment, bbox)
                break

    def multilinestring_action(self, geom: MultiLineString, bbox: Rect):
        for ls in geom:
            if not ls.coordsystem == geom.coordsystem:
                ls = ls.reproject(geom.coordsystem)
            if ls.intersects(Rectangle(bbox, cs=ls.coordsystem)):
                self.geometry_action(ls, bbox)
                break

    def polygon_action(self, geom: Union[Polygon, Rectangle], bbox: Rect):
        ls = geom.to_linestring()
        if ls.intersects(Rectangle(bbox, cs=geom.coordsystem)):
            self.geometry_action(ls, bbox)

    def multipolygon_action(self, geom: MultiPolygon, bbox: Rect):
        for poly in geom:
            if not poly.coordsystem == geom.coordsystem:
                poly = poly.reproject(geom.coordsystem)
            if poly.intersects(Rectangle(bbox, cs=poly.coordsystem)):
                self.geometry_action(poly, bbox)
                break

    def geometrycollection_action(self, geom_coll: GeometryCollection, bbox: Rect):
        for geom in geom_coll:
            if not geom.coordsystem == geom_coll.coordsystem:
                geom = geom.reproject(geom_coll.coordsystem)
            if geom.intersects(Rectangle(bbox, cs=geom.coordsystem)):
                self.geometry_action(geom, bbox)
                break

    def count_angle_az(self, az1: float, az2: float) -> float:
        angle = az1 - az2
        if angle < 0:
            angle = -angle
        if angle > 180:
            angle = 360 - angle
        return angle

    def clear_data(self):
        self.widget.lineEdit.clear()
        self.widget.lineEdit_2.clear()
        self.widget.lineEdit_3.clear()
        self._lines.clear()
        self._lines_draw.clear()
        self.draw_layers.clear()
        self.redraw()
        if not (self.widget.radioButton_plain.isEnabled() or self.widget.radioButton_ellipsoid.isEnabled()):
            if self.view.coordsystem.non_earth:
                self.widget.radioButton_plain.setEnabled(True)
            elif self.view.coordsystem.lat_lon:
                self.widget.radioButton_ellipsoid.setEnabled(True)
            else:
                self.widget.radioButton_plain.setEnabled(True)
                self.widget.radioButton_ellipsoid.setEnabled(True)

    def count_az(self, line: Line):
        if self.widget.radioButton_plain.isChecked():
            return Geometry.distance_by_points(line.begin, line.end, cs=None)[1]
        # elif self.widget.radioButton_sphere.isChecked():
        #     # return Geometry.distance_by_points(line.begin, line.end, cs= CoordSystem.from_prj("1, 999, 12, 0, 0, 0"))[1]
        #     return self.count_az_sphere(line.geometry)[1]
        elif self.widget.radioButton_ellipsoid.isChecked():
            map_cs = self.view.coordsystem
            line = line.reproject(map_cs)
            return Geometry.distance_by_points(line.begin, line.end, cs=map_cs)[1]

    def paintEvent(self, event: QPaintEvent, painter: QPainter):
        painter.save()
        if self.timer_ended:
            self.timer.stop()
            self.timer = QTimer(interval=200, timeout=self.timer_end)
            self.timer.start()
            self.draw_line(painter)
        painter.restore()
        super().paintEvent(event, painter)

    def timer_end(self):
        self.timer_ended = True

    def draw_line(self, painter: QPainter):
        old_pen = QPen(painter.pen())
        old_brush = QBrush(painter.brush())
        lines = []
        if len(self.draw_layers) > 0:
            for i in range(len(self._lines_draw)):
                if self.draw_layers[i].coordsystem == self._lines_draw[i].coordsystem:
                    if isinstance(self.draw_layers[i], CosmeticLayer):
                        coord_transformer = CoordTransformer(self.draw_layers[i].coordsystem, self.view.coordsystem)
                        lines.append(
                            QLineF(
                                self.to_device(coord_transformer.transform(self._lines_draw[i].begin)),
                                self.to_device(coord_transformer.transform(self._lines_draw[i].end)),
                            )
                        )
                    elif not (self.view.coordsystem == self._lines_draw[i].coordsystem):
                        coord_transformer = CoordTransformer(self.draw_layers[i].coordsystem, self.view.coordsystem)
                        lines.append(
                            QLineF(
                                self.to_device(coord_transformer.transform(self._lines_draw[i].begin)),
                                self.to_device(coord_transformer.transform(self._lines_draw[i].end)),
                            )
                        )
                    else:
                        lines.append(
                            QLineF(self.to_device(self._lines_draw[i].begin), self.to_device(self._lines_draw[i].end))
                        )
                else:
                    coord_transformer = CoordTransformer(
                        self.draw_layers[i].coordsystem, self._lines_draw[i].coordsystem
                    )
                    lines.append(
                        QLineF(
                            self.to_device(coord_transformer.transform(self._lines_draw[i].begin)),
                            self.to_device(coord_transformer.transform(self._lines_draw[i].end)),
                        )
                    )
            for line in lines:
                arrow = ArrowLine(line)
                arrow.set_color(Qt.red)
                arrow.drawArrow(painter)
        painter.setPen(old_pen)
        painter.setBrush(old_brush)
