from enum import Enum, auto
from functools import cached_property
from typing import (
    Any,
    List,
    Literal,
    Optional,
    Union,
    cast,
    overload,
)

import axipy
from axipy._internal._decorator import _experimental
from axipy.cpp_gui import (
    InteractionType,
    ShadowMapTool,
    ShadowToolFactory,
)
from axipy.utl import Pnt, Rect
from PySide2.QtCore import (
    QEvent,
    QPoint,
    QPointF,
    QRect,
    QRectF,
    Slot,
)
from PySide2.QtGui import (
    QCursor,
    QKeyEvent,
    QMouseEvent,
    QPainter,
    QPaintEvent,
    QWheelEvent,
)

from .map_tool import DeactivationReason
from .view import MapView, ReportView, View

__all__: List[str] = [
    "ViewTool",
]


class _ShadowMapToolImpl(ShadowMapTool):

    def __init__(self, view_tool_instance: "ViewTool") -> None:
        super().__init__()
        self.__inner_inst = view_tool_instance
        self.ACCEPT = self.__inst.ACCEPT
        self.IGNORE = self.__inst.IGNORE

    @property
    def __inst(self) -> "ViewTool":
        return self.__inner_inst

    def handleEvent(self, event: QEvent) -> bool:
        result = self.__inst.handleEvent(event)
        if result is self.ACCEPT:
            return self.ACCEPT
        return super().handleEvent(event)

    def paintEvent(self, event: QPaintEvent, painter: QPainter) -> None:
        self.__inst.paintEvent(event, painter)

    def mousePressEvent(self, event: QMouseEvent) -> bool:
        result = self.__inst.mousePressEvent(event)
        if result is self.ACCEPT:
            return self.ACCEPT
        return super().mousePressEvent(event)

    def mouseReleaseEvent(self, event: QMouseEvent) -> bool:
        result = self.__inst.mouseReleaseEvent(event)
        if result is self.ACCEPT:
            return self.ACCEPT
        return super().mouseReleaseEvent(event)

    def mouseDoubleClickEvent(self, event: QMouseEvent) -> bool:
        result = self.__inst.mouseDoubleClickEvent(event)
        if result is self.ACCEPT:
            return self.ACCEPT
        return super().mouseDoubleClickEvent(event)

    def mouseMoveEvent(self, event: QMouseEvent) -> bool:
        result = self.__inst.mouseMoveEvent(event)
        if result is self.ACCEPT:
            return self.ACCEPT
        return super().mouseMoveEvent(event)

    def keyPressEvent(self, event: QKeyEvent) -> bool:
        result = self.__inst.keyPressEvent(event)
        if result is self.ACCEPT:
            return self.ACCEPT
        return super().keyPressEvent(event)

    def keyReleaseEvent(self, event: QKeyEvent) -> bool:
        result = self.__inst.keyReleaseEvent(event)
        if result is self.ACCEPT:
            return self.ACCEPT
        return super().keyReleaseEvent(event)

    def wheelEvent(self, event: QWheelEvent) -> bool:
        result = self.__inst.wheelEvent(event)
        if result is self.ACCEPT:
            return self.ACCEPT
        return super().wheelEvent(event)

    def activate(self) -> None:
        super().activate()
        self.__inst.load()
        axipy.view_manager._active_id_changed.connect(self.__slot_on_active_changed)

    def deactivate(self) -> None:
        axipy.view_manager._active_id_changed.disconnect(self.__slot_on_active_changed)
        self.__inst.unload()
        super().deactivate()

    def __check_view(self, view_id: int) -> bool:
        return self.shadow_view().id() == view_id

    @Slot(int)
    def __slot_on_active_changed(self, active_id: int) -> None:
        if self.__check_view(active_id) or self.__check_view(axipy.view_manager._active_view_id_prev):
            self.__inst.on_active_changed()

    def canDeactivate(self, reason: InteractionType) -> bool:
        deactivation_reason = axipy.DeactivationReason(reason)

        if isinstance(self.__inst, axipy.MapTool):
            result = self.__inst.canUnload(deactivation_reason)
        else:
            result = self.__inst.can_unload(deactivation_reason)

        if result is None:
            return True
        return result


# noinspection PyUnusedLocal
class ViewTool:
    """
    Инструмент работы с окном карты или отчета.
    """

    class ViewType(Enum):
        """Тип окна просмотра"""

        MAP_VIEW = auto()
        """Окно карты"""
        REPORT_VIEW = auto()
        """Окно отчета"""

    class __DescTrue:

        def __get__(self, *args: Any, **kwargs: Any) -> Literal[True]:
            return True

    class __DescFalse:

        def __get__(self, *args: Any, **kwargs: Any) -> Literal[False]:
            return False

    ACCEPT: Literal[True] = cast(Literal[True], __DescTrue())
    """
    При возвращении этого атрибута в обработчике событий, событие будет принято обработчиком и не пойдет дальше.
    """

    IGNORE: Literal[False] = cast(Literal[False], __DescFalse())
    """
    При возвращении этого атрибута в обработчике событий, событие будет проигнорировано обработчиком и пойдет дальше.
    """

    @cached_property
    def __shadow(self) -> _ShadowMapToolImpl:
        return _ShadowMapToolImpl(self)

    def load(self) -> None:
        """
        Переопределите, чтобы задать действия при включении инструмента.

        See also:
            :meth:`unload`.
        """
        pass

    def unload(self) -> None:
        """
        Переопределите, чтобы задать действия при выключении инструмента.

        See also:
            :meth:`load`.
        """
        pass

    def can_unload(self, reason: "DeactivationReason") -> bool:
        """
        Переопределите, чтобы обрабатывать причину выключения инструмента.

        Args:
            reason: причина выключения.

        Return:
            False, чтобы прервать выключение, иначе True.
        """
        return True

    @staticmethod
    def reset() -> None:
        """
        Переключает текущий инструмент на инструмент по умолчанию.
        Инструментом по умолчанию является Выбор.
        В результате будет вызван :meth:`unload` текущего инструмента.
        Делает то же самое что :meth:`axipy.ActionManager.activate_selection_tool`.
        """
        ShadowToolFactory.requestReset()

    @property
    def view(self) -> Union[MapView, ReportView]:
        """
        Возвращает окно карты или отчета, с которым работает инструмент.
        """
        return cast(Union[MapView, ReportView], View._wrap(self.__shadow.shadow_view()))

    @property
    @_experimental()
    def _is_scroll_mode(self) -> bool:
        """
        Устанавливает или возвращает режим скроллинга окна карты, если указатель мыши выходит за ее пределы.
        """
        return self.__shadow.isScrollMode()

    @_is_scroll_mode.setter
    @_experimental()
    def _is_scroll_mode(self, v: bool) -> None:
        self.__shadow.setScrollMode(v)

    @property
    def is_active(self) -> bool:
        """
        Возвращает признак активности окна, с которым работает инструмент.
        """
        return self.view.is_active

    def on_active_changed(self) -> None:
        """
        Переопределите, чтобы обрабатывать смену активности окна, с которым работает инструмент.

        See also:
            :attr:`is_active`.
        """
        pass

    @overload
    def to_scene(self, device: QPoint) -> Pnt: ...

    @overload
    def to_scene(self, device: QRect) -> Rect: ...

    def to_scene(self, device: Union[QPoint, QRect]) -> Union[Pnt, Rect]:
        """
        Переводит точки из координат окна(пикселей) в координаты на карте.

        Args:
            device: Точки в координатах окна.

        Return:
            Точки в координатах карты.
        """
        if isinstance(device, QPoint):
            q_point_f = QPointF(device.x(), device.y())
            return Pnt._point_value_from_qt(self.view.device_to_scene_transform.map(q_point_f))
        elif isinstance(device, QRect):
            q_rect_f = QRectF(device.x(), device.y(), device.width(), device.height())
            return Rect._rect_value_from_qt(self.view.device_to_scene_transform.mapRect(q_rect_f))
        raise TypeError("Unsupported type.")

    @overload
    def to_device(self, scene: Pnt) -> QPoint: ...

    @overload
    def to_device(self, scene: Rect) -> QRect: ...

    def to_device(self, scene: Union[Pnt, Rect]) -> Union[QPoint, QRect]:
        """
        Переводит точки из координат на карте в координаты окна(пиксели).

        Args:
            scene: Точки в координатах карты.

        Return:
            Точки в координатах окна.
        """
        if isinstance(scene, Pnt):
            q_point_f = self.view.scene_to_device_transform.map(Pnt._point_value_to_qt(scene))
            return q_point_f.toPoint()
        elif isinstance(scene, Rect):
            q_rect_f = self.view.scene_to_device_transform.mapRect(Rect._rect_value_to_qt(scene))
            return q_rect_f.toRect()
        raise TypeError("Unsupported type.")

    @property
    def cursor(self) -> QCursor:
        """
        Устанавливает или возвращает курсор инструмента.

        Первоначально курсор для инструмента можно установить, переопределив метод :meth:`load`::

            class MyTool(ViewTool):
                def load(self):
                    self.cursor = QCursor(Qt.SizeAllCursor)

        Если же требуется устанавливать различные типы курсора в зависимости от статуса нажатия ПКМ,
        следует переопределить методы
        :meth:`mousePressEvent` и :meth:`mouseReleaseEvent` и установить нужное значение там::

            class MyTool(ViewTool):
                def mousePressEvent(self, event) -> bool:
                    if event.button() == Qt.LeftButton:
                        self.cursor = QCursor(Qt.SizeAllCursor)
        """
        return self.view.widget.cursor()

    @cursor.setter
    def cursor(self, v: QCursor) -> None:
        self.view.widget.setCursor(v)

    def get_select_rect(self, device: QPoint, size: Optional[int] = None) -> Rect:
        """
        Возвращает прямоугольник в координатах карты для точки на экране. Удобно для
        использования при поиске объектов в месте нажатия мыши.

        Args:
            device: Точка в координатах окна.
            size: Размер квадрата в пикселях. При :obj:`None`,
                берется из настроек Аксиомы (:attr:`axipy.CurrentSettings.SensitiveMouse`)

        Return:
            Прямоугольник в координатах карты.

        Пример::

            device_point = event.pos()
            bbox = self.get_select_rect(device_point, 30)
            features = table.items(bbox=bbox)
        """
        size_int: int = size if size is not None else axipy.CurrentSettings.SensitiveMouse
        device_box = QRect(0, 0, size_int, size_int)
        device_box.moveCenter(device)
        return self.to_scene(device_box)

    def redraw(self) -> None:
        """
        Перерисовывает окно карты.

        Создает событие :class:`PySide2.QtGui.QPaintEvent` и помещает его в
        очередь обработки событий. Аналогично
        :meth:`PySide2.QtWidgets.QWidget.update`.
        """
        self.__shadow.redraw()

    def is_snapped(self) -> bool:
        """
        Проверяет, сработала ли привязка к элементам карты или отчета для текущего
        положения указателя мыши.

        See also:
            :meth:`snap`, :meth:`snap_device`.
        """
        return self.__shadow.isSnapped()

    @overload
    def snap(self) -> Optional[Pnt]: ...

    @overload
    def snap(self, default_value: None) -> Optional[Pnt]: ...

    @overload
    def snap(self, default_value: Pnt) -> Pnt: ...

    def snap(self, default_value: Optional[Pnt] = None) -> Optional[Pnt]:
        """
        Возвращает исправленные координаты, если сработала привязка к элементам в
        единицах измерения карты или отчета.

        Args:
            default_value: Значение по умолчанию.

        Возвращает значение по умолчанию, если не сработала привязка к элементам
        карты или отчета.

        Пример::

            point = self.to_scene(event.pos())
            current_point = self.snap(point)

        See also:
            :meth:`is_snapped`, :meth:`snap_device`, :meth:`to_scene`.
        """
        if not self.is_snapped():
            return default_value
        return Pnt._point_value_from_qt(self.__shadow.snapPoint())

    @overload
    def snap_device(self) -> Optional[QPoint]: ...

    @overload
    def snap_device(self, default_value: None) -> Optional[QPoint]: ...

    @overload
    def snap_device(self, default_value: QPoint) -> QPoint: ...

    def snap_device(self, default_value: Optional[QPoint] = None) -> Optional[QPoint]:
        """
        Возвращает исправленные координаты, если сработала привязка к элементам в
        единицах измерения окна карты (виджета).

        Args:
            default_value: Значение по умолчанию.

        Возвращает значение по умолчанию, если не сработала привязка к элементам
        карты или отчета.

        Пример::

            device_point = event.pos()
            current_device_point = self.snap_device(device_point)

        See also:
            :meth:`is_snapped`, :meth:`snap`.
        """
        if not self.is_snapped():
            return default_value
        return self.to_device(cast(Pnt, self.snap()))

    # noinspection PyPep8Naming
    def handleEvent(self, event: QEvent) -> bool:
        """
        Переопределите, чтобы использовать первичный обработчик всех событий инструмента.

        Если событие не блокируется этим обработчиком, то оно будет передано
        дальше в соответствующий специализированный обработчик
        :meth:`mousePressEvent`, :meth:`keyReleaseEvent` и прочие в зависимости
        от типа.

        Args:
            event: Событие.

        Return:
            :attr:`BlockEvent`, чтобы блокировать дальнейшую обработку события.
        """
        return self.IGNORE

    # noinspection PyPep8Naming
    def paintEvent(self, event: QPaintEvent, painter: QPainter) -> None:
        """
        Переопределите, чтобы обрабатывать событие отрисовки.

        Args:
            event: Событие отрисовки.
            painter: QPainter для рисования поверх виджета
        """
        pass

    # noinspection PyPep8Naming
    def mousePressEvent(self, event: QMouseEvent) -> bool:
        """
        Переопределите, чтобы обрабатывать событие нажатия клавиши мыши.

        Args:
            event: Событие нажатия клавиши мыши.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.IGNORE

    # noinspection PyPep8Naming
    def mouseReleaseEvent(self, event: QMouseEvent) -> bool:
        """
        Переопределите, чтобы обрабатывать событие отпускания клавиши мыши.

        Args:
            event: Событие отпускания клавиши мыши.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.IGNORE

    # noinspection PyPep8Naming
    def mouseDoubleClickEvent(self, event: QMouseEvent) -> bool:
        """
        Переопределите, чтобы обрабатывать событие двойного клика мыши.

        Args:
            event: Событие двойного клика мыши.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.IGNORE

    # noinspection PyPep8Naming
    def mouseMoveEvent(self, event: QMouseEvent) -> bool:
        """
        Переопределите, чтобы обрабатывать событие перемещения мыши.

        Args:
            event: Событие перемещения мыши.

        Return:
            :attr:`PassEvent` или :data:`None`, чтобы пропустить событие дальше по цепочке
            обработчиков. :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.IGNORE

    # noinspection PyPep8Naming
    def keyPressEvent(self, event: QKeyEvent) -> bool:
        """
        Переопределите, чтобы обрабатывать событие нажатия клавиши клавиатуры.

        Args:
            event: Событие нажатия клавиши клавиатуры.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.IGNORE

    # noinspection PyPep8Naming
    def keyReleaseEvent(self, event: QKeyEvent) -> bool:
        """
        Переопределите, чтобы обрабатывать событие отпускания клавиши клавиатуры.

        Args:
            event: Событие отпускания клавиши клавиатуры.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.IGNORE

    # noinspection PyPep8Naming
    def wheelEvent(self, event: QWheelEvent) -> bool:
        """
        Переопределите, чтобы обрабатывать событие колеса мыши.

        Args:
            event: Событие колеса мыши.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.IGNORE
