import builtins
import itertools
from typing import (
    TYPE_CHECKING,
    Dict,
    Final,
    List,
    Optional,
    Tuple,
    Union,
    cast,
)

from axipy.cs import CoordSystem, CoordTransformer
from axipy.utl import Rect

__all__: List[str] = [
    "Attribute",
    "Schema",
]


# noinspection PyPep8Naming
class Attribute:
    """
    Атрибут схемы таблицы.

    Используется для создания и инспектирования атрибутов и схем :class:`axipy.Schema`.
    Для создания атрибутов используйте функции :meth:`string`, :meth:`decimal` и другие.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_schema.py
        :caption: Пример создания
        :pyobject: example_attribute_create
        :lines: 2-
        :dedent: 4
    """

    _TYPES: Final[Tuple[str, ...]] = (
        "bool",
        "date",
        "time",
        "datetime",
        "string",
        "double",
        "int",
        "decimal",
        "int64",
        "int16",
    )

    DEFAULT_STRING_LENGTH: Final[int] = 80
    """Длина строки по умолчанию."""
    DEFAULT_DECIMAL_LENGTH: Final[int] = 15
    """Длина поля типа `decimal` по умолчанию."""
    DEFAULT_DECIMAL_PRECISION: Final[int] = 5
    """Точность поля типа `decimal` по умолчанию."""

    def __init__(self, name: str, typedef: str) -> None:
        """
        Конструктор класса.

        Args:
            name: Имя атрибута.
            typedef: Описание типа в формате ``<тип>[:длина][.точность]``.
        """
        self._name: str = name
        self._typedef: str = typedef
        self._alias: Optional[str] = None
        self._unique: bool = False
        self._readOnly: bool = False
        self._comments: Optional[str] = None
        self._index = None

    def __repr__(self) -> str:
        return f"axipy.Attribute({self.name!r}, {self.typedef!r})"

    @staticmethod
    def _create(a: dict) -> "Attribute":
        if not ("name" in a and "type" in a):
            raise TypeError(f"Argument {a} is not valid.")

        t = a["type"]
        n = a["name"]
        length = a["length"] if "length" in a else Attribute.DEFAULT_STRING_LENGTH
        attr = None
        if t == "string":
            attr = Attribute.string(n, length)
        elif t == "int":
            attr = Attribute.integer(n)
        elif t == "int64":
            attr = Attribute.large(n)
        elif t == "int16":
            attr = Attribute.short(n)
        elif t == "double":
            attr = Attribute.double(n)
        elif t == "bool":
            attr = Attribute.bool(n)
        elif t == "date":
            attr = Attribute.date(n)
        elif t == "time":
            attr = Attribute.time(n)
        elif t == "datetime":
            attr = Attribute.datetime(n)
        elif t == "decimal":
            p = a["precision"] if "precision" in a else Attribute.DEFAULT_DECIMAL_PRECISION
            attr = Attribute.decimal(n, length, p)

        if not attr:
            raise TypeError(f"Unsupported type {t}.")

        alias = a["alias"] if "alias" in a else None
        if alias:
            attr.alias = alias
        readOnly = a["readOnly"] if "readOnly" in a else None
        if readOnly:
            attr.readOnly = readOnly
        unique = a["unique"] if "unique" in a else None
        if unique:
            attr.unique = unique
        attr._index = a["index"] if "index" in a else None
        return attr

    @property
    def name(self) -> str:
        """Возвращает имя атрибута."""
        return self._name

    @property
    def typedef(self) -> str:
        """
        Возвращает описание типа.

        Строка вида ``<тип>[:длина][.точность]``.
        """
        return self._typedef

    @property
    def length(self) -> int:
        """Возвращает длину атрибута."""
        return self._type_length(self._typedef)

    @property
    def precision(self) -> int:
        """Возвращает точность."""
        return self._type_precision(self._typedef)

    @property
    def readOnly(self) -> bool:
        """Устанавливает или возвращает признак "Поле только для чтения"."""
        return self._readOnly

    @readOnly.setter
    def readOnly(self, v: bool) -> None:
        self._readOnly = v

    @property
    def unique(self) -> bool:
        """Устанавливает или возвращает признак "Поле является уникальным"."""
        return self._unique

    @unique.setter
    def unique(self, v: bool) -> None:
        self._unique = v

    @property
    def alias(self) -> Optional[str]:
        """Устанавливает или возвращает псевдоним."""
        return self._alias

    @alias.setter
    def alias(self, v: str) -> None:
        self._alias = v

    @property
    def comments(self) -> Optional[str]:
        """Устанавливает или возвращает дополнительную текстовую информацию к полю."""
        return self._comments

    @comments.setter
    def comments(self, v: str) -> None:
        self._comments = v

    @property
    def type_string(self) -> str:
        """Возвращает тип в виде строки без длины и точности."""
        _TYPES_sort = sorted(self._TYPES, key=len, reverse=True)
        return next(filter(lambda t: self._typedef.startswith(t), _TYPES_sort))

    @staticmethod
    def string(name: str, length: int = DEFAULT_STRING_LENGTH) -> "Attribute":
        """
        Создает атрибут строкового типа.

        Args:
            name: Имя атрибута.
            length: Длина атрибута.
        """
        return Attribute(name, f"string:{length}")

    @staticmethod
    def decimal(
        name: str,
        length: int = DEFAULT_DECIMAL_LENGTH,
        precision: int = DEFAULT_DECIMAL_PRECISION,
    ) -> "Attribute":
        """
        Создает атрибут десятичного типа.

        Args:
            name: Имя атрибута.
            length: Длина атрибута. Количество символов, включая запятую.
            precision: Число знаков после запятой.
        """
        return Attribute(name, f"decimal:{length}.{precision}")

    @staticmethod
    def integer(name: str) -> "Attribute":
        """
        Создает атрибут целого типа.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, "int")

    @staticmethod
    def large(name: str) -> "Attribute":
        """
        Создает атрибут целого 64-битного типа.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, "int64")

    @staticmethod
    def short(name: str) -> "Attribute":
        """
        Создает атрибут целого 16-битного типа.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, "int16")

    @staticmethod
    def float(name: str) -> "Attribute":
        """
        Создает атрибут вещественного типа.

        То же, что и :meth:`double`

        Args:
            name: Имя атрибута.
        """
        return Attribute.double(name)

    @staticmethod
    def double(name: str) -> "Attribute":
        """
        Создает атрибут вещественного типа.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, "double")

    @staticmethod
    def bool(name: str) -> "Attribute":
        """
        Создает атрибут логического типа.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, "bool")

    @staticmethod
    def date(name: str) -> "Attribute":
        """
        Создает атрибут типа дата.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, "date")

    @staticmethod
    def time(name: str) -> "Attribute":
        """
        Создает атрибут типа время.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, "time")

    @staticmethod
    def datetime(name: str) -> "Attribute":
        """
        Создает атрибут типа дата и время.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, "datetime")

    def __eq__(self, other: object) -> builtins.bool:
        if isinstance(other, str):
            return self._name == other
        elif isinstance(other, Attribute):
            return self._name == other._name and self._typedef == other._typedef
        return NotImplemented

    @staticmethod
    def _type_length(type_definition: str) -> int:
        pos = type_definition.rfind(":")
        if pos < 0:
            return 0
        start = pos + 1
        end = type_definition.rfind(".")
        end = len(type_definition) if end < 0 else end
        length = type_definition[start:end]
        return int(length)

    @staticmethod
    def _type_precision(type_definition: str) -> int:
        pos = type_definition.rfind(".")
        if pos < 0:
            return 0
        start = pos + 1
        length = type_definition[start:]
        return int(length)


class Schema(list):
    """
    Схема таблицы. Представляет собой список атрибутов :class:`axipy.Attribute`.
    Организован в виде :class:`list` и свойством `coordsystem`. При задании
    `coordsystem` создается геометрический атрибут.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_schema.py
        :caption: Пример создания
        :pyobject: example_schema_create
        :lines: 2-
        :dedent: 4

    .. literalinclude:: /../../tests/doc_examples/da/test_example_schema.py
        :caption: Пример создания из списка
        :pyobject: example_schema_create_from_list
        :lines: 2-
        :dedent: 4

    Имеет стандартные функции работы со списком.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_schema.py
        :caption: Пример операций
        :pyobject: example_schema_attrs
        :lines: 2-
        :dedent: 4

    .. seealso:: Изменение структуры существующей таблицы см. :attr:`Table.schema`
    """

    _PROPERTIES: Final[str] = "properties"
    _GEOMETRY: Final[str] = "geometry"
    _RENDITION: Final[str] = "rendition"

    def __init__(
        self,
        *attributes: Attribute,
        coordsystem: Optional[CoordSystem] = None,
    ) -> None:
        """
        Конструктор класса.

        Args:
            *attributes: Атрибуты.
            coordsystem: Система координат для геометрического атрибута.
        """
        self._geometry: Optional[dict] = None
        self._rendition: Optional[dict] = None
        self._coordsystem: Optional[CoordSystem] = None

        self.coordsystem = coordsystem

        if type(attributes) is tuple and len(attributes) in [1, 2] and type(attributes[0]) is list:
            attributes_packed: Tuple[List[Attribute]] = cast(Tuple[List[Attribute]], attributes)
            super().__init__(attributes_packed[0])
            if len(attributes) == 2 and type(attributes[1]) is CoordSystem:
                attributes_with_cs: Tuple[List[Attribute], CoordSystem] = cast(
                    Tuple[List[Attribute], CoordSystem], attributes
                )
                self.coordsystem = attributes_with_cs[1]
        else:
            super().__init__(attributes)

    def __repr__(self) -> str:
        cs = f"coordsystem={self.coordsystem!r}" if self.coordsystem is not None else ""
        params = ", ".join(itertools.chain((repr(a) for a in self), (cs,)))
        return f"axipy.Schema({params})"

    @classmethod
    def _from_dict(cls, dictionary: dict) -> "Schema":
        attrs = [Attribute._create(attr) for attr in dictionary[cls._PROPERTIES]]
        geom = dictionary[cls._GEOMETRY] if cls._GEOMETRY in dictionary else None
        crs = geom["prj"]["CS"] if geom is not None else None
        schema = Schema(*attrs, coordsystem=crs)
        schema._geometry = geom
        schema._rendition = dictionary[cls._RENDITION] if cls._RENDITION in dictionary else None
        return schema

    def _to_dict(self) -> dict:
        result: Dict[str, Union[Dict[str, str], List[Dict[str, str]]]] = dict()
        if self._geometry is not None:
            result.update({self._GEOMETRY: self._geometry})
        elif self._coordsystem is not None:
            v = {"type": "geometry", "prj": self._coordsystem._shadow.get_str()}
            result.update({self._GEOMETRY: v})
        if self._rendition is not None:
            result.update({self._RENDITION: self._rendition})
        props = []
        for attr in self:
            p = {
                "name": attr.name,
                "type": attr.type_string,
                "length": attr.length,
                "precision": attr.precision,
            }
            if attr.alias:
                p["alias"] = attr.alias
            if attr.comments:
                p["comments"] = attr.comments
            p["readOnly"] = attr.readOnly
            p["unique"] = attr.unique
            if attr._index is not None:
                p["index"] = attr._index
            props.append(p)
        result.update({self._PROPERTIES: props})
        return result

    @property
    def attribute_names(self) -> List[str]:
        """Возвращает список имен атрибутов."""
        return [a.name for a in self]

    def insert(self, index: int, attr: Attribute) -> None:  # type: ignore[override]
        """
        Вставляет атрибут.

        Args:
            index: Индекс, по которому производится вставка.
            attr: Атрибут.
        """
        if not isinstance(attr, Attribute):
            raise TypeError(f"Argument is not of type {Attribute}.")
        super().insert(index, attr)

    def delete(self, index: Union[int, str]) -> None:
        """
        Удаляет атрибут.

        Args:
            index: Индекс удаляемого атрибута.
        """
        if isinstance(index, str):
            idx = self.index_by_name(index)
        else:
            idx = index
        del self[idx]

    if TYPE_CHECKING:

        def __contains__(self, name: object) -> bool:
            return super().__contains__(name)

    @property
    def coordsystem(self) -> Optional[CoordSystem]:
        """
        Устанавливает или возвращает систему координат.

        Returns:
            None, если СК отсутствует.

        See also:
            :attr:`axipy.Table.is_spatial`

        .. literalinclude:: /../../tests/doc_examples/da/test_example_schema.py
            :caption: Пример использования
            :pyobject: example_schema_coordsystem
            :lines: 2-
            :dedent: 4
        """
        return self._coordsystem

    @coordsystem.setter
    def coordsystem(self, cs: Optional[CoordSystem]) -> None:
        old_cs = self._coordsystem
        if isinstance(cs, CoordSystem):
            self._coordsystem = cs
        elif isinstance(cs, str):
            self._coordsystem = CoordSystem.from_string(cs)
        else:
            self._coordsystem = None
            if self._geometry and "prj" in self._geometry:
                del self._geometry["prj"]
            return
        if self._geometry is not None:
            self._geometry["prj"]["CS"] = self._coordsystem._shadow.get_str()
            if "bounds" in self._geometry and old_cs:
                ct = CoordTransformer(old_cs, self._coordsystem)
                b = self.__list_to_bounds(self._geometry["bounds"])
                if b and b.is_valid:
                    try:
                        b_out = ct.transform(b)
                        self._geometry["bounds"] = [
                            b_out.xmin,
                            b_out.ymin,
                            b_out.xmax,
                            b_out.ymax,
                        ]
                    except Exception:
                        del self._geometry["bounds"]
            if "DESCR" in self._geometry["prj"]:
                self._geometry["prj"]["DESCR"] = self._coordsystem._shadow.description()
            if "RECT" in self._geometry["prj"]:
                r = self._coordsystem.rect
                self._geometry["prj"]["RECT"] = [r.xmin, r.ymin, r.xmax, r.ymax]

    def __list_to_bounds(self, v: list) -> Optional[Rect]:
        if isinstance(v, list) and len(v) == 4:
            return Rect(v[0], v[1], v[2], v[3])
        return None

    def by_name(self, n: str) -> Optional[Attribute]:
        """
        Возвращает атрибут по его имени. Если такого имени не существует, возвращает
        None.

        Args:
            n: Наименование атрибута.
        """
        for a in filter(lambda f: f.name.lower() == n.lower(), self):
            return a
        return None

    def index_by_name(self, n: str) -> int:
        """
        Возвращает индекс по имени атрибута.

        Args:
            n: Наименование атрибута.
        """
        attr = self.by_name(n)
        if attr is None:
            return -1
        return self.index(attr)

    if TYPE_CHECKING:

        def __len__(self) -> int:
            return super().__len__()

    def __setitem__(self, index: int, value: Attribute) -> None:  # type: ignore[override]
        if isinstance(index, str):
            index = self.index_by_name(index)
        if 0 <= index < self.__len__():
            if isinstance(value, Attribute):
                old = self[index]
                if old._index is not None:
                    value._index = old._index
            super().__setitem__(index, value)

    def __getitem__(self, index: int) -> Attribute:  # type: ignore[override]
        idx = index
        if isinstance(index, str):
            idx = self.index_by_name(index)
        if idx == -1:
            raise KeyError(index)
        return super().__getitem__(idx)
