from typing import Any, Dict, List, Optional, cast

from axipy.da.data_object import Table

from ..schema import Schema
from .data_provider import DataProvider
from .source import Destination, Source

__all__: List[str] = [
    "ShapeSource",
    "ShapeDestination",
    "ShapeDataProvider",
]


class ShapeSource(Source):
    pass


class ShapeDestination(Destination):
    pass


class ShapeDataProvider(DataProvider):
    """
    Векторный провайдер SHP.

    Note:
        Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.provider_manager.shp`.
    """

    @staticmethod
    def _identifier() -> str:
        return "OgrDataProvider"

    def get_source(
        self,
        filepath: str,
        encoding: str = "utf8",
        prj: Optional[str] = None,
        alias: Optional[str] = None,
    ) -> Source:
        """
        Создает источник данных.

        Args:
            filepath: Путь к файлу.
            encoding: Кодировка.
            prj: Строка Системы Координат.
            alias: Псевдоним для открываемой таблицы.
        """
        return ShapeSource(
            Source._provider(self.id),
            Source._table_file(filepath),
            Source._alias(alias),
            Source._prj(prj),
            {"charset": encoding},
        )

    def open(
        self,
        filepath: str,
        encoding: str = "utf8",
        prj: Optional[str] = None,
        alias: Optional[str] = None,
    ) -> Table:
        """
        Открывает объект данных.

        Пример::

            shp = provider_manager.shp.open('world.shp', prj='1, 104')

        Args:
            filepath: Путь к файлу.
            encoding: Кодировка.
            prj: Строка Системы Координат.
            alias: Псевдоним для открываемой таблицы.
        """
        return cast(Table, self.get_source(filepath, encoding, prj, alias).open())

    def get_destination(
        self,
        filepath: str,
        schema: Schema,
        encoding: str = "utf8",
        create_data: Optional[dict] = None,
    ) -> Destination:
        """
        Создает назначение объекта данных.

        Args:
            filepath: Путь к файлу.
            schema: Схема таблицы.
            encoding: Кодировка.
            create_data: Параметры OGR. Разделены на два раздел, которые организуются как словари :class:`dict`.

                * `"lco"` - используется при создании слоя как дополнительные параметры. Соответствуют "Layer creation options".
                * `"dsco"` - используется при создании источника данных как дополнительные параметры. Соответствуют "Database creation options".
        """
        pars: Dict[str, Any] = {"charset": encoding}
        if create_data is not None:
            pars["data"] = {"createOptions": create_data}
        return ShapeDestination(schema, Source._provider(self.id), Source._table_file(filepath), pars)

    def create_open(self, filepath: str, schema: Schema, encoding: str = "utf8") -> Table:
        """
        Создает и открывает объект данных.

        Args:
            filepath: Путь к файлу.
            schema: Схема таблицы.
            encoding: Кодировка.
        """
        return cast(Table, self.get_destination(filepath, schema, encoding).create_open())

    def open_temporary(self, schema: Schema) -> Table:
        """
        Создает и открывает временную таблицу.

        Args:
            schema: Схема таблицы.
        """
        return cast(Table, ShapeDestination(schema, Source._provider(self.id), Source._table_file("")).create_open())

    def file_extensions(self) -> List[str]:
        return list(filter(lambda v: v in ["shp", "shz"], super().file_extensions()))
