import codecs
from abc import ABC, abstractmethod
from configparser import ConfigParser
from itertools import chain
from pathlib import Path
from typing import Any, Dict, List, TextIO, Union

__all__: List[str] = [
    "IParser",
    "MANIFEST_FILE_NAME",
    "ENTRY_POINT",
]

_ROOT_SECTION_NAME = "general"
MANIFEST_FILE_NAME = "manifest.ini"
ENTRY_POINT = "__init__.py"


def read_manifest_config(manifest: ConfigParser) -> Dict[str, Any]:
    if _ROOT_SECTION_NAME not in manifest:
        raise Exception("_ROOT_SECTION_NAME not in manifest")

    result: Dict[str, Any] = dict(manifest.items(_ROOT_SECTION_NAME))
    general_section = manifest[_ROOT_SECTION_NAME]

    for name_arg in ("name", "description"):
        if name_arg not in general_section:
            raise Exception(f'"{name_arg}" not in general_section')

    names = {"default": general_section["name"]}
    descriptions = {"default": general_section["description"]}
    if "i18n" in manifest:
        section = manifest["i18n"]
        for key, value in section.items():
            if key.startswith("name_"):
                suffix = key[len("name_") :]
                names[suffix] = value
            elif key.startswith("description_"):
                suffix = key[len("description_") :]
                descriptions[suffix] = value
    result["name"] = names
    result["description"] = descriptions
    if "autoload" in general_section:
        result["autoload"] = general_section.getboolean("autoload")
    if "system" in general_section:
        result["system"] = general_section.getboolean("system")
    if "unloadable" in general_section:
        result["unloadable"] = general_section.getboolean("unloadable")
    return result


class IParser(ABC):

    @classmethod
    def create(cls) -> "IParser":
        return cls()

    @abstractmethod
    def parse(self, location_path: Union[str, Path]) -> dict: ...

    @staticmethod
    @abstractmethod
    def supports(location_path: Union[str, Path]) -> bool:
        return False

    @staticmethod
    def read_config(
        config_fileobject: Union[codecs.StreamReader, TextIO],
    ) -> Dict[str, Any]:
        config = ConfigParser(strict=False)
        config.read_file(chain([f"[{_ROOT_SECTION_NAME}]"], config_fileobject))
        return read_manifest_config(config)
