"""
Загружает данные из json.
"""
import json
import re
from datetime import date
from pathlib import Path
from typing import List, Tuple

from PySide2.QtCore import QRegExp
from PySide2.QtGui import QRegExpValidator
from PySide2.QtWidgets import (
    QCheckBox, QComboBox, QDateEdit, QFormLayout, QFrame, QHBoxLayout, QLabel, QLineEdit,
    QScrollArea, QSpinBox, QWidget, QWizardPage, QPushButton)
from axipy import tr, local_file

from .. import helper
from .. import wizardconverter

path_to_json = Path(local_file("drivers.json"))
# Перевод из индекса в текст(имена типов настроек gdal)
options_types_list = ["open_options", "dataset", "layer"]
# Перевод настроек gdal для отображения в gui
json_keys_to_gui_options = {
    "open_options": tr("Параметры открытия"),
    "dataset": tr("Параметры создания набора данных"),
    "layer": tr("Параметры создания слоя"),
}


def load(lang: str) -> dict:
    """
    Загружает список драйверов с параметрами из json файла, проверяет на наличие необходимых ключей.

    :param lang: Значение языка Аксиомы.
    """
    # Загрузка из json в словарь
    path = path_to_json
    with open(path, encoding="utf-8") as f:
        drivers = json.load(f)

    loc = re.compile("^[a-z]{2}$")  # Шаблон ключа локализации, например "en", "ru"

    def modify_nested_dict(d: dict):
        """
        Рекурсивный цикл по словарю и поиск словарей с ключами локализаци(словарей локализации).
        Превращает словари локализации в строковое значение, соответствующее текущей локализации.
        """
        for k, v in d.items():
            if isinstance(v, dict):
                if v.keys() and all(re.match(loc, re_key) for re_key in v.keys()):
                    if v[lang] == "=":
                        d[k] = v["en"]
                    elif v[lang] == "":
                        # Первое не пустое значение
                        non_empty_values = [val for val in v.values() if val != ""]
                        if len(non_empty_values) == 0:
                            d[k] = ""
                        else:
                            d[k] = non_empty_values[0]
                    else:
                        d[k] = v[lang]

                modify_nested_dict(v)

    modify_nested_dict(drivers)

    # Проверка на наличие необходимых ключей драйверов
    for driver_gdal_key, driver_dict_value in drivers.items():
        for elem in ("name", "creation"):
            if elem not in driver_dict_value:
                raise Exception("{} {} {} {}.".format(tr("У формата"), driver_gdal_key,
                                                      tr("нет ключа"), elem))

    return drivers


def get_indx_from_type(type_: str) -> int:
    """
    Получает индекс настройки по её типу.

    :param type_: Тип настройки.
    :return: Индекс настроки.
    """
    return options_types_list.index(type_)


def get_form_view(gui_obj: QWizardPage, driver_name: str, options_list: list, add_param_later: bool = False) -> QScrollArea:
    """
    Создает и возвращает ссылку на созданную разметку формы параметров(в QScrollArea) по имени драйвера и
    списку типов параметров.

    :param QWizardPage gui_obj: Объект QWizardPage.
    :param driver_name: Формат gdal (может быть входной или выходной).
    :param list options_list: Список типов параметров для формы.
    :return: QScrollArea с заполненной формой.
    """

    # Создание формы
    form_layout = QFormLayout()  # layout will be added later to parent
    form_layout.setContentsMargins(0, 0, 0, 0)

    # Получение словаря с параметрами драйверов
    wizard = gui_obj.wizard()  # type: wizardconverter.WizardConverter
    drivers = wizard.drivers

    def fill_form(option_name_arg: str, driver: str, form_arg: QFormLayout):
        """
        Функция, заполняющая форму.
        """
        # Индекс параметра для записи в первый символ строки объектного имени
        option_indx = get_indx_from_type(option_name_arg)
        str_option_indx = str(option_indx)

        def create_form_not_yet_supported() -> None:
            """
            Создание формы для типа настройки если наличие настройки у драйвера неизвестно.
            """
            label_loc = None
            msg1 = tr("В данный момент программа не поддерживает")
            msg2 = tr("Параметры можно добавить вручную в поле ввода, через точку с запятой \";'\".")
            if option_name_arg == "open_options":
                msg = (msg1, tr("параметры открытия для текущего входного формата."), msg2)
                label_loc = QLabel(" ".join(msg))
            elif option_name_arg == "dataset":
                msg = (msg1, tr("параметры создания набора данных для выбранного выходного формата."), msg2)
                label_loc = QLabel(" ".join(msg))
            elif option_name_arg == "layer":
                msg = (msg1, tr("параметры создания слоя для выбранного выходного формата."), msg2)
                label_loc = QLabel(" ".join(msg))
            if label_loc:
                label_loc.setWordWrap(True)
                line_edit = QLineEdit()
                line_edit.setObjectName(str_option_indx + "semicolon")
                form_arg.addRow(label_loc)
                form_arg.addRow(line_edit)

        def create_form_not_supported() -> None:
            """
            Создание формы для типа настройки, которого точно нет у драйвера.
            """
            if add_param_later:
                return None
            label_loc = None
            if option_name_arg == "open_options":
                label_loc = QLabel(tr("Входной формат не имеет параметров открытия."))
            elif option_name_arg == "dataset":
                label_loc = QLabel(tr("Выходной формат не имеет параметров создания набора данных."))
            elif option_name_arg == "layer":
                label_loc = QLabel(tr("Выходной формат не имеет параметров создания слоя."))
            if label_loc:
                form_arg.addRow(label_loc)

        opt_value = drivers[driver].get(option_name_arg, None)
        # option = null or missing
        if opt_value is None:
            create_form_not_yet_supported()
            return None
        # option = true
        if type(opt_value) == bool and opt_value:
            create_form_not_yet_supported()
            return None
        # Если у ключа точно не будет значения
        if opt_value is False:
            create_form_not_supported()
            return None

        # Основное заполнение формы

        params = opt_value
        params_keys = list(params.keys())
        params_values = list(params.values())  # type: List[dict]

        for i, setting in enumerate(params_values):

            # label
            label = QLabel(setting["name"])
            # field
            field = None

            def apply_common_settings(label_arg: QLabel, field_arg: QWidget, setting_arg: dict) -> Tuple:
                """ Общие настройки поля """
                label_arg.setToolTip(params_keys[i])
                label_arg.setWordWrap(True)
                width = wizard.wizard_width * 35 // 100
                label_arg.setFixedWidth(width)

                field_tooltip = setting_arg["tooltip"]
                if field_tooltip == "=name":
                    field_tooltip = setting_arg["name"]
                field_tooltip = helper.add_dot(field_tooltip)
                field_arg.setToolTip(field_tooltip)
                field_arg.setObjectName(str_option_indx + params_keys[i])

                # Иногда поле должно быть выключено при инициализации
                if "disable_if" in setting_arg:
                    disable_if_dict = setting_arg["disable_if"]
                    for key, value in disable_if_dict.items():

                        if key not in params:
                            return label_arg, field_arg

                        current_value = params[key].get("default", None)
                        if not current_value:
                            current_value = list(params[key]["values"].keys())[0]
                        if current_value in helper.ensure_list(value):
                            field_arg.setEnabled(False)

                return label_arg, field_arg

            # LineEdit
            if setting["type"] == "QLineEdit":
                field = QLineEdit()
                if "default" in setting:
                    field.setText(setting["default"])
                if "validator" in setting:
                    field.setValidator(QRegExpValidator(QRegExp(setting["validator"])))
                if "file_name_filter" in setting:
                    # Общие настройки поля
                    label, field = apply_common_settings(label, field, setting)

                    # Добавление кнопки рядом с полем, для выбора файла
                    hbox = QHBoxLayout()
                    hbox.addWidget(field)
                    btn = QPushButton("...")
                    btn.setMaximumWidth(20)
                    obj_name = str_option_indx + params_keys[i]
                    filter_ = setting["file_name_filter"]

                    btn.clicked.connect(
                        lambda x=obj_name, y=filter_:
                        wizard.open_filename_from_options_form(x, y)
                    )
                    hbox.addWidget(btn)
                    # Добавление метки и поля на форму
                    form_arg.addRow(label, hbox)
                    continue

            # ComboBox
            elif setting["type"] == "QComboBox":
                field = QComboBox()
                for v in setting["values"].values():
                    field.addItem(v)
                if "default" in setting and setting["default"] in setting["values"]:
                    field.setCurrentIndex(list(setting["values"]).index(setting["default"]))

                if "disable" in setting:
                    obj_name = str_option_indx + params_keys[i]
                    field.currentIndexChanged.connect(wizard.dynamic_form_validation)

            # CheckBox
            elif setting["type"] == "QCheckBox":
                field = QCheckBox()
                if "default" in setting and setting["default"]:
                    field.setChecked(setting["values"][setting["default"]])
                else:
                    bool_ = list(setting["values"].values())[0]
                    field.setChecked(bool_)

                if "disable" in setting:
                    obj_name = str_option_indx + params_keys[i]
                    field.stateChanged.connect(wizard.dynamic_form_validation)

            # DateEdit
            elif setting["type"] == "QDateEdit":
                field = QDateEdit()
                field.setDate(date.today())

            # SpinBox
            elif setting["type"] == "QSpinBox":
                field = QSpinBox()
                field.setMinimum(setting["values"]["min"])
                field.setMaximum(setting["values"]["max"])
                field.setSingleStep(setting["values"]["single_step"])

            # QLabel - Если требуется просто создать QLabel справа для группировки пар-ов
            elif setting["type"] == "QLabel":
                place_holder = QLabel("")
                label, field = place_holder, label

            label, field = apply_common_settings(label, field, setting)

            # Добавление метки и поля на форму
            form_arg.addRow(label, field)

    # Форма заполняется переданными типами настроек
    for option_name in options_list:
        # Добавление названия параметров
        title_label = QLabel(json_keys_to_gui_options[option_name])
        title_label.setStyleSheet("font-weight: bold")
        form_layout.addRow(title_label)
        # Вызов функции заполняющей форму
        fill_form(option_name, driver_name, form_layout)

    # Создание области прокрутки
    form_widget = QWidget(gui_obj)
    form_widget.setLayout(form_layout)
    scroll_area = QScrollArea(gui_obj)
    scroll_area.setFrameStyle(QFrame.NoFrame)
    scroll_area.setWidget(form_widget)
    scroll_area.setWidgetResizable(True)

    return scroll_area


def read_form(form: QWidget, d_name: str, wizard) -> Tuple[list, list, list]:
    """
    Собирает данные с формы в списки по имени настроек (open_options, dataset, layer).

    :param wizardconverter.WizardConverter wizard:
    :param form: QWidget с разметкой QFormLayout.
    :param d_name: Имя драйвера в формате gdal.
    :return: Списки (Пар-ры открытия, пар-ры набора данных, пар-ры слоя)
    """

    # Пустые списки пар-ов для заполнения данными из формы
    open_options, dataset, layer = [list(), list(), list()]
    settings_dict = {"open_options": open_options, "dataset": dataset, "layer": layer}

    # Список виджетов формы
    widgets_list = form.children()
    # Цикл по виджетам формы
    for widget in widgets_list:
        if (
                isinstance(widget, (QComboBox, QLineEdit, QCheckBox, QDateEdit, QSpinBox)) and
                widget.objectName() not in ["in_line_edit", "out_line_edit"]  # Пропуск полей выбора СК
        ):
            option_type = options_types_list[int(widget.objectName()[:1])]
            name = widget.objectName()[1:]
            gdal_value = None

            if isinstance(widget, QComboBox):
                values = wizard.drivers[d_name][option_type][name]["values"]
                txt = widget.currentText()
                if txt not in ["По умолчанию", "Default"]:
                    gdal_value = helper.get_key(values, txt)

            elif isinstance(widget, QLineEdit):
                x = widget.text()
                # Если поле ввода не пустое и не состоит из пробелов
                if x != "" and not (" " in x and x == len(x) * x[0]):
                    gdal_value = x

                    if name == "semicolon":  # Поддержка параметров через ;
                        settings_dict[option_type].extend(gdal_value.split(";"))
                        continue

            elif isinstance(widget, QCheckBox):
                values = wizard.drivers[d_name][option_type][name]["values"]
                check_state = widget.isChecked()
                gdal_value = helper.get_key(values, check_state)

            elif isinstance(widget, QDateEdit):
                gdal_value = str(widget.date().toPython())

            elif isinstance(widget, QSpinBox):
                gdal_value = str(widget.value())

            if widget.isEnabled() and gdal_value:
                elem = f"{name}={gdal_value}"
                settings_dict[option_type].append(elem)

    return open_options, dataset, layer
