from typing import Any, Generator, Iterable, Optional, Tuple, cast

import axipy


def _init_geom_generator(
    items: Iterable[axipy.Feature], task: axipy.DialogTask
) -> Generator[axipy.Geometry, Any, None]:
    for f in items:
        if task.is_canceled:
            return None
        task.value += 1
        g = f.geometry
        if g is not None:
            yield g
    return None


def _flat_geom_generator(
    items: Iterable[axipy.Feature], task: axipy.DialogTask
) -> Generator[axipy.Geometry, Any, None]:
    for g in _init_geom_generator(items, task):
        if isinstance(g, axipy.GeometryCollection):
            for elem in g:
                yield elem
        else:
            yield g


def _init_feature_geom_generator(
    items: Iterable[axipy.Feature], task: axipy.DialogTask
) -> Generator[Tuple[axipy.Feature, axipy.Geometry], Any, None]:
    for f in items:
        if task.is_canceled:
            return None
        task.value += 1
        g = f.geometry
        if g is not None:
            yield f, g
    return None


def _items_to_row_generator(
    items: Iterable[axipy.Feature],
    task: axipy.DialogTask,
) -> Generator[Tuple[float, float, float, float, float, float], Any, None]:
    for f, g in _init_feature_geom_generator(items, task):
        if isinstance(g, axipy.GeometryCollection):
            for g_col_id, elem in enumerate(cast(Iterable[axipy.Geometry], g), 0):
                yield from _polygon_to_row_generator(elem, f.id, g_col_id)
        else:
            yield from _polygon_to_row_generator(g, f.id, -1)


def _polygon_to_row_generator(
    g: axipy.Geometry,
    f_id: int,
    g_col_id: int,
) -> Generator[Tuple[float, float, float, float, float, float], Any, None]:
    if isinstance(g, axipy.Polygon):
        points = g.points
        if points is not None:
            yield from _list_points_to_row_generator(points, f_id, g_col_id, -1)
        holes = g.holes
        if holes is not None:
            for h_id, hole in enumerate(holes, 0):
                yield from _list_points_to_row_generator(hole, f_id, g_col_id, h_id)
    elif isinstance(g, axipy.LineString) and g._shadow.is_closed():
        yield from _list_points_to_row_generator(g.points, f_id, g_col_id, -1)


def _list_points_to_row_generator(
    list_points: Iterable[axipy.Pnt], f_id: int, g_col_id: int, hole_id: int
) -> Generator[Tuple[float, float, float, float, float, float], Any, None]:
    for i, pnt in enumerate(list_points, 0):
        yield pnt.x, pnt.y, f_id, g_col_id, hole_id, i


def _check_if_any_polygon_selected(task: axipy.DialogTask) -> bool:
    task.title = "Проверка наличия полигонов"
    task.message = "Проверка наличия полигонов в выборке."
    selection_table = axipy.data_manager.selection
    if selection_table is None:
        return False

    return any(map(lambda g: isinstance(g, axipy.Polygon), _flat_geom_generator(selection_table.items(), task)))


def _list_points_generator(
    items: Iterable[axipy.Feature], task: axipy.DialogTask
) -> Generator[axipy.ListPoint, Any, None]:
    for g in _flat_geom_generator(items, task):
        if isinstance(g, axipy.Polygon):
            points = g.points
            if points is not None:
                yield points
            holes = g.holes
            if holes is not None:
                for hole in holes:
                    yield hole
        elif isinstance(g, axipy.LineString) and g._shadow.is_closed():
            yield g.points


# in-place
def _change_first_contour(list_point: axipy.ListPoint, i: int) -> None:
    for _ in range(i):
        list_point._shadow.append(list_point._shadow.pop(0))


def _change_first_closed_line_string(list_point: axipy.ListPoint, i: int) -> None:
    n = len(list_point)
    if n < 2:
        return None
    list_point._shadow.remove(n - 1)
    for _ in range(i):
        list_point._shadow.append(list_point._shadow.pop(0))
    list_point._shadow.append(list_point._shadow.at(0))
    return None


def _modify_from_point_row(table: axipy.Table, row: Tuple[float, float, float, float, float, float]) -> None:
    x, y, f_id, g_col_id, h_id, i = row
    f_id, g_col_id, h_id, i = int(f_id), int(g_col_id), int(h_id), int(i)
    items = table.items(ids=[f_id])
    f = next(items)
    g = f.geometry

    if isinstance(g, axipy.GeometryCollection):
        _process_single_g(g[g_col_id], h_id, i)
    else:
        _process_single_g(g, h_id, i)

    f.geometry = g
    table.update(f)


# in-place
def _process_single_g(g: Optional[axipy.Geometry], h_id: int, i: int) -> None:
    if isinstance(g, axipy.Polygon):
        if h_id == -1:
            _change_first_contour(cast(axipy.ListPoint, g.points), i)
        elif h_id >= 0:
            _change_first_contour(cast(axipy.ListPoint, g.holes[h_id]), i)
    elif isinstance(g, axipy.LineString):
        _change_first_closed_line_string(cast(axipy.ListPoint, g.points), i)
