import traceback
from typing import List, Optional, Union

import axipy
from axipy import ActiveToolPanel, Observer, Plugin
from axipy.cs import CoordSystem, LinearUnit, Unit
from axipy.gui import (
    AxipyAcceptableActiveToolHandler,
    DeactivationReason,
    MapTool,
    MapView,
    view_manager,
)
from axipy.utl import Pnt, Rect
from PySide2.QtCore import QItemSelectionModel, QLineF, QModelIndex, QPoint, QPointF, Qt
from PySide2.QtGui import QColor, QKeyEvent, QMouseEvent, QPainter, QPaintEvent, QPen
from PySide2.QtWidgets import QMessageBox, QWidget

from ...helper import Connection
from ..cad_polyline.data_types import AngleType, CPModelMode, CSPoint, RawNodeData
from ..cad_polyline.model import Column, CPModel
from ..cad_polyline.utils import (
    BIG_CROSS_SIZE,
    HALF_LINE_SIZE,
    SMALL_CROSS_SIZE,
    DeactivateAction,
    NotifyContext,
    clone_cs,
    cs_name,
    find_map_view_ref,
    normalize_point,
    print_points_to_log,
    try_save_points,
    try_save_points_on_deactivation_impl,
)
from ..cad_polyline.widget import CPWidget

mb = None


class CadPolyline(MapTool):

    def __init__(self, iface: Plugin, title: str, observer_id: Observer) -> None:
        super().__init__()
        self.title = title
        self.iface = iface
        self.observer_id = observer_id
        # При завершении работы инструмента мы патаемся сохранить данные,
        # чтобы случайно не спрашивать об этом пользователя несколько раз защитимся флагом
        self.__saved = False

        self.__model: Optional[CPModel] = None
        self.__tool_panel = None
        self.__connections = None
        self.__old_map_view_cs = None

    def load(self):
        initial_angle_type = AngleType.CounterClockWiseRight
        self.__model = self.__init_model(initial_angle_type)
        self.__old_map_view_cs = None
        widget = CPWidget(self.iface, self.__model, None)
        self.connect_widget(widget)
        self.__tool_panel = self.__init_tool_panel(widget)
        self.__connections = []  # type: List[Connection]
        self.__connect(view_manager.active_changed, self.reset)
        self.__old_map_view_cs = self.view.coordsystem
        self.widget.set_view(self.view)
        self.__update_model_cs()
        self.__connect(self.view.coordsystem_changed, self.__update_model_cs)
        self.widget.selection_model.currentChanged.connect(self.redraw)
        if view_manager.active.widget == self.view.widget:
            self.__tool_panel.activate()
        if self.view.coordsystem.non_earth:
            self.widget.set_current_unit(self.view.coordsystem.unit)
        elif self.view.coordsystem.lat_lon:
            self.widget.set_current_unit(LinearUnit.m)
        else:
            # пока оставлю тоже самое, что и для план-схемы
            self.widget.set_current_unit(self.view.coordsystem.unit)
        # чистим стек комманд перед началом работы
        self.__model.start_commands_recording()

    def __init_model(self, initial_angle_type: AngleType) -> CPModel:
        model = CPModel(initial_angle_type)
        model.dataChanged.connect(lambda: self.redraw())
        model.modelReset.connect(lambda: self.redraw())

        def update_view_cs(cs: CoordSystem):
            if self.view.coordsystem == cs or cs is None:
                return
            self.view.coordsystem = cs

        model.set_map_view_cs(self.view.coordsystem, self.view.coordsystem)
        model.map_view_coordsystem_chagned.connect(update_view_cs)
        return model

    def __init_tool_panel(self, widget: QWidget) -> AxipyAcceptableActiveToolHandler:
        panel_manager = ActiveToolPanel()
        tool_panel = panel_manager.make_acceptable(self.title, self.observer_id, widget)

        def save_points():
            make_polygon = self.widget.get_settings().make_polygon
            ok = try_save_points(
                self.__model.points_in_cs(self.view.coordsystem),
                self.view,
                NotifyContext(self.title, self.iface),
                make_polygon,
            )
            if ok:
                self.__model.reset()

        tool_panel.accepted.connect(save_points)
        tool_panel.panel_was_closed.connect(self.reset)
        return tool_panel

    def __connect(self, signal, slot, auto_connect=True):
        self.__connections.append(Connection(signal, slot, auto_connect))

    @property
    def widget(self) -> CPWidget:
        return self.__tool_panel.widget

    def connect_widget(self, widget: CPWidget):
        widget.add_point.connect(self.add_point)
        widget.remove_point.connect(self.remove_point)
        widget.angle_type_changed.connect(self.update_angle_type)
        widget.unit_type_chagned.connect(self.update_current_unit)
        widget.clear_all.connect(lambda: self.__model.clear_all())

        def current_index_changed(current: QModelIndex, _previous: QModelIndex):
            # если точка находиться за границей карты, то смещаем карту так, чтобы точку было видно
            index = current.row()
            if index == self.__model.size():
                return None

            pnt = self.__model.points_in_cs(self.view.coordsystem)[index]
            scene_rect = self.view.scene_rect
            offset = 1 / 20
            dw = scene_rect.width * offset
            dh = scene_rect.height * offset
            xmin = scene_rect.xmin + dw
            xmax = scene_rect.xmax - dw
            ymin = scene_rect.ymin + dh
            ymax = scene_rect.ymax - dh
            scene_rect = Rect(xmin=xmin, ymin=ymin, xmax=xmax, ymax=ymax)
            if not scene_rect.contains(pnt):
                self.view.center = pnt

        widget.selection_model.currentChanged.connect(current_index_changed)

        def move_selection_after_delete(_: int):
            if self.__model.is_last_row(widget.selection_model.currentIndex()) and self.__model.size() >= 1:
                new_select_index = self.__model.index(self.__model.size() - 1, Column.first.value)
                flags = QItemSelectionModel.ClearAndSelect | QItemSelectionModel.Rows
                widget.selection_model.select(new_select_index, flags)
                widget.selection_model.setCurrentIndex(new_select_index, QItemSelectionModel.Select)

        self.__model.point_was_deleted.connect(move_selection_after_delete)

        def change_mode(mode: CPModelMode):
            self.__model.mode = mode

        widget.cartesian_mode_changed.connect(change_mode)

    def update_current_unit(self, unit: Unit):
        self.__model.unit = unit

    def update_angle_type(self, angle_type: AngleType):
        self.__model.angle_type = angle_type

    def is_same_cs_for_map_and_editable_layer(self, view: MapView) -> bool:
        if self.view.editable_layer is None or view is None:
            return False
        # Если КС карты и слоя не совпадают, то дадим пользователю возможность
        # работать с координатами в нужной КС.
        return self.view.coordsystem == self.view.editable_layer.coordsystem

    def __update_model_cs(self):
        if self.view is None:
            return
        cs = self.view.coordsystem
        use_macro = self.__old_map_view_cs != cs and self.__model.can_push_map_view_change_cs_command(cs)

        if use_macro:
            # Если между begin и end макроса не будет ни одной комманды, то QUndoStack
            # валится при отмене/повторе
            self.__model.begin_command_macro(axipy.tr(f"установку кс карты {cs_name(cs)}"))
        try:
            is_same = self.is_same_cs_for_map_and_editable_layer(self.view)
            self.widget.set_coord_cs_widget_visibility(not is_same)
            user_cs = self.widget.user_cs_for_view()
            if user_cs is None:
                self.__model.coordsystem = cs
            else:
                self.__model.coordsystem = self.widget.user_cs_for_view()

            if use_macro:
                self.__model.set_map_view_cs(self.view.coordsystem, clone_cs(self.__old_map_view_cs))
        except (Exception,):
            traceback.print_exc()
        finally:
            if use_macro:
                self.__model.end_commamnd_macro()
        self.__old_map_view_cs = self.view.coordsystem

    def remove_point(self, row: int):
        self.__model.removeRow(row)

    def add_point(self):
        pnt = normalize_point(self.view, self.view.scene_rect.center)  # type: Pnt

        if self.__model.is_empty:
            node_data = RawNodeData(CSPoint(pnt, self.view.coordsystem))
        else:
            points = self.__model.points_in_cs(self.view.coordsystem)
            cs_point = CSPoint(points[self.__model.size() - 1], self.view.coordsystem)
            node_data = RawNodeData(cs_point)
        added_row = self.__model.size()
        if self.widget.selection_model.hasSelection():
            # добавляем новую строчку после выделенной
            added_row = self.widget.selection_model.currentIndex().row() + 1
            self.__model.add_point(node_data, added_row)
        else:
            self.__model.add_point(node_data)
        # Выделяем добавленную строчку
        self.widget._table_view.setFocus()
        index = self.__model.index(added_row, 0)
        flags = QItemSelectionModel.Clear | QItemSelectionModel.SelectCurrent | QItemSelectionModel.Rows
        self.widget.selection_model.select(index, flags)
        self.widget.selection_model.setCurrentIndex(index, flags)

    def mouseReleaseEvent(self, event: QMouseEvent) -> Optional[bool]:
        # Блокируем контекстное меню
        if event.button() == Qt.RightButton:
            return self.BlockEvent
        # Чтобы работало перемещение карты средней кнопкой мыши
        if event.button() != Qt.LeftButton:
            return self.PassEvent
        scene_pos = self.snap(event.pos())  # type: Union[Pnt, QPoint]
        if not self.is_snapped():
            scene_pos = self.to_scene(scene_pos)  # type: Pnt
        point = CSPoint(normalize_point(self.view, scene_pos), self.view.coordsystem)
        selection_model = self.widget.selection_model
        if selection_model.hasSelection() and not self.__model.is_last_row(selection_model.currentIndex()):
            index = selection_model.currentIndex().row()
            point = point.reproject(self.__model.coordsystem)
            self.__model.update_coordinates(index, point)
        else:
            self.__model.add_point(RawNodeData(point, evaluate_other=True))
        self.redraw()

    def draw_point(self, painter: QPainter, point: QPointF, color: QColor):
        self.draw_points(painter, [point], color)

    @staticmethod
    def draw_points(painter: QPainter, points: List[QPointF], color):
        # При отрисовке точек изпользуется небольшая оптимизация. Сначала готовятся
        # все точки, а потом уже группой рисуются
        painter.save()
        lines = []  # type: List[QLineF]
        for p in points:
            line_h = QLineF(p.x() - HALF_LINE_SIZE, p.y(), p.x() + HALF_LINE_SIZE, p.y())
            line_v = QLineF(p.x(), p.y() - HALF_LINE_SIZE, p.x(), p.y() + HALF_LINE_SIZE)
            lines.append(line_h)
            lines.append(line_v)

        def draw_cross_lines(painter_arg: QPainter, pen: QPen, lines_arg: List[QLineF]):
            painter_arg.setPen(pen)
            painter_arg.drawLines(lines_arg)

        # рисуем белую подложку
        draw_cross_lines(painter, QPen(Qt.white, BIG_CROSS_SIZE), lines)
        # рисуем сами точки, размером поменьше, чтобы был контраст с подложкой
        draw_cross_lines(painter, QPen(color, SMALL_CROSS_SIZE), lines)
        painter.restore()

    @staticmethod
    def draw_lines_between_points(painter: QPainter, points: List[QPointF]):
        if len(points) <= 1:
            return
        old_pen = painter.pen()
        painter.setPen(QPen(Qt.green, 2))
        lines = []  # type: List[QLineF]
        for index in range(1, len(points)):
            lines.append(QLineF(points[index - 1], points[index]))
        painter.drawLines(lines)
        painter.setPen(old_pen)

    def draw_current_point(self, painter: QPainter, points: List[QPointF]):
        sel_model = self.widget.selection_model
        index = sel_model.currentIndex()
        if not index.isValid() or self.__model.is_last_row(index):
            return
        current_point = points[index.row()]
        self.draw_point(painter, current_point, Qt.red)

    def paintEvent(self, event: QPaintEvent, painter: QPainter):
        if self.__model.is_empty:
            return
        points = [self.to_device(p) for p in self.__model.points_in_cs(self.view.coordsystem)]
        self.draw_lines_between_points(painter, points)
        self.draw_points(painter, points, Qt.blue)
        self.draw_current_point(painter, points)
        return super().paintEvent(event, painter)

    def keyPressEvent(self, event: QKeyEvent) -> Optional[bool]:
        if event.key() == Qt.Key_Escape:
            self.reset()
            return
        return super().keyPressEvent(event)

    def sync_save_points_on_deactivation(self) -> DeactivateAction:
        """
        Эта функция блокирует поток выполнения до закрытие диалогового окна. Её безопасно
        вызывать, когда есть уверенность что никиие объекты и окна ещё не успели закрыться.
        Например при обработке метода MapTool::canUnload(reson).
        """
        self.__saved = True
        # Внутри QMessageBox.question используется exec_()
        size = self.__model.size()
        text = axipy.tr(f"Работа инструмента будет завершена. Сохранить точки ({size})? ")
        res = QMessageBox.question(
            view_manager.global_parent, self.title, text, QMessageBox.Yes | QMessageBox.No | QMessageBox.Cancel
        )
        if res == QMessageBox.Cancel:
            self.__saved = False
            return DeactivateAction.Reject
        elif res == QMessageBox.No:
            # точки не сохраняем, но инструмент всё равно нужно вылючить
            return DeactivateAction.Deactivate
        elif res == QMessageBox.Yes:
            make_polygon = self.widget.get_settings().make_polygon
            try_save_points_on_deactivation_impl(
                self.__model.points_in_cs(self.view.coordsystem),
                self.view,
                NotifyContext(self.title, self.iface),
                make_polygon,
            )
            return DeactivateAction.Deactivate

    def async_save_points_on_deactivation(self, text: str):
        """
        Эта функция сохранения точек работает в предположении, что любой объект (карта/данные и тд)
        мог уже закрыть и удалиться. В случае невозможность сохранения набранные точки печатаются
        в лог Аксиомы и питоновской консоли.
        """
        global mb
        mb = QMessageBox(view_manager.global_parent)
        mb.setText(text)
        mb.setStandardButtons(QMessageBox.Yes | QMessageBox.No)
        mb.setIcon(QMessageBox.Question)
        cs = self.view.coordsystem
        points = self.__model.points_in_cs(cs)
        # Ссылка на self.view удалиться сразу после закрытия инструмента, но
        # на самом деле view возможно ещё не удалился, т.к. мы просто переключили
        # карту
        view = find_map_view_ref(self.view)
        context = NotifyContext(self.title, self.iface)
        if view is None or view.widget is None:
            print_points_to_log(context, self.__model.points_in_cs(cs))
            return

        def final_save(button):
            if button != mb.button(QMessageBox.Yes):
                return
            make_polygon = self.widget.get_settings().make_polygon
            try_save_points_on_deactivation_impl(points, view, context, make_polygon)
            if view is not None:
                view_manager.activate(view)

        mb.buttonClicked.connect(final_save)
        # важно использовать именно open() вместо exec() т.к. exec() прокручивает цикл событий что может
        # привести к проблемам при удалении и закрытии карты/данных
        mb.open()
        mb.show()

    def try_save_points_on_deactivation(self, block_call=False) -> DeactivateAction:
        """
        Функция пытается сохранить созданные точки либым доступным способом.

        Returns: True если пользователь согласился сохранить точки
        """
        # Проверяем флаг, чтобы случайно не вызвать функцию несколько раз.
        if self.__saved:
            return DeactivateAction.Deactivate
        size = self.__model.size()
        if size == 0:
            return DeactivateAction.Deactivate
        if block_call:
            # если выключение инструмента инициированно явно
            return self.sync_save_points_on_deactivation()
        else:
            # если выключение произошло извне из-за того что изменились условия
            # работоспособности инструмента ( закрытие программы, данных и тп)
            text = axipy.tr(f"Работа инструмента завершена. Сохранить точки ({size})? ")
            self.async_save_points_on_deactivation(text)

    def unload(self):
        try:
            self.try_save_points_on_deactivation()
        except (Exception,):
            traceback.print_exc()
        # отключаем все соедениения
        for conn in self.__connections:
            conn.disconnect()
        # отключаем панель инструментов
        self.__tool_panel.deactivate()
        self.redraw()

    def canUnload(self, reason: DeactivationReason) -> bool:
        if self.__model.size() == 0:
            return True
        if reason in (
            DeactivationReason.ObjectClose,
            DeactivationReason.ActionClick,
            DeactivationReason.ActionShortcut,
            DeactivationReason.WindowClose,
            DeactivationReason.LayerClick,
        ):
            deactivate_action = self.try_save_points_on_deactivation(block_call=True)
            if deactivate_action == DeactivateAction.Deactivate:
                self.reset()
                return True
            elif deactivate_action == DeactivateAction.Reject:
                return False
        return True
