from typing import Union

from PySide2.QtCore import QRectF, QPointF, QSizeF, Signal, QObject, Slot
from PySide2.QtPrintSupport import QPrinter
from axipy.cpp_render import (
    ShadowReport,
    ShadowGeometryReportItem,
    ShadowMapReportItem,
    ShadowRasterReportItem,
    ShadowTableReportItem,
    ShadowLegendReportItem,
    ShadowScaleBarReportItem,
    ShadowReportItems
)

from axipy.cs import LinearUnit
from axipy.da import Geometry, Style, Table
from axipy.utl import Rect, Pnt
from .context import Context
from .legend import Legend
from .map_ import Map

__all__ = [
    "ReportItem",
    "ReportItems",
    "Report",
    "GeometryReportItem",
    "MapReportItem",
    "RasterReportItem",
    "TableReportItem",
    "LegendReportItem",
    "ScaleBarReportItem",
]
from axipy._internal._shadow_instance_factory import _shadow_manager


class ReportItem:
    """Базовый класс элемента отчета.
    """

    @property
    def rect(self) -> Rect:
        """Размер (ограничивающий прямоугольник) элемента отчета в единицах измерения отчета.
        """
        return Rect.from_qt(self._shadow.get_rect())

    @rect.setter
    def rect(self, r: Union[Rect, QRectF]):
        self._shadow.set_rect(Rect._rect_value_to_qt(r))

    def intersects(self, checkRect: Union[Rect, QRectF]):
        """Пересекается ли с переданным прямоугольником.

        Args:
            checkRect: Прямоугольник для анализа.
        """
        return self._shadow.intersects(Rect._rect_value_to_qt(checkRect))

    @property
    def border_style(self) -> Style:
        """Стиль обводки элемента отчета."""
        return Style._wrap(self._shadow.get_borderStyle())

    @border_style.setter
    def border_style(self, s: Style):
        self._shadow.set_borderStyle(s._shadow)

    @property
    def fill_style(self) -> Style:
        """Стиль заливки элемента отчета."""
        return Style._wrap(self._shadow.get_fillStyle())

    @fill_style.setter
    def fill_style(self, s: Style):
        self._shadow.set_fillStyle(s._shadow)


class ReportItems:
    """Список элементов отчета."""

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap(cls, _shadow):
        result = cls.__new__(cls)
        result._shadow = _shadow
        return result

    def add(self, item: ReportItem):
        """Добавляет новый элемент в отчет.

        Args:
            item: Вставляемый элемент
        """
        self._shadow.add(item._shadow)

    @property
    def count(self) -> int:
        """Количество элементов отчета в текущем отчете на данный момент.
        """
        return self.__len__()

    def at(self, idx: int) -> ReportItem:
        """Возвращает элемент отчета по его индексу.

        Args:
            idx: Индекс.

        Returns:
            Элемент отчета. Возвращает None в случае, если не найдено.
        """
        i = ShadowReportItems.at(self._shadow, idx)
        if isinstance(i, ShadowGeometryReportItem):
            return GeometryReportItem.create(i)
        elif isinstance(i, ShadowMapReportItem):
            return MapReportItem.create(i)
        elif isinstance(i, ShadowRasterReportItem):
            return RasterReportItem.create(i)
        elif isinstance(i, ShadowTableReportItem):
            return TableReportItem.create(i)
        elif isinstance(i, ShadowLegendReportItem):
            return LegendReportItem.create(i)
        elif isinstance(i, ShadowScaleBarReportItem):
            return ScaleBarReportItem.create(i)
        else:
            return None

    def __getitem__(self, idx: int):
        return self.at(idx)

    def remove(self, idx: int):
        """Удаляет элемент по его индексу. Если индекс корректен, элемент будет удален.

        Args:
            idx:  Индекс удаляемого элемента.
        """
        self._shadow.remove(idx)

    def __len__(self):
        return self._shadow.count()

    def __iter__(self):
        return (self.__getitem__(idx) for idx in range(self.__len__()))


class Report:

    """План отчета для последующей печати.

    .. literalinclude:: /../../tests/doc_examples/render/test_example_report.py
        :caption: Пример создания пустого отчета и вывод его в pdf.
        :pyobject: test_run_example_report
        :lines: 3-
        :dedent: 4
    """

    class _ReportSignals(QObject):
        _need_redraw = Signal(QRectF)

    def __init__(self, printer: QPrinter):
        self._signals = Report._ReportSignals()
        self._signal = self._signals._need_redraw
        self._shadow = ShadowReport(printer)
        self._items = None
        self._shadow.needRedraw.connect(self._need_redraw)

    @classmethod
    def _wrap(cls, _shadow: ShadowReport):
        result = cls.__new__(cls)
        result._shadow = _shadow
        result._items = None
        return result

    @property
    def need_redraw(self) -> Signal:
        """
        Сигнал о необходимости перерисовки части или всего отчета.

        Args:
            rect: Часть отчета, которую необходимо обновить.

        :rtype: Signal[QRectF]
        """
        return self._signal

    @Slot(QRectF)
    def _need_redraw(self, rect: QRectF):
        self._signal.emit(Rect._rect_value_to_qt(rect))

    def draw(self, context: Context):
        """Выводит отчета в заданном контексте.

        Args:
            context: Контекст, в котором будет отрисован отчет.

        """
        self._shadow.draw(context._shadow)

    @property
    def items(self) -> ReportItems:
        """Элементы отчета."""
        if self._items is None:
            self._items = ReportItems._wrap(ShadowReport.get_items(self._shadow))
        return self._items

    @property
    def horisontal_pages(self) -> int:
        """Количество страниц отчета по горизонтали. """
        return self._shadow.get_horisontal_pages()

    @horisontal_pages.setter
    def horisontal_pages(self, v: int):
        self._shadow.set_horisontal_pages(v)

    @property
    def vertical_pages(self) -> int:
        """Количество страниц отчета по вертикали. """
        return self._shadow.get_vertical_pages()

    @vertical_pages.setter
    def vertical_pages(self, v: int):
        self._shadow.set_vertical_pages(v)

    @property
    def unit(self) -> LinearUnit:
        """Единицы измерения в отчете."""
        if self._shadow is None:
            return None
        return LinearUnit._wrap(ShadowReport.unit(self._shadow))

    @unit.setter
    def unit(self, unit: LinearUnit):
        self._shadow.set_unit(unit._shadow)

    @property
    def page_size(self) -> QSizeF:
        """Размеры одного листа отчета. """
        return self._shadow.page_size()

    @property
    def name(self) -> str:
        """Наименование отчета. """
        return self._shadow.get_name()

    @name.setter
    def name(self, n):
        self._shadow.set_name(n)

    def fill_on_pages(self):
        """Максимально заполняет страницу(ы) отчета. При этом элементы отчета пропорционально масштабируются."""
        self._shadow.fill_on_pages()
    
    def fit_pages(self):
        """Подгоняет число страниц отчета под размер существующих элементов отчета. При этом параметры элементов отчета не меняются."""
        self._shadow.fit_pages()


class GeometryReportItem(ReportItem):
    """Элемент отчета типа геометрия.
    
    .. literalinclude:: /../../tests/doc_examples/render/test_example_report.py
        :caption: Пример создания полигона и добавления его в отчет.
        :pyobject: test_run_example_report_geometry
        :lines: 3-
        :dedent: 4

    .. literalinclude:: /../../tests/doc_examples/render/test_example_report.py
        :caption: Пример создания текста и добавления его в отчет.
        :pyobject: test_run_example_report_geometry_text
        :lines: 3-
        :dedent: 4
    """

    def __init__(self):
        self._shadow = ShadowGeometryReportItem()

    @classmethod
    def create(cls, _shadow: ShadowGeometryReportItem):
        obj = cls.__new__(cls)
        obj._shadow = _shadow
        return obj

    @property
    def geometry(self) -> Geometry:
        """Геометрическое представление объекта."""
        return Geometry._wrap(self._shadow.get_geometry())

    @geometry.setter
    def geometry(self, g: Geometry):
        self._shadow.set_geometry(g._shadow)

    @property
    def style(self) -> Style:
        """Стиль геометрического представления объекта."""
        return Style._wrap(self._shadow.get_style())

    @style.setter
    def style(self, s: Style):
        self._shadow.set_style(s._shadow)


class MapReportItem(ReportItem):
    """Элемент отчета, основанный на созданной ранее карте. 

    Note:
        Перед созданием элемента отчета необходимо предварительно создать карту,
        на основе которой будет создан элемент отчета.

    Args:
        rect: Размер элемента отчета в единицах измерения отчета.
        map: Карта, на базе которой будет создан элемент отчета.

    .. literalinclude:: /../../tests/doc_examples/render/test_example_report.py
        :caption: Пример создания карты и добавления ее в отчет.
        :pyobject: test_run_example_report_map
        :lines: 4-8
        :dedent: 4
    """

    def __init__(self, rect: Union[Rect, QRectF], map: Map):
        self._shadow = ShadowMapReportItem(
            Rect._rect_value_to_qt(rect), map._shadow)

    @classmethod
    def create(cls, _shadow: ShadowMapReportItem):
        obj = cls.__new__(cls)
        obj._shadow = _shadow
        return obj

    def map(self) -> Map:
        """Возвращает элемент типа карта, на основании которой создается элемент отчета."""
        return Map._wrap(self._shadow.map())

    @property
    def center(self) -> Pnt:
        """Центр карты в координатах карты."""
        return Pnt.from_qt(self._shadow.get_center())

    @center.setter
    def center(self, c: Union[Pnt, QPointF]):
        self._shadow.set_center(Pnt._point_value_to_qt(c))

    @property
    def scale(self) -> float:
        """Текущее значение масштаба карты."""
        return self._shadow.get_scale()

    @scale.setter
    def scale(self, s: float):
        self._shadow.set_scale(s)

    @property
    def map_rect(self) -> Rect:
        """Прямоугольник карты в единицах измерения карты."""
        return Rect.from_qt(self._shadow.rectMap())

    def show_all(self):
        """Меняет масштаб карты чтобы показать ее полностью.
        
        Пример замены масштаба для всех элементов отчета::

            for item in reportView.report.items:
                if isinstance(item, MapReportItem):
                    item.show_all()
        """
        self._shadow.show_all()


class RasterReportItem(ReportItem):
    """Элемент отчета, основанный на растре. 

    Note:
        В качестве источника может быть как локальный файл, расположенный в файловой системе,
        так и базе растра, размещенного на Web ресурсе.

    Args:
        rect: Размер элемента отчета в единицах измерения отчета.
        data: Путь к растровому файлу или его URL.

    .. literalinclude:: /../../tests/doc_examples/render/test_example_report.py
        :caption: Пример элемента на базе URL.
        :pyobject: test_run_example_report_raster_url
        :lines: 3-
        :dedent: 4

    .. literalinclude:: /../../tests/doc_examples/render/test_example_report.py
        :caption: Пример элемента на базе локального файла.
        :pyobject: test_run_example_report_raster_file
        :lines: 4-
        :dedent: 4

    """

    def __init__(self, rect: Union[Rect, QRectF], data: str):
        self._shadow = ShadowRasterReportItem(
            Rect._rect_value_to_qt(rect), data, _shadow_manager.core)

    @classmethod
    def create(cls, _shadow: ShadowRasterReportItem):
        obj = cls.__new__(cls)
        obj._shadow = _shadow
        return obj

    @property
    def preserve_aspect_ratio(self) -> bool:
        """Сохранять пропорции при изменении размеров элемента."""
        return self._shadow.get_preserveAspectRatio()

    @preserve_aspect_ratio.setter
    def preserve_aspect_ratio(self, v: bool):
        self._shadow.set_preserveAspectRatio(v)


class TableReportItem(ReportItem):
    """Элемент отчета табличного представления данных.

    Note:
        Позволяет отображать как таблицу целиком, так и накладывая дополнительные ограничения при отображении. 

    Args:
        rect: Размер элемента отчета в единицах измерения отчета.
        table: Таблица.

    .. literalinclude:: /../../tests/doc_examples/render/test_example_report.py
        :caption: Пример.
        :pyobject: test_run_example_report_table
        :lines: 4-
        :dedent: 4
    """

    def __init__(self, rect: Union[Rect, QRectF], table: Table):
        self._shadow = ShadowTableReportItem(
            Rect._rect_value_to_qt(rect), table._shadow)

    @classmethod
    def create(cls, _shadow: ShadowTableReportItem):
        obj = cls.__new__(cls)
        obj._shadow = _shadow
        return obj

    @property
    def columns(self) -> list:
        """Перечень наименований для отображения. Если задать пустой список, будут отображены все поля таблицы."""
        return self._shadow.get_selectedColumns()

    @columns.setter
    def columns(self, l):
        self._shadow.set_selectedColumns(l)

    def refreshValues(self):
        """"Обновление данных из таблицы."""
        self._shadow.refreshValues()

    @property
    def row_from(self) -> int:
        """Номер первой строки из таблицы или запроса."""
        return self._shadow.get_rowFrom()

    @row_from.setter
    def row_from(self, l):
        self._shadow.set_rowFrom(l)

    @property
    def row_count(self) -> int:
        """Количество записей. Если указано -1, то берутся все оставшиеся записи."""
        return self._shadow.get_rowCount()

    @row_count.setter
    def row_count(self, l):
        self._shadow.set_rowCount(l)

    @property
    def start_number(self) -> int:
        """Нумерация записей. Порядковый номер первой записи."""
        return self._shadow.get_startNumber()

    @start_number.setter
    def start_number(self, l):
        self._shadow.set_startNumber(l)

    @property
    def show_row_number(self) -> bool:
        """Показывать ли номера строк."""
        return self._shadow.get_showRowNumber()

    @show_row_number.setter
    def show_row_number(self, v):
        self._shadow.set_showRowNumber(v)

    def table(self) -> Table:
        """Базовая таблица или запрос."""
        return Table._wrap(self._shadow.table())


class LegendReportItem(ReportItem):
    """Элемент отчета, основанный на легенде векторного или тематического слоя. 

    Args:
        rect: Размер элемента отчета в единицах измерения отчета.
        legend: Предварительно созданная легенда. Она может относиться как к векторному, так и к тематическому слою.

    .. literalinclude:: /../../tests/doc_examples/render/test_example_report.py
        :caption: Пример создания легенды для тематического слоя.
        :pyobject: test_run_example_report_legend
        :lines: 4-
        :dedent: 4
    """

    def __init__(self, rect: Union[Rect, QRectF], legend: Legend):
        self._shadow = ShadowLegendReportItem(
            Rect._rect_value_to_qt(rect), legend._shadow)

    @classmethod
    def create(cls, _shadow: ShadowLegendReportItem):
        obj = cls.__new__(cls)
        obj._shadow = _shadow
        return obj

    @property
    def legend(self) -> Legend:
        """Легенда на базе которой создан элемент отчета."""
        return Legend._wrap_typed(ShadowLegendReportItem.legend(self._shadow))


class ScaleBarReportItem(ReportItem):
    """Элемент отчета - масштабная линейка для карты.

    .. literalinclude:: /../../tests/doc_examples/render/test_example_report.py
        :caption: Пример создания масштабной линейки на базе существующего элемента - карты.
        :pyobject: test_run_example_report_map
        :lines: 10-11
        :dedent: 4
    """

    def __init__(self, rect: Union[Rect, QRectF], map: MapReportItem):
        self._shadow = ShadowScaleBarReportItem(
            Rect._rect_value_to_qt(rect), map._shadow)

    @classmethod
    def create(cls, _shadow: ShadowScaleBarReportItem):
        obj = cls.__new__(cls)
        obj._shadow = _shadow
        return obj
