from typing import List

from axipy.da.data_object import DataObject
from .data_provider import DataProvider
from .source import Source

__all__ = [
    "OgrSource",
    "OgrDataProvider",
]


class OgrSource(Source):
    pass


class OgrDataProvider(DataProvider):
    """Провайдер для векторных данных OGR.
    
    Note:
        Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.provider_manager.ogr`.
    """
    _identifier = 'OgrDataProvider'

    def get_source(self, data: str, dataobject: str, alias: str = None, encoding: str = 'utf8') -> Source:
        """Создает источник данных.

        Args:
          data: Источник данных или имя файла.
          dataobject: Наименование таблицы
        """
        return OgrSource(
            Source._provider(self.id),
            Source._alias(alias),
            {
                'src': data,
                'dataobject': dataobject,
                'charset': encoding
            }
        )

    def open(self, data: str, dataobject: str, alias: str = None, encoding: str = 'utf8') -> DataObject:
        """Открывает объект данных.

        Args:
          data: Источник данных или имя файла.
          dataobject: Наименование таблицы
          alias: Псевдоним для открываемой таблицы.
          encoding: Кодировка.
        """
        return self.get_source(data, dataobject, alias).open()

    def file_extensions(self) -> List[str]:
        return list(filter(lambda v: v not in ['shp', 'shz'], super().file_extensions()))

    def get_destination(self):
        """
        Attention:
            Не поддерживается.

        Raises:
            NotImplementedError
        """
        raise NotImplementedError

    def create_open(self):
        """
        Attention:
            Не поддерживается.

        Raises:
            NotImplementedError
        """
        raise NotImplementedError
