import typing
from enum import IntFlag
from typing import Union, Iterator, List, Iterable, overload, Any, cast

from PySide2.QtCore import QCoreApplication
from axipy._internal._utils import _NoInitTypeError
from axipy.cpp_core_dp import (
    RasterFacade as CppRasterFacade,
    ShadowDataObject,
    ShadowQueryTable,
    ShadowSelectionTable,
    ShadowTable,
    AccessMode,
    ShadowTransactionalTable,
    ShadowRaster,
    ShadowRasteredTable
)
from axipy.cpp_render import ShadowCosmeticTable
from axipy.cs import CoordSystem
from axipy.utl import Rect

from .feature import Feature, _FeatureIterator
from .geometry import Geometry
from .raster import GCP
from .schema import Schema

if typing.TYPE_CHECKING:
    from PySide2.QtGui import QTransform
    from PySide2.QtCore import QRectF, QSize, SignalInstance

__all__ = [
    'DataObject',
    'Table',
    'SupportedOperations',
    'Raster',
    'QueryTable',
    'SelectionTable',
    'CosmeticTable',
    'RasteredTable'
]


class _DataObject:
    _inner_shadow: ShadowDataObject

    def __init__(self) -> None:
        raise _NoInitTypeError

    @classmethod
    def _wrap_typed(cls, shadow: ShadowDataObject) -> 'DataObject':
        result = cast(DataObject, cls.__new__(cls))
        result._shadow = shadow
        return result

    @classmethod
    @overload
    def _wrap(cls, obj: None) -> None:
        ...

    @classmethod
    @overload
    def _wrap(cls, obj: ShadowQueryTable) -> 'QueryTable':
        ...

    @classmethod
    @overload
    def _wrap(cls, obj: ShadowCosmeticTable) -> 'CosmeticTable':
        ...

    @classmethod
    @overload
    def _wrap(cls, obj: ShadowSelectionTable) -> 'SelectionTable':
        ...

    @classmethod
    @overload
    def _wrap(cls, obj: ShadowTransactionalTable) -> 'Table':
        ...

    @classmethod
    @overload
    def _wrap(cls, obj: ShadowTable) -> 'Table':
        ...

    @classmethod
    @overload
    def _wrap(cls, obj: ShadowRaster) -> 'Raster':
        ...

    @classmethod
    @overload
    def _wrap(cls, obj: ShadowRasteredTable) -> 'RasteredTable':
        ...

    @classmethod
    @overload
    def _wrap(cls, obj: ShadowDataObject) -> 'DataObject':
        ...

    @classmethod
    @overload
    def _wrap(cls, obj: Any) -> None:
        ...

    @classmethod
    def _wrap(cls, obj):
        if obj is None:
            return None

        if isinstance(obj, ShadowQueryTable):
            obj_type = QueryTable
        elif isinstance(obj, ShadowCosmeticTable):
            obj_type = CosmeticTable
        elif isinstance(obj, ShadowSelectionTable):
            obj_type = SelectionTable
        elif isinstance(obj, ShadowTransactionalTable):
            obj_type = Table
        elif isinstance(obj, ShadowTable):
            obj_type = Table
        elif isinstance(obj, ShadowRaster):
            obj_type = Raster
        elif isinstance(obj, ShadowRasteredTable):
            obj_type = RasteredTable
        elif isinstance(obj, ShadowDataObject):
            obj_type = DataObject
        else:
            obj_type = None
        if obj_type is None:
            return None
        return obj_type._wrap_typed(obj)

    @property
    def _shadow(self) -> ShadowDataObject:
        if self._inner_shadow is None:
            raise RuntimeError("Internal C++ object is already deleted.")
        return self._inner_shadow

    @_shadow.setter
    def _shadow(self, value: ShadowDataObject) -> None:
        self._inner_shadow = value


class DataObject(_DataObject):
    """
    Объект данных.

    Открываемые объекты из источников данных представляются объектами
    этого типа. Возможные реализации: таблица, растр, грид, чертеж,
    панорама, и так далее.

    Пример::

        table = provider_manager.openfile('path/to/file.tab')
        ...
        table.close()  # Закрывает таблицу

    Для закрытия объекта данных можно использовать менеджер контекста -
    выражение :any:`with<with>`. В таком случае таблица будет закрыта при выходе
    из блока. См. :meth:`close`.

    Пример::

        with provider_manager.openfile('path/to/file.tab') as raster:
            ...
            # При выходе из блока растр будет закрыт

    """

    # Свойства

    @property
    def name(self) -> str:
        """Название объекта данных."""
        return self._shadow.name()

    @name.setter
    def name(self, name: str):
        self._shadow.setName(name)

    @property
    def provider(self) -> str:
        """Провайдер изначального источника данных."""
        return self._shadow.provider()

    @property
    def is_hidden(self) -> bool:
        return self._shadow.isHidden()

    @property
    def is_spatial(self) -> bool:
        """Признак того, что объект данных является пространственным."""
        return self._shadow.isSpatial()

    @property
    def properties(self) -> dict:
        """Дополнительные свойства объекта данных."""
        return self._shadow.properties()

    # Методы

    def close(self) -> None:
        """
        Пытается закрыть таблицу.

        Raises:
            RuntimeError: Ошибка закрытия таблицы.

        Note:

            Объект данных не всегда может быть сразу закрыт. Например, для
            таблиц используется транзакционная модель редактирования и перед
            закрытием необходимо сохранить или отменить изменения, если они
            есть.
            См. :attr:`Table.is_modified`.
        """
        if not isinstance(self._shadow, ShadowCosmeticTable):
            self._shadow.closeRequest()
            self._shadow = None
            QCoreApplication.processEvents()
        else:
            raise RuntimeError("It is not allowed to close cosmetic table")

    # Сигналы

    @property
    def destroyed(self) -> 'SignalInstance':
        """Сигнал оповещения об удалении объекта."""
        # noinspection PyTypeChecker
        return self._shadow.innerDestroyed

    # Специальные методы

    def __eq__(self, other) -> bool:
        return self._shadow.isEqual(other._shadow)

    def __enter__(self) -> 'DataObject':
        return self

    def __exit__(self, exception_type, exception_value, traceback) -> None:
        self.close()

    def __repr__(self) -> str:
        if self._shadow is None:
            return f"<axipy.{self.__class__.__name__} name=None, provider=None>"
        else:
            return f"<axipy.{self.__class__.__name__} name={self.name}, provider={self.provider}>"


class SupportedOperations(IntFlag):
    """
    Флаги доступных операций.

    Реализованы как перечисление :class:`enum.IntFlag` и поддерживают побитовые
    операции.

    .. csv-table::
        :header: Атрибут, Значение

        `Empty`, 0
        `Insert`, 1
        `Update`, 2
        `Delete`, 4
        `Write`, `Insert` | `Update` | `Delete` = 7
        `Read`, 8
        `ReadWrite`, `Read` | `Write` = 15

    .. literalinclude:: /../../tests/doc_examples/da/test_example_supported_operations.py
        :caption: Пример использования
        :pyobject: example_supported_operations
        :lines: 2-
        :dedent: 4

    See Also:
        :class:`enum.IntFlag`, :attr:`axipy.Table.supported_operations`
    """
    Empty = 0
    Insert = 1
    Update = 2
    Delete = 4
    Write = 7
    Read = 8
    ReadWrite = 15


class Table(DataObject):
    """
    Таблица.

    Менеджер контекста сохраняет изменения и закрывает таблицу.

    Пример::

        with provider_manager.openfile('path/to/file.tab') as table:
            ...
            # При выходе из блока таблица будет сохранена и закрыта

    See also:
        :meth:`commit`, :meth:`DataObject.close`.
    """

    @property
    def data_changed(self) -> 'SignalInstance':
        """
        Сигнал об изменении данных таблицы. Испускается когда были изменены данные таблицы.
        В качестве параметра передается дополнительная информация в виде dict.

        .. csv-table:: Доступные параметры
            :header: Наименование, Значение, Описание

            operation, insert, Произведена вставка данных
            operation, update, Данные были обновлены
            operation, remove, Данные были удалены
            operation, unknown, Тип операции не определен
            ids, list , "Перечень затронутых идентификаторов, если он доступен"

        .. literalinclude:: /../../tests/doc_examples/da/test_example_table.py
            :caption: Пример подписки на изменение таблицы.
            :pyobject: test_run_example_table_change_table
            :lines: 3-
            :dedent: 4
        """
        # noinspection PyTypeChecker
        return self._shadow.dataChanged

    @property
    def schema_changed(self) -> 'SignalInstance':
        """Сигнал об изменении схемы таблицы. Испускается когда была изменена структура таблицы."""
        # noinspection PyTypeChecker
        return self._shadow.schemaChanged

    @property
    def is_editable(self) -> bool:
        """Признак того, что таблица является редактируемой."""
        return self._shadow.accessMode() == AccessMode.Access_ReadWrite

    @property
    def is_temporary(self) -> bool:
        """
        Признак того, что таблица является временной.
        Это в первую очередь касается таблиц, созданных в памяти.
        А также таблица косметического слоя.
        """
        return self._shadow.isTemporary()

    @property
    def is_modified(self) -> bool:
        """Таблица содержит несохраненные изменения."""
        return self._shadow.hasModified() if self.is_editable else False

    @property
    def supported_operations(self) -> SupportedOperations:
        """
        Доступные операции.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_supported_operations.py
            :caption: Пример использования
            :pyobject: example_supported_operations
            :lines: 2-
            :dedent: 4
        """
        return SupportedOperations(self._shadow.supportedOperations())

    @property
    def schema(self) -> Schema:
        """Схема таблицы."""
        return Schema._from_dict(self._shadow.schema())

    @schema.setter
    def schema(self, new_schema: Schema):
        self.ensure_editable()
        self._shadow.setSchema(new_schema.to_dict())

    @property
    def coordsystem(self) -> Union[CoordSystem, None]:
        """Система координат таблицы."""
        return CoordSystem._wrap(ShadowTable.coordSystem(self._shadow))

    def ensure_editable(self) -> None:
        if not self.is_editable:
            raise RuntimeError('Table is not editable')

    def insert(self, features: Union[Feature, Iterable[Feature]]) -> None:
        """
        Вставляет записи в таблицу.

        Args:
            features: Записи для вставки.
        """
        self.ensure_editable()
        if isinstance(features, Feature):
            features = [features]
        self._shadow.insert(f._shadow for f in features)
        QCoreApplication.processEvents()

    def update(self, features: Union[Feature, Iterable[Feature]]) -> None:
        """
        Обновляет записи в таблице.

        Args:
            features: Записи для обновления.

        При обновлении проверяется :attr:`Feature.id`. Если запись с таким
        идентификатором не найдена, то она пропускается.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_table.py
            :caption: Пример использования
            :pyobject: example_table_update
            :lines: 2-
            :dedent: 4

        See also:
            :attr:`Feature.id`, :meth:`commit`, :attr:`is_modified`.
        """
        self.ensure_editable()
        if isinstance(features, Feature):
            features = [features]
        self._shadow.update(f._shadow for f in features)
        QCoreApplication.processEvents()

    def remove(self, ids: Union[int, Iterator[int]]):
        """
        Удаляет записи из таблицы.

        Args:
            ids: Идентификаторы записей для удаления.
        """
        self.ensure_editable()
        if isinstance(ids, int):
            ids = [ids]
        self._shadow.remove(ids)
        QCoreApplication.processEvents()

    def commit(self) -> None:
        """Сохраняет изменения в таблице.

        Если таблица не содержит несохраненные изменения, то команда
        игнорируется.

        Raises:
            RuntimeError: Невозможно сохранить изменения.
        """
        self.ensure_editable()
        if self.is_temporary:
            raise RuntimeError("Невозможно сохранить изменения во временной таблице.")
        self._shadow.commit()
        QCoreApplication.processEvents()

    def rollback(self) -> None:
        """
        Отменяет несохраненные изменения в таблице.

        Если таблица не содержит несохраненные изменения, то команда
        игнорируется.
        """
        self.ensure_editable()
        self._shadow.rollback()
        QCoreApplication.processEvents()

    @property
    def can_undo(self) -> bool:
        """
        Возможен ли откат на один шаг назад.
        """
        return self._is_transaction_table and self._shadow.canUndo(1)

    def undo(self, steps: int = 1) -> None:
        """
        Производит откат вперед на заданное количество шагов. При этом возвращается состояние до последней отмены.
        
        Args:
            steps: Количество шагов.
        """
        if self.can_undo:
            self._shadow.undo(steps)

    @property
    def can_redo(self) -> bool:
        """
        Возможен ли откат на один шаг вперед.
        """
        return self._is_transaction_table and self._shadow.canRedo(1)

    def redo(self, steps: int = 1) -> None:
        """
        Производит откат вперед на заданное количество шагов. При этом возвращается состояние до последней отмены.
        
        Args:
            steps: Количество шагов.
        """
        if self.can_redo:
            self._shadow.redo(steps)

    # noinspection PyPep8Naming
    def itemsByIds(self, ids: List[int]) -> Iterator[Feature]:
        """
        Запрашивает записи по списку :class:`list` с идентификаторами записей,
        либо перечень идентификаторов в виде списка. Идентификаторы несохраненных
        записей имеют отрицательные значения.

        Args:
            ids: Список идентификаторов.

        Returns:
            Итератор по записям.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_table.py
            :caption: Пример
            :pyobject: test_run_example_table_query_ids
            :lines: 3-
            :dedent: 4
        """
        return _FeatureIterator(self._shadow.itemsById(ids))

    # noinspection PyPep8Naming
    def itemsInObject(self, obj: Geometry) -> Iterator[Feature]:
        """
        Запрашивает записи с фильтром по геометрическому объекту.

        Args:
            obj: Геометрия. Если для нее не задана СК, используется СК таблицы.

        Returns:
            Итератор по записям.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_table.py
            :caption: Пример запроса по полигону
            :pyobject: test_run_example_table_query_obj
            :lines: 3-
            :dedent: 4
        """
        geom = obj
        if isinstance(geom, Geometry) and geom.coordsystem is None:
            geom = obj.clone()
            geom.coordsystem = self.coordsystem
        return _FeatureIterator(self._shadow.itemsInObject(geom._shadow))

    # noinspection PyPep8Naming
    def itemsInRect(self, bbox: Union[Rect, 'QRectF', tuple]) -> Iterator[Feature]:
        """
        Запрашивает записи с фильтром по ограничивающему прямоугольнику.

        Args:
            bbox: Ограничивающий прямоугольник.

        Returns:
            Итератор по записям.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_table.py
            :caption: Пример запроса (таблица в проекции `Робинсона`)
            :pyobject: test_run_example_table_query_mbr
            :lines: 3-
            :dedent: 4

        """

        return _FeatureIterator(self._shadow.itemsInMbr(Rect._rect_value_to_qt(bbox)))

    @property
    def hotlink(self) -> str:
        """
        Наименование атрибута таблицы для хранения гиперссылки.

        .. csv-table:: Возможны следующие варианты
            :header: Значение, Описание

            axioma://world.tab, Открывает файл или рабочее пространство в аксиоме
            addlayer://world, Добавляет слой world в текущую карту
            exec://gimp,  Запускает на выполнение программу gimp
            https://axioma-gis.ru/, Открывает ссылку в браузере

        Если префикс отсутствует, то производится попытка запустить по ассоциации.

        See also:
            :attr:`axipy.render.VectorLayer.hotlink`.
        """

        return self._shadow.get_hotlink()

    @hotlink.setter
    def hotlink(self, v: str):
        self._shadow.set_hotlink(v)

    def items(self, bbox: Union[Rect, 'QRectF', tuple] = None, ids: List[int] = None) -> Iterator[Feature]:
        """
        Запрашивает записи, удовлетворяющие параметрам.
        В качестве фильтра может быть указан либо ограничивающий прямоугольник,
        либо перечень идентификаторов в виде списка.

        Args:
            bbox: Ограничивающий прямоугольник.
            ids: Список идентификаторов.

        Returns:
            Итератор по записям.
        """
        if bbox is not None:
            if not self.is_spatial:
                raise RuntimeError('Table is not spatial')
            return self.itemsInRect(bbox)
        if ids is not None:
            return self.itemsByIds(ids)
        return _FeatureIterator(self._shadow.selectAll())

    def count(self, bbox: Union[Rect, 'QRectF', tuple] = None) -> int:
        """
        Возвращает количество записей, удовлетворяющих параметрам.

        Данный метод является наиболее предпочтительным для оценки
        количества записей. При этом используется наиболее оптимальный
        вариант выполнения запроса для каждого конкретного провайдера
        данных.

        Args:
            bbox: Ограничивающий прямоугольник.

        Returns:
            Количество записей.
        """
        if bbox is not None:
            if not self.is_spatial:
                raise RuntimeError('Table is not spatial')
            return self._shadow.countInMbr(Rect._rect_value_to_qt(bbox))
        return self._shadow.count()

    def get_bounds(self) -> Rect:
        """Возвращает область, в которую попадают все данные таблицы."""
        return Rect.from_qt(self._shadow.calculateMbr())

    def __len__(self) -> int:
        return self.count()

    def __iter__(self) -> Iterator[Feature]:
        return self.items()

    def __exit__(self, exception_type, exception_value, traceback) -> None:
        self._commit_if_possible()
        super().__exit__(exception_type, exception_value, traceback)

    _shadow: Union[ShadowTable, ShadowTransactionalTable]

    @property
    def _is_transaction_table(self) -> bool:
        return isinstance(self._shadow, ShadowTransactionalTable)

    def _commit_if_possible(self) -> None:
        if not self.is_editable or self.is_temporary:
            return None

        self.commit()


class SelectionTable(Table):
    """
    Таблица, построенная на основе текущей выборки. Носит временный характер. Создается, когда в
    выборку добавляются объекты и удаляется, когда выборка очищается.

    Пример доступа к выборке с получением идентификаторов::

        for f in data_manager.selection.items():
            print('>>>', f.id)
    """

    @property
    def base_table(self) -> Table:
        """Таблица, на которой основана данная выборка."""
        return DataObject._wrap(self._shadow.originalObject())

    _shadow: ShadowSelectionTable


class QueryTable(Table):
    """
    Таблица, построенная на основе SQL запроса.

    Пример::

        table = provider_manager.openfile('world.tab')
        query = data_manager.query('select * from world')
    """

    @property
    def sql_text(self) -> str:
        """Текст SQL запроса."""
        return self._shadow.sqlText()

    _shadow: ShadowQueryTable


class CosmeticTable(Table):
    """Объект данных косметического слоя :class:`axipy.render.CosmeticLayer`"""


class Raster(DataObject):
    """Растровый объект."""

    @property
    def coordsystem(self) -> Union[CoordSystem, None]:
        """Система координат растра."""
        return CoordSystem._wrap(ShadowRaster.coordSystem(self._shadow))

    @property
    def size(self) -> 'QSize':
        """Размер растра в пикселях."""
        return self._shadow.size()

    @property
    def device_to_scene_transform(self) -> 'QTransform':
        """Матрица преобразования из точек на изображении (пиксели) в точки на карте."""
        return CppRasterFacade.getTransform(self._shadow)

    @property
    def scene_to_device_transform(self) -> 'QTransform':
        """Матрица преобразования из точек на карте в точки на изображении (пиксели)."""
        return self.device_to_scene_transform.inverted()

    def get_gcps(self) -> List[GCP]:
        """Возвращает точки привязки."""
        cpp_result = CppRasterFacade.getGCPs(self._shadow)
        return list(map(GCP._wrap, cpp_result))

    _shadow: ShadowRaster


class RasteredTable(DataObject):
    """Данные из таких источников, как ГИС Панорама и AutoCAD."""

    @property
    def coordsystem(self) -> Union[CoordSystem, None]:
        """
        Система координат.
        """
        return CoordSystem._wrap(ShadowRaster.coordSystem(self._shadow))
    
    @property
    def schema(self) -> Schema:
        """
        Схема таблицы.
        """
        return Schema._from_dict(self._shadow.schema())
    
    @property
    def layers(self):
        """
        Список доступных для запроса данных слоев.
        """
        return self._shadow.layerList()

    def items(self, layer_name: str = None) -> Iterator[Feature]:
        """
        Запрашивает записи из источника.

        Args:
            layer_name Наименование слоя, по ктоорому будет произведена выборка.
            Если значение пустое, выдаются данные по всем слоям.

        Returns:
            Итератор по записям.
        """
        if layer_name is None or layer_name == '':
            return _FeatureIterator(self._shadow.selectAll())
        return _FeatureIterator(self._shadow.selectForLayer(layer_name))

    _shadow: ShadowRasteredTable
