import typing
from typing import Union, Optional, List, Tuple, Generator

from axipy._internal._shadow_instance_factory import _shadow_manager
from axipy._internal._utils import _NoInitTypeError

from .data_object import DataObject, Table, SelectionTable, QueryTable
from .providers import opener_instance
from .sqldialect import TypeSqlDialect

if typing.TYPE_CHECKING:
    from axipy.cpp_core_dp import ShadowDataCatalog
    from PySide2.QtCore import SignalInstance

__all__ = ["DataManager", "data_manager"]


class _DataManager:
    _inner_shadow: 'ShadowDataCatalog' = None

    def __init__(self) -> None:
        raise _NoInitTypeError

    @property
    def _shadow(self) -> 'ShadowDataCatalog':
        if self._inner_shadow is None:
            self._inner_shadow = _shadow_manager.gui.catalog()
        return self._inner_shadow

    @property
    def _all_tables(self: 'DataManager') -> List[Table]:
        return list(filter(self._is_table, self.all_objects))

    def _get_dialect(self: 'DataManager', dialect):
        d = self.sql_dialect
        if isinstance(dialect, TypeSqlDialect):
            d = 'sqlite' if dialect == TypeSqlDialect.sqlite else 'axioma'
        elif isinstance(dialect, str) and dialect in ['sqlite', 'axioma']:
            d = dialect
        return d

    def _query_internal(
            self: 'DataManager',
            query_text: str,
            is_hidden,
            dialect: Union[TypeSqlDialect, str] = None
    ) -> Optional[Table]:
        result = opener_instance.query(query_text, *self._all_tables, dialect=self._get_dialect(dialect))
        if result is not None:
            if is_hidden:
                result._shadow.setIsHidden(True)
            self.add(result)
        return result

    @staticmethod
    def _is_table(obj: DataObject) -> bool:
        return isinstance(obj, Table)


class DataManager(_DataManager):
    """
    Хранилище объектов данных.
    При открытии таблицы или растра эти объекты автоматически попадают в данный каталог.
    Для отслеживания изменений в каталоге используются события :attr:`added` и :attr:`removed`.

    Если же разработка ведется не в рамках приложения и ядро инициализировано явно посредством
    :func:`axipy.init_axioma`, то объект в каталог надо добавлять явно :meth:`DataManager.add`


    Note:
        Используйте готовый экземпляр этого класса
        :attr:`axipy.data_manager`.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_dp.py
        :caption: Пример использования.
        :pyobject: test_run_example_catalog
        :lines: 3-
        :dedent: 4
    """

    @property
    def updated(self) -> 'SignalInstance':
        """
        Сигнал об изменении каталога. В качестве параметра передается дополнительная информация в виде dict.

        .. csv-table:: Доступные параметры
            :header: Наименование, Значение, Описание

            operation, added, Таблица добавлена в каталог
            operation, removed, Таблица удалена из каталога
            operation, nameChanged, Изменено наименование таблицы
            operation, selectionChanged, Произведены изменения в выборке
            name, , "Наименование таблицы, если оно доступно"


        :rtype: Signal[]
        """
        # noinspection PyTypeChecker
        return self._shadow.updated

    @property
    def added(self) -> 'SignalInstance':
        """
        Сигнал о добавлении объекта. В качестве параметра передается наименование таблицы.

        :rtype: Signal[str]
        """
        # noinspection PyTypeChecker
        return self._shadow.added

    @property
    def removed(self) -> 'SignalInstance':
        """
        Сигнал об удалении объекта.

        :rtype: Signal[str]
        """
        # noinspection PyTypeChecker
        return self._shadow.removed

    @property
    def count(self) -> int:
        """Количество объектов данных."""
        return self.__len__()

    def __len__(self) -> int:
        return self._shadow.count()

    def __iter__(self) -> Generator[DataObject, None, None]:
        return (t for t in self.objects)

    def __contains__(self, name: str) -> bool:
        return self.find(name) is not None

    def __getitem__(self, key: str) -> DataObject:
        result = self.find(key)
        if result is None:
            raise KeyError
        return result

    def add(self, data_object: DataObject):
        """
        Добавляет объект данных в хранилище.

        Args:
            data_object: Объект данных для добавления.
        """
        if data_object is not None and self._shadow is not None:
            self._shadow.add(data_object._shadow)
        else:
            raise ValueError("None is not allowed.")

    def remove(self, data_object: DataObject):
        """
        Удаляет объект данных.

        Объект данных при этом закрывается.

        Args:
            data_object: Объект данных для удаления.
        """
        if isinstance(data_object, str):
            data_object = self.find(data_object)
        if data_object is None:
            return None
        data_object.close()

    def remove_all(self) -> None:
        """Удаляет все объекты данных."""
        for obj in self.objects:
            try:
                self.remove(obj)
            except RuntimeError as e:
                if isinstance(obj, (QueryTable, SelectionTable)):
                    pass
                else:
                    raise e

    def find(self, name: str) -> Optional[DataObject]:
        """
        Производит поиск объект данных по имени.

        Args:
            name: Имя объекта данных.

        Returns:
            Искомый объект данных или None.
        """
        for obj in self.objects:
            if obj.name != name:
                continue
            return obj
        return None

    def exists(self, obj: DataObject) -> bool:
        """
        Проверяет, присутствует ли объект в каталоге.
        Проверяет так-же и скрытые объекты, которые отсутствуют в общем списке.

        Args:
            obj: проверяемый объект данных.
        """
        if obj is None or obj._shadow is None:
            return False
        return self._shadow.exists(obj._shadow)

    @property
    def objects(self) -> List[DataObject]:
        """Список объектов."""
        return [DataObject._wrap(obj) for obj in self._shadow.dataObjects()]

    @property
    def all_objects(self) -> List[DataObject]:
        """Список всех объектов, включая скрытые."""
        return [DataObject._wrap(obj) for obj in self._shadow.dataObjects(True)]

    def _is_table(self, obj):
        return isinstance(obj, Table)

    @property
    def tables(self) -> List[Table]:
        """Список таблиц."""
        return list(filter(self._is_table, self.objects))

    @property
    def selection(self) -> Optional[SelectionTable]:
        """
        Таблица выборки, если она существует.

        See also:
            :attr:`axipy.selection_manager`
        """
        return DataObject._wrap(self._shadow.selectionTable())

    @property
    def sql_dialect(self) -> TypeSqlDialect:
        """
        Тип используемого диалекта по умолчанию для выполнения SQL-предложений. Если необходимо
        переопределить, то для конкретного sql предложения необходимо указывать диалект явно
        :meth:`query`

        Returns:
            Тип диалекта. Возможные значения `TypeSqlDialect.axioma` или `TypeSqlDialect.sqlite`.
        """
        return TypeSqlDialect(opener_instance.sql_dialect)

    def check_query(self, query_text: str, dialect: TypeSqlDialect = TypeSqlDialect.sqlite) -> Tuple[bool, str]:
        """
        Производит проверку SQL-запроса на корректность.

        Args:
            query_text: Текст запроса.
            dialect: Диалект, который используется при выполнении запроса.
                Значение по умолчанию установлено как значение свойства :attr:`sql_dialect`.

        Returns:
            Пара значений [Успешность проверки, Сообщение].

        .. literalinclude:: /../../tests/doc_examples/da/test_example_dp.py
            :caption: Пример использования, если работа ведется в рамках Аксиома .
            :pyobject: test_run_example_query
            :lines: 2, 4-6, 11-
            :dedent: 4

        """
        return opener_instance.check_query(query_text, *self._all_tables, dialect=self._get_dialect(dialect))

    def query(self, query_text: str, dialect: TypeSqlDialect = TypeSqlDialect.sqlite) -> Optional[Table]:
        """
        Выполняет SQL-запрос к перечисленным таблицам.

        Args:
            query_text: Текст запроса.
            dialect: Диалект, который используется при выполнении запроса.
                Значение по умолчанию установлено как значение свойства :attr:`sql_dialect`.

        Returns:
            Таблица, если результатом запроса является таблица.

        Raises:
            RuntimeError: При возникновении ошибки.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_dp.py
            :caption: Пример использования, если работа ведется в рамках Аксиома .
            :pyobject: test_run_example_query
            :lines: 2, 4-10
            :dedent: 4

        """
        return self._query_internal(query_text, False, dialect)

    def query_hidden(self, query_text: str, dialect: TypeSqlDialect = TypeSqlDialect.sqlite) -> Optional[Table]:
        """
        Выполняет SQL-запрос к таблицам. В отличие от :meth:`query` результирующий объект :class:`Table`
        добавляется в каталог как скрытый объект. Он не учитывается в общем списке и от него из этого каталога
        не приходят события.

        Args:
            query_text: Текст запроса.
            dialect: Диалект, который используется при выполнении запроса.
                Значение по умолчанию установлено как :attr:`sql_dialect`.

        Returns:
            Таблица, если результатом запроса является таблица.
        """
        return self._query_internal(query_text, True, dialect)


data_manager = DataManager.__new__(DataManager)
