import itertools
from dataclasses import dataclass
from pathlib import Path
from typing import List, Optional

from PySide2.QtCore import Qt
from PySide2.QtGui import QKeyEvent
from PySide2.QtUiTools import QUiLoader
from PySide2.QtWidgets import QMessageBox

from axipy import ObserverManager
from axipy.cs import CoordTransformer, Unit
from axipy.da import Feature, LineStyle
from axipy.da import GeometryType, Line, LineString, Polygon, Geometry, GeometryCollection
from axipy.da import Table
from axipy.gui import MapTool, MapView
from axipy.utl import Pnt, Rect
from ..helper import print_exc_

ui_file = Path(__file__).parents[1] / "ui" / "RightAngle.ui"

supported_types = [GeometryType.Line, GeometryType.LineString, GeometryType.Polygon, GeometryType.GeometryCollection]
supported_types_string = ['Линия', 'Полилиния', 'Полигон', 'Коллекция']


def show_warning_no_supported_geometry():
    title = 'Ошибка'
    message = 'Геометрия не найдена или не поддерживается.'
    mbox = QMessageBox(QMessageBox.Information, title, message, QMessageBox.Close)
    mbox.setDetailedText(
        f'Список поддерживаемых геометрий: "{", ".join(supported_types_string)}".')
    mbox.exec()


def is_allowed(geometry):
    return geometry is not None and geometry.type in supported_types or issubclass(
        type(geometry), GeometryCollection)


def to_points(geometry) -> List[Pnt]:
    geom_type = type(geometry)
    if geom_type is Line:
        return [geometry.begin, geometry.end]
    elif geom_type is LineString:
        return geometry.points
    return []


@dataclass
class CollisionResult:
    segment: LineString
    segment_index: int
    part_index: int = None
    geom_index: int = None


@dataclass
class GeometryCollisionResult:
    feature: Feature
    geometry: Geometry
    collision: CollisionResult

    def _add_point_line(self, geom, point):
        if isinstance(geom, Line):
            return LineString([geom.begin, point, geom.end])
        geom.points.insert(self.collision.segment_index, point)
        return geom

    def _add_point_single(self, geom, point):
        if self.collision.part_index is not None:
            if self.collision.part_index == 0:
                geom.points.insert(self.collision.segment_index + 1, point)
            else:
                geom.holes[self.collision.part_index - 1].insert(self.collision.segment_index + 1, point)
            return None

        return self._add_point_line(geom, point)

    def add_point(self, point) -> Geometry:
        if self.collision.geom_index is not None:
            if isinstance(self.geometry, GeometryCollection):
                geom = self.geometry[self.collision.geom_index]
                self.geometry[self.collision.geom_index] = self._add_point_single(geom, point)
            return self.geometry
        return self._add_point_single(self.geometry, point)


def find_segment_impl(points: List[Pnt], geometry: Geometry, box: Polygon) -> Optional[CollisionResult]:
    for i in range(len(points) - 1):
        segment = LineString([points[i], points[i + 1]], cs=geometry.coordsystem)
        if box.intersects(segment):
            return CollisionResult(segment, i)
    return None


def find_in_polygon(polygon: Polygon, bbox: Polygon) -> Optional[CollisionResult]:
    segment = find_segment_impl(polygon.points, polygon, bbox)
    if segment is not None:
        segment.part_index = 0
        return segment
    for index, hole in enumerate(polygon.holes):
        segment = find_segment_impl(hole, polygon, bbox)
        if segment is not None:
            segment.part_index = index + 1
            return segment
    return None


def find_in_single_geometry(geometry: Geometry, bbox: Polygon) -> Optional[CollisionResult]:
    if isinstance(geometry, Polygon):
        return find_in_polygon(geometry, bbox)
    return find_segment_impl(to_points(geometry), geometry, bbox)


def find_segment(geometry: Geometry, bbox: Polygon) -> Optional[CollisionResult]:
    if not isinstance(geometry, GeometryCollection):
        return find_in_single_geometry(geometry, bbox)
    for index, sub_geometry in enumerate(geometry):
        sub_geometry.coordsystem = geometry.coordsystem
        segment = find_in_single_geometry(sub_geometry, bbox)
        if segment is not None:
            segment.geom_index = index
            return segment
    return None


def project(segment, p):
    p0 = segment.points[0]
    p1 = segment.points[1]
    dx = p1.x - p0.x
    dy = p1.y - p0.y
    len2 = dx * dx + dy * dy
    r = ((p.x - p0.x) * dx + (p.y - p0.y) * dy) / len2
    return Pnt(p0.x + r * (p1.x - p0.x), p0.y + r * (p1.y - p0.y))


def find_in_table(table: Table, scene_box: Rect) -> Optional[GeometryCollisionResult]:
    layer_box = Polygon.from_rect(scene_box, table.coordsystem)  # type: Polygon
    features = table.items(bbox=scene_box)
    for feature in features:
        geometry = feature.geometry
        if not is_allowed(geometry):
            continue
        collision = find_segment(geometry, layer_box)
        if collision is None:
            continue
        return GeometryCollisionResult(feature, geometry, collision)
    return None


def find_in_map_view(map_view: MapView, bbox: Rect) -> Optional[GeometryCollisionResult]:
    for layer in itertools.chain([map_view.map.cosmetic], map_view.map.layers):
        layer_bbox = Rect(
            xmin=bbox.xmin,
            xmax=bbox.xmax,
            ymin=bbox.ymin,
            ymax=bbox.ymax)
        if not isinstance(layer.data_object, Table):
            continue
        if map_view.coordsystem != layer.coordsystem:
            try:
                layer_bbox = CoordTransformer(
                    map_view.coordsystem, layer.coordsystem).transform(layer_bbox)
            except (Exception,):
                print_exc_()
                continue
        result = find_in_table(layer.data_object, layer_bbox)
        if result is not None:
            return result
    return None


class RightAngleTool(MapTool):
    enable_on = ObserverManager.Editable

    def __init__(self):
        super().__init__()

        self._start_scene_point = None
        self._start_scene_box = None
        self._end_device_point = None

        self._reset()

    def _reset(self):
        self._start_scene_point = None
        self._start_scene_box = None
        self._end_device_point = None

    def paintEvent(self, event, painter):
        if not self.is_drawing() or self._end_device_point is None:
            return
        painter.drawPolyline([self.to_device(
            self._start_scene_point), self.snap_device(self._end_device_point)])

    def mousePressEvent(self, event):
        if event.button() != Qt.LeftButton:
            return self.PassEvent
        self._reset()
        self._start_scene_point = self.snap(self.to_scene(event.pos()))
        self._start_scene_box = self.get_select_rect(event.pos(), 30)
        return self.BlockEvent

    def mouseReleaseEvent(self, event):
        if event.button() != Qt.LeftButton:
            return self.PassEvent
        if not self.is_drawing():
            return self.PassEvent
        end_scene_point = self.snap(self.to_scene(event.pos()))
        view_cs = self.view.coordsystem
        collision = find_in_map_view(self.view, self._start_scene_box)
        if collision is None:
            self._reset()
            show_warning_no_supported_geometry()
            return self.PassEvent
        segment = collision.collision.segment
        if segment.coordsystem != view_cs:
            segment = segment.reproject(view_cs)
        projection_point = project(segment, self._start_scene_point)
        distance_m, dist_angle = Geometry.distance_by_points(projection_point, end_scene_point, view_cs)
        map_units = self.view.map.distanceUnit
        distance, segmenting = self.show_dialog(Unit.m.to_unit(map_units, distance_m), map_units.name)
        if distance is None:
            self._reset()
            return self.BlockEvent
        distance_m = map_units.to_unit(Unit.m, distance)
        _, angle = Geometry.distance_by_points(segment.points[0], segment.points[1], view_cs)
        clockwise: bool = ((dist_angle % 360) - (angle % 360)) % 360 < 180
        direction = angle + (90 if clockwise else -90)
        offset_point = Geometry.point_by_azimuth(projection_point, direction, distance_m, view_cs)
        if segmenting:
            collision.feature.geometry = collision.add_point(projection_point)
            table = self.view.editable_layer.data_object
            if isinstance(table, Table):
                table.update(collision.feature)
        self.insert(projection_point, offset_point)
        self._reset()
        return self.BlockEvent

    @staticmethod
    def show_dialog(value: float = 1, units_name: str = 'm'):
        dialog = QUiLoader().load(str(ui_file))
        dialog.setWindowFlags(dialog.windowFlags() & ~Qt.WindowContextHelpButtonHint)
        dialog.lblUnits.setText(units_name)
        dialog.leValue.setValue(value)
        dialog.leValue.selectAll()
        dialog.leValue.setFocus()
        if dialog.exec():
            return dialog.leValue.value(), dialog.cbSegmenting.isChecked()
        return None, None

    def mouseMoveEvent(self, event):
        if not self.is_drawing():
            return
        self._end_device_point = event.pos()
        self.redraw()

    def keyPressEvent(self, event: QKeyEvent) -> Optional[bool]:
        if event.key() == Qt.Key_Escape:
            self.reset()
            return
        return super().keyPressEvent(event)

    def insert(self, start, end):
        line = Line(start, end, self.view.coordsystem)
        feature = Feature(geometry=line, style=LineStyle())

        table = self.view.editable_layer.data_object
        if isinstance(table, Table):
            table.insert(feature)

    def is_drawing(self):
        return self._start_scene_point is not None

    def editable_table(self):
        return self.view.editable_layer.data_object
