from axipy.cpp_core_core import ShadowTabFile

from axipy._internal._shadow_instance_factory import _shadow_manager
from .data_object import DataObject

__all__ = ["TabFile"]


class TabFile:
    """Класс поддержки файла TAB формата MapInfo."""

    def __init__(self):
        self._shadow = ShadowTabFile(_shadow_manager.core)

    def generate_tab(self, data_object: DataObject, out_file: str, override: bool = True) -> bool:
        """Генерирует файл TAB для переданного открытого объекта, если такую возможность поддерживает провайдер данных.
        
        Args:
            data_object: открытый объект данных, для которого необходимо создать файл TAB.
            out_file: Имя файла c расширением tab. Как вариант, можно использовать результат :meth:`suggest_tab_name`.
            override: Перезаписывать файл. Если установлено False и файл существует, будет выброшено исключение FileExistsError

        Returns:
            Возвращает True, если успешно.

        Создание TAB файла для открытой таблицы или растра:

        .. literalinclude:: /../../tests/doc_examples/da/test_example_tabfile.py
            :pyobject: test_run_example_tabfile_gen
            :lines: 2-4, 7-
            :dedent: 4

        Создание TAB файла для открытого источника тайлового сервиса::

            prj_mercator = 'Earth Projection 10, 104, "m", 0 Bounds (-20037508.34, -20037508.34) (20037508.34, 20037508.34)'
            osm_raster = provider_manager.tms.open('http://maps.axioma-gis.ru/osm/{LEVEL}/{ROW}/{COL}.png', prj=prj_mercator)
            tab = TabFile()
            out_file_name = tab.suggest_tab_name(osm_raster)
            tab.generate_tab(osm_raster, out_file_name)
        """
        from PySide2.QtCore import QFileInfo
        fi = QFileInfo(out_file)
        if fi.exists() and not override:
            raise FileExistsError("File '{}' already exists".format(fi.absoluteFilePath()))
        return self._shadow.generateTab(data_object._shadow, fi)

    def suggest_tab_name(self, data_object: DataObject):
        """Сервисная функция. Предлагает наименование TAB файла для объекта данных.
        Результат можно использовать в методе :meth:`generate_tab` в качестве имени выходного файла."""
        return self._shadow.suggestTabName(data_object._shadow)
