from typing import Optional

from PySide2.QtCore import QVersionNumber
from axipy.cpp_core_core import ShadowInformation

from axipy._internal._decorator import _deprecated_by

__all__ = ["Version"]


class Version:
    """
    Информация о версии ГИС Аксиома.

    Пример использования::

        from axipy import Version

        print('Версия:', Version.string(), Version.compare(4,3))
    """

    @staticmethod
    def number() -> int:
        """Возвращает версию в виде одного числа, в котором каждый сегмент
        располагается в отдельном байте."""
        return ShadowInformation.currentVersion()

    @staticmethod
    @_deprecated_by("axipy.Version.qt_format")
    def qtFormat() -> QVersionNumber:
        """Возвращает версию в Qt формате."""
        return QVersionNumber(*Version.segments())

    @staticmethod
    def qt_format() -> QVersionNumber:
        """Возвращает версию в Qt формате."""
        return QVersionNumber(*Version.segments())

    @staticmethod
    def segments() -> tuple:
        """Возвращает кортеж чисел - сегменты версии: (`major`, `minor`, `patch`)."""
        v = Version.number()
        return (v & 0xFF0000) >> 16, (v & 0xFF00) >> 8, v & 0xFF

    @staticmethod
    def string() -> str:
        """Возвращает версию в виде строки."""
        return ShadowInformation.currentVersionString()

    @staticmethod
    def compare(major: int, minor: Optional[int] = 0, patch: Optional[int] = 0) -> int:
        """Сравнивает переданное значение с текущей версией Аксиомы.

        Args:
            major: Основная версия
            minor: Минорная версия
            patch: Исправления

        Возвращает -1, если переданная меньше, 1 если больше и 0 если равны
        """
        return ShadowInformation.compareVersion(major, minor, patch)
