from abc import abstractmethod
from typing import Callable, Any

from PySide2.QtWidgets import QDialog
from axipy.cpp_gui import ShadowProgressElementFactory

from axipy._internal._decorator import InitOnce
from .task import AxipyAnyCallableTask, AxipyTask
from .task_utils import ProgressSpecification, _to_cpp_progress_spec

__all__ = ["TaskManager", "task_manager"]


class _ProgressElementFactoryInterace:

    @abstractmethod
    def make_dialog_for_task(self, task: AxipyTask, spec: ProgressSpecification):
        pass


class _ProgressElementFactory(_ProgressElementFactoryInterace):

    def __init__(self):
        self.__default_factory = ShadowProgressElementFactory()
        self.__current_facotry = self.__default_factory

    @property
    def factory(self) -> _ProgressElementFactoryInterace:
        return self.__current_facotry

    @factory.setter
    def factory(self, factory: _ProgressElementFactoryInterace):
        if not issubclass(type(factory), _ProgressElementFactoryInterace):
            raise TypeError(f"Не верный тип фабрики элементов прогресса. Передан \
                {type(factory)}, вместо наследника _ProgressElementFactoryInterace")
        self.__current_facotry = factory

    def make_dialog_for_task(self, task: AxipyTask, spec: ProgressSpecification):
        return self.factory.make_dialog_for_task(task, spec)

    def restore(self):
        self.__current_facotry = self.__default_factory


_progress_element_factory_instance = _ProgressElementFactory()


class TaskManager:
    """
    Сервис, содержащий вспомогательные функции для запуска и конфигурирования
    пользовательских задач.
    """

    def __init__(
            self,
            progress_element_factory=_progress_element_factory_instance) -> None:
        self.__progress_element_factory = progress_element_factory

    def start_task(self, task):
        """
        Добавляет переданную задачу в очередь на выполнение.

        Args:
            task: Пользовательская задача.

        .. literalinclude:: /../../tests/doc_examples/concurrent/test_example_concurrent.py
            :caption: Пример использования.
            :pyobject: test_task_service_run_and_get
            :lines: 2-
            :dedent: 4

        """
        self.__service().start_task(task)

    def run_and_get(
            self,
            spec: ProgressSpecification,
            func: Callable,
            *args,
            **kwargs) -> Any:
        """
        Превращает пользовательскую функцию в задачу и запускает её с отображением
        прогресса выполнения. 

        Args:
            spec: Описание задачи.
            func: Пользовательская функция, которая будет выполняться.
                В нее передается список `args` и словарь `kwargs`.
            args: Список аргументов, передаваемый в функцию при запуске.
            kwargs: Словарь, передаваемый в функцию при запуске.
        
        Returns:
            Результат выполнения пользовательской функции или None при ошибке.

        .. literalinclude:: /../../tests/doc_examples/concurrent/test_example_concurrent.py
            :caption: Пример использования.
            :pyobject: test_task_service_run_and_get_full
            :lines: 2-
            :dedent: 4

        """
        task = AxipyAnyCallableTask(func, *args, **kwargs)
        task.with_handler(spec.with_handler)
        ph = task.progress_handler()
        dialog = self.generate_dialog_for_task(task, spec)
        self.start_task(task)
        if not ph.is_finished():
            dialog.exec_()
        return ph.result

    def run_in_gui(self, func: Callable,
                   *args,
                   **kwargs) -> Any:
        """
        Выполняет переданную задачу в потоке интерфейса. 
        
        Это может быть удобно когда в процессе выполнения длительной фоновой задачи 
        нужно спросить о чем нибудь пользователя отобразив диалог. 
        Так же создавать/взаимодействовать с некоторыми объектами можно только из 
        потока интерфейса.
        """
        task = AxipyAnyCallableTask(func, *args, **kwargs)
        task.with_handler(False)
        ph = task.progress_handler()
        self.__service().run_in_gui(task)
        return ph.result

    def generate_dialog_for_task(
            self, task: AxipyTask, spec: ProgressSpecification) -> QDialog:
        """
        Возвращает диалог, следящий за выполнением задачи.

        Args:
            task: Пользовательская задача.
            spec: Описание задачи.

        Returns:
            Диалог, который будет отображать прогресс выполнения задачи.

        .. literalinclude:: /../../tests/doc_examples/concurrent/test_example_concurrent.py
            :caption: Пример использования.
            :pyobject: test_task_service_generate_dialog
            :lines: 2-
            :dedent: 4

        """
        return self.__progress_element_factory.make_dialog_for_task(
            task, _to_cpp_progress_spec(spec))

    @InitOnce
    def __service(self):
        from axipy.cpp_core_core import ShadowTaskManager
        return ShadowTaskManager()


task_manager = TaskManager()
