from PySide2.QtCore import Signal, QObject, QWriteLocker, QReadLocker, QReadWriteLock
from axipy.cpp_core_core import ShadowProgressHandler


__all__ = ["AxipyProgressHandler"]


class AxipyProgressHandler(QObject):
    """
    Класс, объекты которого выполняют функцию канала передачи данных между выполняемой
    задачей и элементом отображающим прогресс.

    .. attribute:: error

    Сигнал об ошибке, содержащий информацию о исключении.

    :type: Signal[:class:`tuple`]
    """

    error = Signal(tuple)

    def __init__(self):
        super().__init__()
        self.__shadow_wrapper = ShadowProgressHandler()
        self.__lock = QReadWriteLock()
        self.__result = None

    def _wrapper(self):
        return self.__shadow_wrapper

    def _shadow(self):
        return self.__shadow_wrapper.shadow()

    def is_finished(self) -> bool:
        """
        Проверяем не завершилась ли задача.
        """
        return self._shadow().is_finished()

    def is_canceled(self) -> bool:
        """
        Проверяем не была ли задача отменена.
        """
        return self._shadow().is_canceled()

    def is_running(self) -> bool:
        """
        Возвращает True если задача сейчас выполняется.
        """
        return self._shadow().is_running()

    def set_description(self, description: str):
        """
        Устанавливаем описание для задачи. Эта информация может
        быть использована элементами отображающими прогресс выполнения.

        Args:
            description: Новое описание задачи.
        """
        self._shadow().set_description(description)

    def progress(self) -> float:
        """
        Возвращает текущий прогресс выполнения.
        """
        return self._shadow().progress()

    def set_progress(self, value: float):
        """
        Устанавливает текущий прогресс задачи.

        Args:
            value: Новое значение прогресса.
        """
        self._shadow().set_progress(value)

    def add_progress(self, value: float):
        """
        Добавляет к текущему прогрессу переданное значение.

        Args:
            value: Значение, которое будет добавлено к прогрессу.
        """
        self._shadow().set_progress(self.progress() + value)

    def set_window_title(self, title: str):
        """
        Устанавливает заголовок диалога с прогрессом.

        Args:
            title: Новый заголовок.
        """
        self._shadow().set_window_title(title)

    def set_max_progress(self, value: float):
        """
        Устанавливает максимальное значение прогресса.  Минимальное значение берется за ноль.

        Args:
            value: Верхний порог для прогресса операции.
        """
        self._shadow().set_max_progress(value)

    def cancel(self):
        """
        Отменяет задачу, связанную с обработчиком.

        Note:
            Эта функция посылает только запрос на отмену операции. Реальное
            прерывание операции возможно только если есть поддержка в
            пользовательском коде. Например, с помощью функций :attr:`is_canceled`
            или :attr:`raise_if_canceled`

        """
        self._shadow().cancel()
        self.canceled.emit()

    @property
    def canceled(self) -> Signal:
        """Уведомляет, что задача была отменена.
        Сигнал испускается когда была вызвана функция :attr:`cancel`.

        :rtype: Signal[]

        .. warning::
            Получение этого сигнала не означает, что задача была завершена. Если
            в пользовательском коде не обрабатывается отмена, то задача будет продолжать
            выполняться до логического завершения. 
         """
        return self._shadow().canceled

    @property
    def started(self) -> Signal:
        """
        Уведомляет о старте выполнения задачи.

        :rtype: Signal[]
        """
        return self._shadow().started

    @property
    def finished(self) -> Signal:
        """
        Уведомляет о завершении выполняемой задачи.

        :rtype: Signal[]
        """
        return self._shadow().finished

    @property
    def description_changed(self) -> Signal:
        """
        Уведомляет о изменении описания задачи.

        :rtype: Signal[str], где str - текущее описание задачи.
        """
        return self._shadow().description_changed

    @property
    def progress_changed(self) -> Signal:
        """
        Уведомляет о изменении значения прогресса.

        :rtype: Signal[float]
        """
        return self._shadow().progress_changed

    @property
    def window_title_changed(self) -> Signal:
        """
        Уведомляет о изменении заголовка диалога отображающего прогресс.

        :rtype: Signal[str]
        """
        return self._shadow().window_title_changed

    def raise_if_canceled(self):
        """ 
        Если задача была отменена выбрасывает исключение. Удобно при работе 
        с большим количеством вложенных циклов или вызовов функции.
        """
        if self.is_canceled():
            raise RuntimeError(self.tr("Задача была отменена"))

    @property
    def result(self):
        """
        Содержит результат выполнения задачи, связанной с обработчиком. 
        Возвращается None если произошла ошибка, либо задача не предполагает 
        возвращение результата.

        Returns:
            Результат выполнения задачи или None.
        """
        locker = QReadLocker(self.__lock)
        return self.__result

    @result.setter
    def result(self, result):
        locker = QWriteLocker(self.__lock)
        self.__result = result

    def prepare_to_write_changes(self, description: str = ""):
        """
        Делает индикатор выполнения бесконечным, убирает кнопку отмены и добавляет переданное 
        описание. По умолчанию описание содержит запись о том, что производится запись изменений.

        Args:
            description: Сообщение которое будет отображаться.
        """
        if len(description) == 0:
            self._wrapper().prepare_to_write_changes()
        else:
            self._shadow().disable_cancelable()
            self.set_max_progress(0)
            self.set_description(description)
