import runpy
import sys
from pathlib import Path
from typing import Optional, Union

from PySide2.QtWidgets import QFileDialog
from axipy.app import mainwindow
from axipy.axioma_plugin import tr
from axipy.settings import CurrentSettings

__all__ = [
    "open_file_dialog",
    "execfile",
]


def open_file_dialog(filter_arg: str = None) -> Optional[Path]:
    """
    Открывает диалог выбора файла. Возвращает путь к выбранному файлу.
    Если нет главного окна Аксиомы, спрашивает путь к файлу в консоли.

    Args:
        filter_arg: Типы файлов. Например: ``'MapInfo Tab (*.tab);;Таблицы Excel (*.xls *.xlsx)'``.

    """
    if mainwindow.is_valid:
        params = {
            "parent": mainwindow.qt_object(),
            "dir": str(CurrentSettings.LastOpenPath)
        }
        if filter_arg is not None:
            params["filter"] = filter_arg
        if not CurrentSettings.UseNativeFileDialog:
            params["options"] = QFileDialog.DontUseNativeDialog
        result = QFileDialog.getOpenFileName(**params)[0]
    else:
        result = input(tr("Введите путь к файлу: "))

    if result:
        result_str = result
        result = Path(result)
        if not (result.is_file() or result.exists()):
            raise FileNotFoundError(f"File not found: \n{result_str}")
        else:
            return result
    else:
        return None


def execfile(path: Union[str, Path]):
    """
    Выполняет скрипт на языке python из файла.

    Args:
        path: Путь к исполняемому файлу.

    """

    def run_path(path_arg: str):
        sys.argv = [path_arg]
        runpy.run_path(path_arg)

    if isinstance(path, str):
        run_path(path)
    elif isinstance(path, Path):
        run_path(str(path))
    elif isinstance(path, list) and len(path) > 0:
        sys.argv = path
        runpy.run_path(path[0])
    else:
        raise TypeError('Parameter is not supported.')
