import logging
from functools import reduce


from .optimizer import Optimizer
from .axobjects import Path, AxTick, AxAngl, AxPolygon
from .mipen import Element, OType
from .trtp import DefaultColor, Unit, UnitValue, MapInfoSubPattern, BeginMarker, ContinuousMarker, EndMarker, \
    MainLine, JoinStyle

"""
    Конвертация из MapInfo pen в Axioma trtp
    точка входа Utils.convert() 
"""


class Utils:
    @staticmethod
    def convert(pen):
        content = ""
        opts = []
        logging.debug("pen {}".format(pen.id))
        for l in pen.layers:
            opts += Utils.optimize_elements(l.elements)
        for els in opts:
            content += AxLayer(els).to_subpattern().__str__()
        return content

    @staticmethod
    def without_styles(els):
        return OType.without_styles(els)

    @staticmethod
    def set_style(elems, base):
        styles = list(filter(lambda e: e.type in (OType.style_types()), elems))
        if len(styles) > 0:
            for s in styles:
                if s.type == OType.PEN_COLOR:
                    base.defaultColor = DefaultColor(s.color)
                    base.useStyleColor = False
                # if s.type == ElementType.BRUSH_CLR_OWN:
                #     logging.warning("BRUSH_CLR_OWN not implement")
                if s.type == OType.BRUSH_CLR_TRANSPARENT:
                    base.fill = False
                if s.type == OType.PEN_COLOR_USER:
                    base.useStyleColor = True
                if s.type == OType.PEN_WIDTH_USER_OWN:
                    base.baseWidth = s.int_param
                    base.useStyleWidth = True
                    if type(base) == MainLine:
                        base.dashPattern = list(map(lambda x: x / s.int_param if s.int_param > 0 else x,
                                                    base.dashPattern))
                if s.type == OType.PEN_WIDTH_USER:
                    base.baseWidth = 1
                    base.useStyleWidth = True
                if s.type == OType.PEN_WIDTH_OWN:
                    # base.useStyleWidth = False todo тут разное поведение с аксиомой
                    base.useStyleWidth = True
                    base.defaultWidth = UnitValue(s.int_param, Unit.px)
                    base.baseWidth = s.int_param
                    if type(base) == MainLine:
                        base.dashPattern = list(map(lambda x: x / s.int_param if s.int_param > 0 else x,
                                                    base.dashPattern))

    @staticmethod
    def optimize_4_dash_pattern(els):
        """ todo move to Optimizer
            Оптимизация для создания dash pattern
            1) убрать все draw tool элементы
            2) если линия из множества отрезков то сокращаем до одной линии
            3) если рядом два STUB объединяем
        """
        els = Utils.without_styles(els)

        def merge_stub(l):
            result = []
            for i, e in enumerate(l):
                if e.type == OType.STAB and len(l) > i + 1 and l[i + 1].type == OType.STAB:
                    l[i + 1] = Element.stub(e.int_param + l[i + 1].int_param)
                else:
                    result.append(e)
            return result

        els = merge_stub(els)

        sum_func = lambda a, b: bytes([int.from_bytes(a, 'little') + int.from_bytes(b, 'little')])

        def a(ret, inp):
            if len(ret) == 0:
                return [inp]
            if ret[-1].type == OType.DRAW and inp.type == OType.DRAW:
                ret[-1] = Element(OType.DRAW, sum_func(ret[-1].param, inp.param))
                return ret
            else:
                return ret + [inp]

        return reduce(lambda x, y: a(x, y), els, [])

    @staticmethod
    def layer_size(objects):
        result = 0
        for i, obj in enumerate(objects):
            if obj.type in [OType.TICK_MARK, OType.ANGLE_TICK_RIGHT, OType.ANGLE_TICK_LEFT]:
                if len(objects) - 1 == i or \
                        (len(objects) - i == 2
                         and objects[i + 1].type == OType.STAB
                         and objects[i + 1].int_param > 1):
                    result += 1
            elif obj.type in [OType.POLYGON_BEGIN, OType.POLYLINE_BEGIN]:
                result += 1
            elif obj.type in [OType.STAB, OType.DRAW]:
                result += obj.int_param
            elif obj.type == OType.POLYLINE_POINT:
                y2, x2 = obj.int_params
                result += x2
            elif obj.type == OType.POLYGON_TICK:
                param = obj.int_params
                result += param[0]
        return result

    @staticmethod
    def optimize_elements(objs):
        """
            Config optimizer
            Order is important
        """
        return Optimizer(objs) \
            .first_loop_remover() \
            .draw_stub_feature() \
            .polygon_merge() \
            .polyline_to_line() \
            .polygon_to_line() \
            .style_optimize() \
            .style_separator() \
            .tick_mark_as_point_case2() \
            .overlapping() \
            .result()

    @staticmethod
    def dash_pattern(objs):
        dash_pattern = []
        elms = Utils.optimize_4_dash_pattern(objs)
        TYPES_TO_HANDLE = [OType.DRAW, OType.DRAW_2_END, OType.STAB]
        elms = [elm for elm in elms if elm.type in TYPES_TO_HANDLE]
        for i in range(len(elms)):
            elm = elms[i]
            if elm.type == OType.DRAW:
                if len(elms) == 1:  # if only DRAW means line
                    return [elm.int_param, 0]
                if len(dash_pattern) % 2 == 0:
                    dash_pattern.append(elm.int_param)
                else:
                    i = len(dash_pattern) - 1
                    dash_pattern.insert(i, dash_pattern[i] + elm.int_param)
            elif elm.type == OType.DRAW_2_END:
                return [1, 0]
            elif elm.type == OType.STAB and i != 0:
                if not len(dash_pattern) % 2 == 0:
                    dash_pattern.append(elm.int_param)
                else:
                    i = len(dash_pattern) - 1
                    dash_pattern.insert(i, dash_pattern[i] + elm.int_param)
            elif elm.type == OType.STAB and i == 0:
                dash_pattern.append(0)
                dash_pattern.append(elm.int_param)
        return dash_pattern if len(dash_pattern) > 1 else dash_pattern + [0]


class PartLayer:
    def __init__(self, elems):
        self.elements = elems
        self.last_size = 0
        self.size = 0
        self.draw_size = 0
        self._paths = None

    @property
    def paths(self):
        if self._paths is None:
            self._paths = Path() if len(self.elements) == 0 else self.make_path()
        return self._paths

    @property
    def path(self):
        return "" if self.paths.is_empty() else self.paths.__str__()

    def is_empty(self):
        return self.elements is None or len(self.elements) == 0

    def make_path(self):
        path = Path()
        pos_x = 0
        elements = Utils.without_styles(self.elements)
        for i, el in enumerate(elements):
            if el.type == OType.STAB:
                pos_x += el.int_param

            elif el.type == OType.TICK_MARK:
                bottom, top = el.int_params
                path.add(AxTick(top, bottom, pos_x).tick_path())

            elif el.type == OType.DRAW:
                path.m(pos_x, 0)
                pos_x += el.int_param
                path.l(pos_x, 0)
                path.z()

            elif el.type == OType.ANGLE_TICK_LEFT:  # <
                top, bottom = el.int_params
                path.add(AxAngl(top, bottom, pos_x).angl_left_path())

            elif el.type == OType.ANGLE_TICK_RIGHT:  # >
                top, bottom = el.int_params
                path.add(AxAngl(top, bottom, pos_x).angl_right_path())

            elif el.type == OType.POLYLINE_BEGIN:
                ind = i + 1
                y1, x1 = el.int_params
                path.m(pos_x, y1 * -1)
                while len(elements) > ind and elements[ind].type == OType.POLYLINE_POINT:
                    y2, x2 = elements[ind].int_params
                    y2 = y2 * -1
                    pos_x += x2
                    path.l(pos_x, y2)
                    path.z()
                    path.m(pos_x, y2)
                    ind += 1

            elif el.type == OType.POLYGON_BEGIN:
                ind = i + 1
                param = el.int_params
                pol_path = Path()
                # param[0] fix возможно это  позиция в слое
                y1 = param[1] * -1
                x2 = 0
                l1 = param[2]  # высота
                up_line = []
                # (pos_x + x2 + 0.5, y1 - 0.5) конец
                pol_path.m(x2, y1)  # начало левая верхня точка
                pol_path.l(x2, y1 + l1)
                pol_path.l(x2, y1 + l1)
                c_x = x2
                while len(elements) > ind and elements[ind].type == OType.POLYGON_TICK:
                    param = elements[ind].int_params
                    x2 = param[0]
                    y1 = param[1] * -1
                    l1 = param[2]
                    pol_path.l(c_x + x2, y1 + l1)
                    pol_path.l(c_x + x2, y1 + l1)
                    up_line.append((c_x + x2, y1))
                    up_line.append((c_x + x2, y1))
                    c_x += x2
                    ind += 1
                for l in reversed(up_line):
                    pol_path.l(l[0], l[1])
                pol_path.z()
                pol = AxPolygon(pol_path)
                path.add(pol.path.move_x(pos_x + 0.5))
                pos_x += c_x

        self.last_size = (Utils.layer_size(elements) - path.delta_x())
        self.size = pos_x if pos_x == 0 else pos_x
        self.draw_size = path.max_x()
        # https://intellij-support.jetbrains.com/hc/en-us/community/posts/115000164750-Possible-bug-SystemError-unknown-opcode
        return path


class AxLayer:
    def __init__(self, elements):
        self.elements = elements
        self.body = PartLayer([])
        self.begin = PartLayer([])
        self.end = PartLayer([])
        self.separate_layers()

    @property
    def shift_y(self):
        """ this is hack todo move to metainfo """
        shift = set(map(lambda x: x.shift_y,
                        filter(lambda x: x.type == OType.DRAW and "shift_y" in dir(x), self.elements)))
        return shift.pop() if len(shift) == 1 else 0

    def round_cap(self):
        """ for hack """
        result = len(list(filter(lambda x: "round_cap" in dir(x), self.elements))) > 0
        if result:
            logging.debug("find round cup !")
        return result

    def begin_marker(self):
        if not self.begin.is_empty():
            begin_marker = BeginMarker()
            begin_marker.path = self.begin.path
            Utils.set_style(self.begin.elements, begin_marker)
            return begin_marker
        return None

    def is_dash_pattern(self):
        exclude = OType.style_types()
        exclude.append(OType.STAB)
        es = [x for x in self.elements if x.type not in exclude]
        return all([e.type == OType.DRAW or
                    e.type == OType.DRAW_2_END or
                    e.type == OType.DRAW_2_END_ALT
                    for e in es])

    def data_2_main_line(self, msp):
        is_pattern = self.is_dash_pattern()
        msp.mainLine.enabled = self.main_line_detect() or is_pattern
        if is_pattern:
            msp.mainLine.dashPattern = self.to_dash_pattern()
            msp.mainLine.offset = self.shift_y
        if msp.mainLine.enabled:
            Utils.set_style(self.elements, msp.mainLine)

    def main_line_detect(self):
        return any([e.type == OType.DRAW_2_END or e.type == OType.DRAW_2_END_ALT for e in self.elements])

    def to_dash_pattern(self):
        if self.is_dash_pattern():
            return Utils.dash_pattern(self.elements)
        return None

    def separate_layers(self):
        draw_2_end = [x for x in range(len(self.elements))
                      if self.elements[x].type == OType.DRAW_2_END_ALT or
                      self.elements[x].type == OType.DRAW_2_END]
        skip_2_end = [x for x in range(len(self.elements))
                      if self.elements[x].type == OType.SKIP_2_END]
        if len(skip_2_end) > 0:
            self.begin = PartLayer(self.elements[0:skip_2_end[0]])
            self.end = PartLayer(self.elements[skip_2_end[0]:])
        elif len(draw_2_end) > 0:
            self.begin = PartLayer(self.elements[0:draw_2_end[0]])
            self.body = PartLayer([self.elements[draw_2_end[0]]])
            self.end = PartLayer(self.elements[draw_2_end[0]:])
        else:
            begin_sep = [x for x in range(len(self.elements)) if self.elements[x].type == OType.LOOP]
            self.begin = PartLayer(self.elements[0:begin_sep[0]] if len(begin_sep) > 0 else [])
            body_start = begin_sep[0] + 1 if len(begin_sep) > 0 else 0
            self.body = PartLayer(self.elements[body_start:] if len(begin_sep) > 0 else self.elements)

    def continuous_marker(self):
        body = self.body
        if not body.is_empty():
            body_marker = ContinuousMarker()
            if not self.begin.is_empty():
                body_marker.dashOffset = self.begin.size * -1
            if self.round_cap():  # todo move metadata and rename
                body_marker.joinStyle = JoinStyle.RoundJoin
            body_marker.dashPattern = [self.body.last_size]
            body_marker.path = body.path
            Utils.set_style(self.begin.elements, body_marker)  # todo reset style ?
            Utils.set_style(self.body.elements, body_marker)
            return body_marker
        return None

    def end_marker(self):
        if not self.end.is_empty():
            end_marker = EndMarker()
            end_marker.path = self.end.path
            Utils.set_style(self.begin.elements, end_marker)
            Utils.set_style(self.body.elements, end_marker)
            Utils.set_style(self.end.elements, end_marker)
            return end_marker
        return None

    def to_subpattern(self):
        if len(self.elements) == 0:
            return MapInfoSubPattern.empty_pattern()
        msp = MapInfoSubPattern()
        if self.begin.path.strip():
            msp.beginMarker = self.begin_marker()
        if not self.is_dash_pattern() and self.body.path.strip():
            msp.continuousMarker = self.continuous_marker()
        if self.end.path.strip():
            msp.endMarker = self.end_marker()
        self.data_2_main_line(msp)
        return msp

    def offset(self):
        if self.main_line_detect():
            p = PartLayer(self.elements).paths
            return p.y_offset()
        else:
            return 0
