from typing import Iterator, Dict, Union
from axipy.cpp_core_dp import Converter
from ..FeatureWrapper import Feature
from ..attribute_schema import Schema
from ..DataObjectWrapper import Table, DataObject
from .opener import opener_instance as _opener


class Source(dict):
    """Источник данных.

    Используется для открытия данных или для указания источника при конвертации.

    Пример открытия::

        table = source.open()

    Пример конвертации::

        destination.export_from(source)

    Note:
        Не все провайдеры поддерживают открытие и конвертацию. См. описание
        конкретного провайдера данных.
    """

    def __init__(self, *args):
        if type(self) is Source:
            raise NotImplementedError
        _merged = dict()
        for d in args:
            _merged.update(d)
        super().__init__(_merged)

    @classmethod
    def _table_file(cls, filepath: str) -> Dict:
        return {'src': filepath}

    @classmethod
    def _table_db(cls, db_name: str, host: str, user: str, password: str,
                  port: str) -> Dict:
        return {
            'src': host,
            'db': db_name,
            'user': user,
            'password': password,
            'port': port
        }

    @classmethod
    def _provider(cls, provider_id: str) -> Dict:
        return {'provider': provider_id}

    @classmethod
    def _alias(cls, alias: str) -> Dict:
        if alias:
            return {'alias': alias}
        return {}

    @classmethod
    def _prj(cls, prj: str) -> Dict:
        if prj:
            return {'prj': prj}
        return {}


    def open(self) -> DataObject:
        """Открывает объект данных."""
        return _opener.open(self)


class Destination(dict):
    """Назначение объекта данных.

    Используется для создания данных или для указания назначения при конвертации.

    Пример создания::

        table = destination.create_open()

    Пример конвертации::

        destination.export_from(source)

    Note:
        Не все провайдеры поддерживают создание и конвертацию. См. описание
        конкретного провайдера данных.
    """

    def __init__(self, schema: Schema, *args):
        if type(self) is Destination:
            raise NotImplementedError
        _merged = {'schema': schema.to_dict()}
        for d in args:
            _merged.update(d)
        super().__init__(_merged)

    def create_open(self) -> DataObject:
        """Создает и открывает объект данных."""
        return _opener.create(self)

    def export(self, features: Iterator[Feature]):
        """Создает объект данных и экспортирует в него записи.

        Args:
            features: Записи.
        """
        Converter.saveAs((f.shadow for f in features), self)

    def export_from(self, source: Source, copy_schema: bool = False):
        """Создает объект данных и экспортирует в него записи из источника данных.

        Args:
            source: Источник данных.
            copy_schema: Копировать схему источника без изменений.
        """
        dest: dict = self.copy()
        if copy_schema:
            del dest['schema']
        Converter.convert(source, dest)

    def export_from_table(self, table: Table, copy_schema: bool = False):
        """Создает объект данных и экспортирует в него записи из таблицы.

        Args:
            table: Таблица.
            copy_schema: Копировать схему источника без изменений.
        """
        dest: dict = self.copy()
        if copy_schema:
            dest.update({'schema': table.schema.to_dict()})
        Converter.saveAs((f.shadow for f in table), dest)
