"""
Точка входа в модуль.
"""
import importlib
import traceback
from pathlib import Path

from PySide2.QtWidgets import QMessageBox
from axipy import AxiomaPlugin, Position, tr
from osgeo import gdal

from . import wizardconverter

is_master = str(Path(__file__).parents[0].name).endswith("master")


class Plugin(AxiomaPlugin):

    def load(self) -> None:

        self.__action = self.create_action(
            title=tr("Конвертер"),
            on_click=self.show_plugin_window,
            icon=self.local_file("icon", "Translate-01.png"),
            tooltip=tr("Конвертер векторных форматов"),
            doc_file="index.html"
        )
        position = Position(tr("Основные"), tr("Команды"))
        position.add(self.__action)

        self.__plugin_window = None

    def unload(self) -> None:
        self.__action.remove()

        self.__plugin_window = None

    def show_plugin_window(self) -> None:
        """
        Функция показа окна плагина.
        """
        if is_master:
            self.__plugin_window = None
            importlib.reload(wizardconverter)

        if self.__plugin_window is None:
            success = self.init_plugin_window()
            if not success:
                return None

        # Запомнить состояние режима исключений gdal при показе окна
        self.__plugin_window.use_exceptions = gdal.GetUseExceptions()
        self.__plugin_window.show()

    def init_plugin_window(self) -> bool:
        """
        Создание окна плагина
        """
        try:
            self.__plugin_window = wizardconverter.WizardConverter(self.window(), self.language)
        except Exception:
            traceback.print_exc()

            m_box = QMessageBox(
                QMessageBox.Warning,  # icon
                tr("Ошибка"),  # title
                tr("Не удалось создать окно Помощника."),  # text
                buttons=QMessageBox.Ok,
                parent=self.window(),
            )
            m_box.exec_()

            self.__plugin_window = None
            return False
        else:
            return True
