from PySide2.QtWidgets import QDialog, QPushButton, QWidget
from PySide2.QtGui import QResizeEvent,  QPaintEvent, QPainter
from axipy.cpp_gui import ShadowChooseCoordSystemWidget, ShadowStyleButton, PasswordWidget
from axipy.cs import CoordSystem
from axipy.da import Style
from PySide2.QtCore import QObject, Signal


class ChooseCoordSystemDialog(QDialog):
    """Диалог выбора координатной системы.

        Args:
            coordsystem: Система координат по умолчанию. Если не указана, то задается как значение по умолчанию :meth:`axipy.cs.CoordSystem.current`

        Пример::

            csMercator = CoordSystem.from_prj('10, 104, 7, 0')
            dialog = ChooseCoordSystemDialog(csMercator)
            if dialog.exec() == QDialog.Accepted:
                result_cs = dialog.chosenCoordSystem()
                print("Выбрано:", result_cs.description)    
    """

    def __init__(self, coordsystem: CoordSystem = None):
        from axipy.cpp_gui import ShadowGui
        super().__init__(ShadowGui.global_parent_static())
        if self.layout() is not None:
            self.layout().setContentsMargins(0, 0, 0, 0)
        cs = coordsystem if coordsystem is not None else CoordSystem.current()
        self.shadow = ShadowChooseCoordSystemWidget(cs.shadow, self)

    def chosenCoordSystem(self) -> CoordSystem:
        """Возвращает выбранную в диалоге систему координат."""
        return CoordSystem._wrap(self.shadow.chosenCoordSystem())


class StyledButton(QPushButton):
    """Кнопка, отображающая стиль и позволяющая его менять

    Note:
        Сигнал `styleChanged` испускается при смене стиля.

    Args:
        style: Стиль по умолчанию.

    Пример добавления кнопки на диалог::

        style = Style.from_mapinfo("Pen (1, 2, 8421504)  Brush (2, 255, 0)")

        class Dialog(QDialog):
            def __init__(self, parent = None):
                QDialog.__init__(self, parent)
                self.pb = StyledButton( style, self)
                self.pb.styleChanged.connect(self.styleResult)
                self.pb.setGeometry(100, 100, 100, 50)

            def styleResult(self):
                print('Стиль изменен', self.pb.style())

        dialog = Dialog()
        dialog.exec()
    """
    styleChanged = Signal()

    def __init__(self, style: Style, parent: QWidget = None):
        from axipy.cpp_gui import ShadowGui
        super().__init__(parent if parent is not None else ShadowGui.global_parent_static())
        self.shadow = ShadowStyleButton(self, style.shadow)
        self.shadow.styleChanged.connect(self.styleChanged)

    def style(self) -> Style:
        """Результирующий стиль."""
        return Style._wrap(self.shadow.get_style())

    def paintEvent(self, event: QPaintEvent) -> None:
        super().paintEvent(event)
        painter = QPainter(self)
        painter.drawPixmap(2, 2, self.shadow.pixmap())

    def resizeEvent(self, event: QResizeEvent) -> None:
        super().resizeEvent(event)
        self.shadow.reinitPixmap()



class PasswordDialog(QDialog):
    """Диалог задания или корректировки данных аутентификации пользователя.

    Пример::

        dialog = PasswordDialog()
        dialog.user_name = 'user'
        dialog.password = 'pass'
        if dialog.exec() == QDialog.Accepted:
            print(dialog.user_name, dialog.password)
    """
    def __init__(self):
        from axipy.cpp_gui import ShadowGui
        super().__init__(ShadowGui.global_parent_static())
        if self.layout() is not None:
            self.layout().setContentsMargins(0, 0, 0, 0)
        self.shadow = PasswordWidget(self)

    @property
    def user_name(self) -> str:
        """Имя пользователя."""
        return self.shadow.userName()

    @user_name.setter
    def user_name(self, v):
        self.shadow.setUserName(v)

    @property
    def password(self) -> str:
        """Пароль."""
        return self.shadow.password()

    @password.setter
    def password(self, v):
        self.shadow.setPassword(v)
