from .data_provider import DataProvider, Table
from .source import Source, Destination, Schema
from typing import Optional


class OracleSource(Source):
    pass


class OracleDestination(Destination):
    pass


class OracleDataProvider(DataProvider):
    """Провайдер для Базы Данных Oracle.

    Note:
        Для подключения к БД Oracle необходимо настроить ``Oracle Instant Client``.

        См. Руководство по установке и активации.

    Note:
        Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.da.provider_manager.oracle`.
    """
    _identifier = 'OracleDataProvider'

    def get_source(self, host: str, db_name: str, user: str,
                   password: str, port: int = 1521, dataobject: Optional[str] = None, sql: Optional[str] = None, alias: str = None) -> Source:
        """Создает описательную структуру для источника данных. Она в дальнейшем может быть использована при открытии данных :meth:`ProviderManager.open`.

        В качестве таблицы можно указать либо ее наименование `dataobject` либо текст запроса `sql`.

        Args:
            host: Адрес сервера.
            db_name: Имя базы данных.
            user: Имя пользователя.
            password: Пароль.
            port: Порт.
            dataobject: Имя таблицы.
            sql: SQL-запрос. Если указан, то он имеет более высокий приоритет по отношению к значению `dataobject`.

        Пример с указанием имени таблицы::

            definition = provider_manager.oracle.get_source('localhost', 'test', 'oracle', 'oracle', dataobject='world')
            table = provider_manager.open(definition)

        Пример с указанием текста запроса::

            definition = provider_manager.oracle.get_source('localhost', 'test', 'oracle', 'oracle', sql="select * from world where Страна like 'Р%'")
            table = provider_manager.open(definition)
        """
        return OracleSource(
            Source._provider(self.id),
            Source._alias(alias),
            {
                'host': host,
                'port': port,
                'db': db_name,
                'user': user,
                'password': password,
                'dataobject': dataobject,
                'sql' : sql
            }
        )

    def open(self, host: str, db_name: str, user: str,
                   password: str, port: int = 1521, dataobject: Optional[str] = None, sql: Optional[str] = None, alias: str = None) -> Table:
        """Открывает объект данных.

        В качестве таблицы можно указать либо ее наименование `dataobject` либо текст запроса `sql`.

        Args:
            host: Адрес сервера.
            db_name: Имя базы данных.
            user: Имя пользователя.
            password: Пароль.
            port: Порт.
            dataobject: Имя таблицы.
            sql: SQL-запрос. Если указан, то он имеет более высокий приоритет по отношению к значению `dataobject`.
            alias: Псевдоним для открываемой таблицы.
        """
        return self.get_source(host, db_name, user, password, port, dataobject, sql, alias).open()

    def get_destination(self, schema: Schema, dataobject: str, db_name: str,
                        host: str, user: str, password: str, port: int = 1521
                        ) -> Destination:
        """Создает назначение объекта данных.

        Args:
            schema: Схема таблицы.
            dataobject: Имя таблицы.
            db_name: Имя базы данных.
            host: Адрес сервера.
            user: Имя пользователя.
            password: Пароль.
            port: Порт.
        """
        return OracleDestination(schema,
                                 Source._provider(self.id),
                                 {
                                     'src': host,
                                     'port': port,
                                     'db': db_name,
                                     'user': user,
                                     'password': password,
                                     'dataobject': dataobject
                                 }
                                 )

    def create_open(self, schema: Schema, dataobject: str, db_name: str,
                        host: str, user: str, password: str, port: int = 1521
                        ) -> Table:
        """Создает и открывает объект данных.

        Args:
            schema: Схема таблицы.
            dataobject: Имя таблицы.
            db_name: Имя базы данных.
            host: Адрес сервера.
            user: Имя пользователя.
            password: Пароль.
            port: Порт.
        """
        return self.get_destination(schema, dataobject, db_name, host, user, password, port).create_open()
