"""
Функция конвертации
"""
import importlib
import os
import re
import tempfile
import time
import traceback
import xml.etree.ElementTree as ET
from datetime import datetime
from pathlib import Path

from PySide2.QtWidgets import QWizardPage
from axipy import tr, provider_manager
from osgeo import gdal, ogr
from osgeo.ogr import Layer

from .. import helper
from .. import wizardconverter

is_master = str(Path(__file__).parents[1].name).endswith("master")
if is_master:
    for module in (helper,):
        importlib.reload(module)


def convert(wzrd_page: QWizardPage, params: list) -> None:
    """
    Функция конвертации, выполняется в отдельном потоке.
    Посылает сигналы изменения прогресса в польз. интерфейс по ходу конвертации.

    :param wzrd_page:
    :param params: Список параметров конвертирования.
    """

    try:  # Основной процесс конвертации

        wzrd = wzrd_page.wzrd  # type: wizardconverter.WizardConverter

        def get_time(start: datetime, end: datetime) -> str:
            """
            Получение времени между двумя промежутками типа datetime.
            На выходе строка в формате %H:%M:%S либо в долях секунды.
            """
            time_delta = end - start
            seconds = time_delta.total_seconds()
            if seconds < 1:
                return "{:.3f}s".format(seconds)
            else:
                return time.strftime("%H:%M:%S", time.gmtime(seconds))

        def handler(err_level=None, err_no=None, err_msg=None) -> None:
            """
            Обработчик ошибок gdal.
            Отправляет строковый сигнал на поле вывода в польз. интерфейс.
            *args в пармаетрах функции, потому что gdal при ошибке может отправить недействительный callback,
            что приводит к падению программы.

            err_level: Уровень исключения gdal.
            err_no: Идентификатор исключения gdal.
            err_msg: Описание исключения gdal.
            """
            try:

                err_level = wzrd_page.err_msg_dict.get(err_level, "Unexpected")

                # Если gdal выдает сообщение о том что драйвер не поддерживает callback
                if str(err_msg).startswith("Progress turned off"):
                    if is_master:
                        print(f"callback turned off. {wzrd.inp_gdal_format} -> {wzrd.out_gdal_format}")
                    creation_options.pop("callback")  # Отключение callback
                    return None

                # Удаление новой строки и пробелов в конце строки
                p = re.compile(r"(\n|\s)*\.?$")
                err_msg = p.sub("", err_msg)
                err_str = "{} {}: {}".format(err_level, err_no, err_msg)
                full_string = ""

                # Обработка частного сообщения gdal (MapInfo File to ESRI Shapefile)
                p = re.compile(r"(?P<err_level>{}) (?P<err_no>{}): {}".format(
                    wzrd_page.err_msg_dict[2], 1,
                    r"(?P<txt1>Value) (?P<value>.*) (?P<txt2>of field) (?P<field>.*) (?P<txt3>of feature) "
                    r"(?P<feature>.*) (?P<txt4>not successfully written\."
                    r"\sPossibly due to too larger number with respect to field width)"))
                if p.match(err_str):
                    full_string = err_str
                    err_str = p.sub("(?P=txt1)(?P=txt2)(?P=field)(?P=txt3)(?P=txt4)", err_str)

                if err_str not in wzrd_page.err_str_dict.keys():  # Если строки ошибки нет в ключах
                    # Добавить ошибку в ключ со значением 1
                    wzrd_page.err_str_dict[err_str] = {"full_string": full_string, "count": 1}
                else:  # Если строка ошибки есть в ключах
                    wzrd_page.err_str_dict[err_str]["count"] += 1  # Значение по ключу увеличивается на 1
            except Exception as e:
                if is_master:
                    print("\n" * 2)
                    traceback.print_exc()
                    print("\n" * 4)
                wzrd_page.signals.text_edit_append.emit(str(e) + "\n")

        def print_and_clear_err_str_dict() -> None:
            """
            Вывод ошибок и предупреждений после каждого файла.
            """
            try:
                for k, v in wzrd_page.err_str_dict.items():
                    if v["full_string"]:
                        err_str = v["full_string"]
                    else:
                        err_str = k
                    # Точка в конце строки если строка не заканчивается на точку
                    err_str = helper.add_dot(err_str)
                    if v["count"] > 1:
                        wzrd_page.signals.text_edit_append.emit("{}".format(err_str))
                        wzrd_page.signals.text_edit_append.emit("{}: {}.".format(
                            tr("Пропущено похожих сообщений"), v["count"]))
                    else:
                        wzrd_page.signals.text_edit_append.emit("{}".format(err_str))
                    wzrd_page.signals.text_edit_append.emit("")
                wzrd_page.err_str_dict.clear()
            except Exception as e:
                wzrd_page.signals.text_edit_append.emit(str(e))

        def change_cs_for_mif(out_path_arg: Path) -> None:
            """
            Замена proj на prj для MapInfo File формата MIF,т.к. строка СК, формируемая gdal не вполне
            корректная (например, 9999 для долготы широты).
            """
            p = re.compile(r'^(CoordSys\s+)Earth\s+Projection\s+(?:\s*(?:\S+|"[^"]*")\s*,)*\s*(?:\S+|"[^"]*")(.*)')

            if (wzrd.out_gdal_format == "MapInfo File" and
                    "FORMAT=MIF" in creation_options["datasetCreationOptions"]):
                # Чтение из сконвертированного файла
                with open(str(out_path_arg), "r", encoding="cp1251") as f:
                    lines = f.readlines()
                # Поиск и замена строчки системы координат
                for line_indx, line in enumerate(lines):
                    # Проверка отмены конвертации
                    if wzrd_page.task.progress_handler().is_canceled():
                        return None
                    if p.match(line):
                        prj = wzrd.out_cs.prj
                        lines[line_indx] = p.sub(r"\1{}\2".format(prj), line)
                        break
                # Запись обратно в сконвертированный файл
                with open(str(out_path_arg), "w") as f:
                    f.write("".join(lines))

        def change_cs_for_tab(out_path_arg: Path) -> None:
            """
            Если на выходе tab;
            Если в выходной системе координат есть афинные преобразования;
            Система координат файла меняется вручную.
            """
            if (wzrd.out_gdal_format == "MapInfo File" and
                    "FORMAT=TAB" in creation_options["datasetCreationOptions"]):
                p = re.compile(r'Affine\s+Units\s+".*",\s+(.*)')
                out_affine = p.search(wzrd.out_cs.prj)
                if out_affine:
                    provider_manager.tab.change_coordsystem(str(out_path_arg), wzrd.out_cs)

        def change_name_if_exists(path: Path) -> Path:
            """
            Проверка существования пути к файлу, если путь существует, возвращает путь к файлу с номером в скобках.
            Например: Файл(2), Файл(3) и т.д. Функция выполняется рекурсивно, пока не найдется несуществующий путь.
            """
            if path.name in input_names:
                p = re.compile(r"\((\d+)\)$")
                search = p.search(str(path.stem))
                if search:
                    digit = int(search.group(1))
                    new_stem = p.sub(f"({digit + 1})", str(path.stem))
                else:
                    new_stem = path.stem + "(2)"
                new_path = (path.parent / new_stem).with_suffix(path.suffix)
                if new_path.name in input_names:
                    return change_name_if_exists(new_path)
                else:
                    input_names.append(new_path.name)
                    return new_path
            else:
                input_names.append(path.name)
                return path

        start_function = datetime.now()  # Время старта конвертации

        gdal.PopErrorHandler()  # Очистка обработчика ошибок gdal
        gdal.PushErrorHandler(handler)  # Инициализация обработчика

        # Распаковка пар-ов
        inp_paths, [dir_path, out_ext], open_options, creation_options = params

        # Создать директорию, без исключения, если существует
        Path.mkdir(dir_path, exist_ok=True)

        # For mode many_to_one
        gml_dict = {}.fromkeys(inp_paths)
        for gml_dict_key in gml_dict.keys():
            gml_dict[gml_dict_key] = list()

        input_names = list()

        # Основной цикл по файлам
        for inp_path_index, inp_path in enumerate(inp_paths):

            # Настройка полоски прогресса для работы с/без callback, и для одного файла
            if inp_path_index == 1:
                wzrd_page.signals.change_pbar_range.emit()  # Меняем с бесконечного на обычный range после 1ого элем.
            elif inp_path_index > 1:
                if "callback" in creation_options:
                    wzrd_page.signals.change_pbar.emit()
                else:
                    wzrd_page.signals.change_pbar_no_cb.emit()

            wzrd_page.signals.text_edit_append.emit("{}: {}\n".format(tr("Входной файл"), str(inp_path)))

            # Конструкция выходной строки
            out_path = dir_path.joinpath(inp_path.stem).with_suffix(out_ext)

            # gdal
            try:
                src_ds = gdal.OpenEx(str(inp_path), open_options=open_options)
                mode = wzrd.file_creation_mode

                if mode == "many_to_many":  # Аналогично исходному
                    out_path = change_name_if_exists(out_path)
                    wzrd_page.signals.text_edit_append.emit("{}: {}\n".format(tr("Выходной файл"), str(out_path)))
                    gdal.VectorTranslate(str(out_path), src_ds, **creation_options)
                    print_and_clear_err_str_dict()
                    wzrd_page.signals.text_edit_append.emit(wzrd_page.console_line)

                elif mode == "one_to_one":  # Каждую таблицу в отдельный файл
                    layer_count = src_ds.GetLayerCount()
                    if layer_count > 1:  # Создание выходной папки для многослойного файла на входе
                        new_dir_path = dir_path / inp_path.stem
                        Path.mkdir(new_dir_path, exist_ok=True)
                        wzrd_page.signals.text_edit_append.emit(
                            "{}: {}\n".format(tr("Выходная папка"), str(new_dir_path)))
                        wzrd_page.signals.text_edit_append.emit(wzrd_page.console_line)
                    for l_index in range(layer_count):  # Основной цикл по слоям
                        skip_layer = False
                        layer = src_ds.GetLayer(l_index)  # type: Layer
                        layer_name = layer.GetName()
                        creation_options["layers"] = layer_name
                        """
                        Driver specific
                        """
                        if layer_count > 1:  # Имя слоя вместо имени файла для многослойных файлов на входе
                            # У входных слоев DGN и DXF одинак. имена
                            if wzrd.inp_gdal_format in ("DGN", "DXF"):
                                out_path = new_dir_path.joinpath(f"{layer_name}_{inp_path_index}").with_suffix(out_ext)
                            else:
                                out_path = new_dir_path.joinpath(layer_name).with_suffix(out_ext)
                        # Выбор поля геометрии для выходных форматов DGN, DXF (Не поддерживают другие поля)
                        if wzrd.out_gdal_format in ("DGN", "DXF"):
                            geom_column = layer.GetGeometryColumn()
                            if geom_column == "":
                                geom_column = "_ogr_geometry_"
                            sql_str = f'SELECT "{geom_column}" FROM "{layer_name}"'  # Для DXF
                            if wzrd.out_gdal_format == "DGN":
                                sql_str = f'SELECT {240} AS "ColorIndex" FROM "{layer_name}"'  # Для DGN
                            creation_options["SQLStatement"] = sql_str
                            creation_options.pop("layers", None)
                        # Selafin поддерживает только слои типа POINT
                        elif wzrd.out_gdal_format == "Selafin":
                            geometry_name = helper.geometry_type_to_name(layer.GetGeomType())
                            if geometry_name != "POINT":
                                msg = f"" + \
                                      f"{tr('Драйвер Selafin поддерживает геометрию слоя одного типа (POINT)')}. " + \
                                      f"{tr('слой').capitalize()} {layer_name} " + \
                                      f"{tr('пропущен')}."
                                handler(2, 1, msg)
                                skip_layer = True

                        elif (
                                wzrd.out_gdal_format == "ESRI Shapefile"
                                # and False
                        ):
                            # Получение названий столбцов и их типов
                            column_name_type_dict = {}
                            ldfn = layer.GetLayerDefn()
                            field_count = ldfn.GetFieldCount()
                            for field_indx in range(field_count):
                                fdfn = ldfn.GetFieldDefn(field_indx)
                                column_name = fdfn.name
                                column_type = ogr.FieldDefn.GetFieldTypeName(fdfn, fdfn.GetType())
                                assert column_name not in column_name_type_dict
                                column_name_type_dict[column_name] = column_type

                            def less_than_n_bytes(str_arg, n_bytes):
                                """Сокращение длины строчки в байтах до заданного значения. (Меньше либо равно)"""
                                while len(str_arg.encode("utf-8")) > n_bytes:
                                    str_arg = str_arg[:-1]
                                return str_arg

                            def if_in_list_add_number(list_arg, str_arg):
                                """
                                Обработка одинаковых названий столбцов. Строчка сокращается до 8 байт
                                и в конец строчки добавляется порядковый номер от 1 до 99.
                                """
                                p = re.compile(r"[_\d]\d$")
                                while str_arg in list_arg:
                                    result = p.search(str_arg)
                                    if result:
                                        current_digit = int(result.group().replace("_", "0"))
                                        new_digit = current_digit + 1
                                        if 0 <= new_digit <= 9:
                                            str_arg = less_than_n_bytes(str_arg, 8) + "_" + str(new_digit)
                                        elif 10 <= new_digit <= 99:
                                            str_arg = less_than_n_bytes(str_arg, 8) + str(new_digit)
                                    else:
                                        str_arg = less_than_n_bytes(str_arg, 8) + "_1"

                                    str_arg = if_in_list_add_number(list_arg, str_arg)

                                return str_arg

                            # Получение новых имен не превышающих 10 байтов
                            new_names = []
                            for name in column_name_type_dict.keys():
                                size = 0
                                new_name = ""
                                for char in name:
                                    char_b = char.encode("utf-8")
                                    size += len(char_b)
                                    if size > 10:
                                        break
                                    else:
                                        new_name += char
                                new_name = if_in_list_add_number(new_names, new_name)
                                new_names.append(new_name)

                            # Создание SQL запроса для переименования новых полей и преобразования
                            # Real -> float (По умолчанию Gdal обнуляет поля Real больше 4 знаков)
                            new_column_names = []
                            for arg1, arg2 in zip(column_name_type_dict.keys(), new_names):
                                if column_name_type_dict[arg1] == "Real":
                                    new_column_names.append(f'CAST("{arg1}" as float(24)) as "{arg2}"')
                                elif arg1 != arg2:
                                    new_column_names.append(f'"{arg1}" as "{arg2}"')
                                elif arg1 == arg2:
                                    new_column_names.append(f'"{arg1}"')

                                if arg1 != arg2:
                                    msg = tr("Формат ESRI Shapefile поддерживает названия столбцов длиной "
                                             "не больше 10 байт. ") +\
                                          tr(f"Столбец '{arg1}' переименован в столбец '{arg2}'.")
                                    handler(2, 6, msg)

                            column_names_str = ", ".join(new_column_names)
                            sql_str = f'SELECT {column_names_str} FROM "{layer_name}"'

                            creation_options["SQLStatement"] = sql_str
                            creation_options.pop("layers", None)

                        """
                        Конвертация
                        """
                        if layer_count > 1:
                            wzrd_page.signals.text_edit_append.emit(
                                "{}: {}\n".format(tr("Входной слой"), layer_name))
                        # Обработка пропуска слоев
                        if skip_layer:
                            wzrd_page.signals.text_edit_append.emit(
                                "{}\n".format(tr("Выходной файл не создан.")))
                        else:  # Конвертация
                            out_path = change_name_if_exists(out_path)
                            wzrd_page.signals.text_edit_append.emit(
                                "{}: {}\n".format(tr("Выходной файл"), str(out_path)))
                            gdal.VectorTranslate(str(out_path), src_ds, **creation_options)
                            change_cs_for_mif(out_path)
                            change_cs_for_tab(out_path)
                        # Завершающее оформление
                        print_and_clear_err_str_dict()
                        wzrd_page.signals.text_edit_append.emit(wzrd_page.console_line)

                elif mode == "many_to_one":  # Собрать все таблицы в один файл

                    # Конвертация в промежуточный виртуальный формат для создания многослойного GML.
                    if wzrd.out_gdal_format == "GML":
                        for layer_index in range(src_ds.GetLayerCount()):
                            gml_dict[inp_path].append(src_ds.GetLayer(layer_index).GetName())
                        if inp_path_index == len(inp_paths) - 1:
                            if wzrd.file_name_for_append:
                                single_path = out_path.with_name(wzrd.file_name_for_append).with_suffix(
                                    out_ext)
                            else:
                                single_path = out_path

                            root_elem = ET.Element("OGRVRTDataSource")
                            for ds_path, layer_names in gml_dict.items():
                                for layer_name in layer_names:
                                    layer_elem = ET.SubElement(root_elem, "OGRVRTLayer", name=layer_name)
                                    ET.SubElement(layer_elem, "SrcDataSource").text = str(ds_path)
                                    if open_options:
                                        open_options_elem = ET.SubElement(layer_elem, "OpenOptions")
                                        for open_option in open_options:
                                            key, val = open_option.split("=")
                                            ET.SubElement(open_options_elem, "OOI", key=key).text = val
                            tree = ET.ElementTree(root_elem)

                            with tempfile.TemporaryDirectory() as tmpdirname:

                                vrt_path = Path(tmpdirname) / "vrt.xml"  # type: Path
                                vrt_path_str = str(vrt_path)
                                tree.write(vrt_path_str, encoding="utf-8")

                                # Конвертация
                                try:
                                    vrt_ds = gdal.OpenEx(str(vrt_path))
                                    wzrd_page.signals.text_edit_append.emit(
                                        "{}: {}\n".format(tr("Выходной файл"), str(single_path)))
                                    gdal.VectorTranslate(str(single_path), vrt_ds, **creation_options)
                                except Exception as exception:
                                    raise Exception("Ошибка виртуального формата") from exception
                                finally:
                                    vrt_ds = None

                            # Завершающие оформление
                            print_and_clear_err_str_dict()
                            wzrd_page.signals.text_edit_append.emit(wzrd_page.console_line)
                            break
                        else:
                            continue
                    else:
                        if inp_path_index == 0:  # На первом входном файле, задаем имя для одного собираемого файла
                            if wzrd.file_name_for_append:
                                single_path = out_path.with_name(
                                    wzrd.file_name_for_append).with_suffix(out_ext)
                            else:
                                single_path = out_path
                        if inp_path_index == 1:  # На втором входном файле, меняем режим на "append"
                            creation_options["accessMode"] = "append"
                            # Datasource creation options ignored since an existing datasource being updated.
                            creation_options.pop("datasetCreationOptions")
                        gdal.VectorTranslate(str(single_path), src_ds, **creation_options)
                        wzrd_page.signals.text_edit_append.emit("{}: {}".format(tr("Выходной файл"), str(single_path)))
                        print_and_clear_err_str_dict()
                        wzrd_page.signals.text_edit_append.emit(wzrd_page.console_line)

            except Exception as exception:
                raise Exception("Ошибка конвертации GDAL") from exception
            finally:
                src_ds = None

            # Проверка отмены конвертации
            if wzrd_page.task.progress_handler().is_canceled():
                break

        # Вывод длительности конвертации
        finish_function = datetime.now()
        wzrd_page.signals.text_edit_append.emit("{} ({}: {}).".format(
            tr("Конвертация завершена"),
            tr("длительность").capitalize(),
            get_time(start_function, finish_function)
        ))

    except Exception:
        if is_master:
            traceback.print_exc()
        wzrd_page.signals.text_edit_append.emit(tr("Ошибка в процессе конвертации.") + "\n")
