"""
Класс wizard и вспомогательные функции для страниц wizard.
"""
import importlib
import traceback
from pathlib import Path
from typing import Optional

from PySide2.QtCore import Qt, QStandardPaths
from PySide2.QtGui import QIcon
from PySide2.QtWidgets import QWizard, QComboBox, QMessageBox, QFileDialog, QLineEdit, QFormLayout, \
    QPushButton, QLabel
from axipy import tr, CoordSystem
from osgeo import gdal

from . import helper
from .json_pkg import jsonloader
from .wizardpage import creationpage, opensettingspage, resultspage, welcomepage

is_master = str(Path(__file__).parents[0].name).endswith("master")

if is_master:
    for module in (helper, jsonloader, creationpage, opensettingspage, resultspage, welcomepage):
        importlib.reload(module)

window_title = tr("Конвертер")
icon_path = str(Path(__file__).parent / "icon" / "Translate-03.png")


class WizardConverter(QWizard):
    """
    В классе создается Wizard, определяются его параметры, происходит добавление страниц и настройка действий кнопок
    навигации внизу страниц ("Далее", "Назад" и т.д.).
    В классе находятся переменные, в которые собирается пользовательский ввод со всех страниц, для последующей передачи
    в функцию конвертирования.
    """

    def __init__(self, main_window, lang):
        """
        Выполняется при создании экземпляра класса.
        """
        super().__init__(main_window)

        # Загрузка словаря из json
        self.drivers = jsonloader.load(lang)  # type: Optional[dict]

        self.wizard_width = 550  # type: int
        self.wizard_height = 700  # type: int

        """
        Значения запоминаются.
        (До перезагрузки плагина и перезапуска аксиомы).
        """
        self.documents_path = Path(QStandardPaths.writableLocation(QStandardPaths.DocumentsLocation))
        # Входная папка
        self.inp_dir_path = self.documents_path  # type: Optional[Path]
        # Выходная папка.
        self.out_dir_path = None  # type: Optional[str]
        # Полный путь (Выходная папка + папка результатов)
        self.out_path = None  # type: Optional[Path]
        # Выходной формат
        # Задается при валидации начальной страницы добавления файлов
        self.out_gdal_format = None  # type: Optional[str]
        self.skip_failures = True  # Пропуск исключений
        # Значение СК по умолчанию
        self.default_cs = CoordSystem.from_proj("+proj=longlat +ellps=WGS84 +towgs84=0,0,0,0,0,0,0 +no_defs")
        """
        Значения не запоминаются.
        (Сбрасываются при рестарте и при закрытии окна).
        """
        # Список всех входных файлов
        self.input_path_list = None  # type: Optional[list]
        # Формат gdal входных файлов
        self.inp_gdal_format = None  # type: Optional[str]
        # Выходное расширение
        self.out_ext = None  # type: Optional[str]

        # Система координат

        # Первоначально прочитанная СК
        self.inp_cs_initial = None  # type: Optional[CoordSystem]
        # Входная СК
        self.inp_cs = None  # type: Optional[CoordSystem]
        # Выходная СК
        self.out_cs = None  # type: Optional[CoordSystem]
        # Флаг перепроецирования СК
        self.reproject = None  # type: Optional[bool]
        # Настройки конвертирования вводимые пользователем
        self.open_options, self.dataset, self.layer = [None, None, None]

        # Режим создания файлов
        self.file_creation_mode = None  # type: Optional[str]
        self.file_creation_modes = {
            "many_to_many": tr("Аналогично исходному"),
            "one_to_one": tr("Каждую таблицу в отдельный файл"),
            "many_to_one": tr("Собрать все таблицы в один файл")
        }
        self.file_name_for_append = None  # type: Optional[str]

        # Состояние режима исключений gdal
        self.use_exceptions = None
        # Прерывание задачи из потока
        self.task_rejected = False
        # Сообщение для прерывания конвертации
        self.m_box_q = QMessageBox(
            QMessageBox.Question, tr("Вопрос"),
            tr("Прервать процесс конвертации?"),
            buttons=QMessageBox.Yes | QMessageBox.No, parent=self)

        self.init_gui()

        self.init_pages()

    def init_gui(self):
        # Настройки главного окна
        self.setWindowTitle(window_title)  # Отображаемое имя окна
        # Фикс. размер окна
        self.setFixedSize(self.wizard_width, self.wizard_height)
        # Размер шрифта
        font = self.font()
        font.setPointSize(9)
        self.setFont(font)
        # Модальность окна
        self.setModal(True)
        # Значок wizard
        icon = QIcon(icon_path)
        self.setWindowIcon(icon)
        # Включение и настройка Пользов. Кнопки 1
        self.setOption(QWizard.HaveCustomButton1)
        self.setButtonText(self.CustomButton1, tr("Сконвертировать ещё"))
        self.button(QWizard.CustomButton1).clicked.connect(self.custom_btn1_press)
        # Включение и настройка Пользов. Кнопки 2
        self.setOption(QWizard.HaveCustomButton2)
        self.setButtonText(self.CustomButton2, tr("Прервать"))
        self.button(QWizard.CustomButton2).clicked.connect(self.custom_btn2_press)
        # Флаги
        self.setWindowFlags(self.windowFlags() & ~Qt.WindowContextHelpButtonHint)  # Выключение значка "?"
        # Стиль wizard
        self.setWizardStyle(QWizard.ClassicStyle)

    def init_pages(self):
        # Начальная страница
        welcome_page = welcomepage.WelcomePage(self)
        # Страница с параметрами открытия входных файлов
        open_settings_page = opensettingspage.OpenSettingsPage(self)
        # Страница с параметрами создания выходных файлов
        creation_page = creationpage.CreationPage(self)
        # # Страница результатов конвертирования
        self.results_page = resultspage.ResultsPage(self)

        # Добавление страниц в wizard
        for elem in (
                welcome_page,
                open_settings_page,
                creation_page,
                self.results_page
        ):
            self.addPage(elem)

    # Wizard buttons

    def reject(self) -> None:
        """
        Отмена задачи из потока при закрытии окна.
        """

        def prepare_to_close() -> None:
            """
            Возвращение значения режима исключений gdal в состояние до запуска окна плагина
            """
            set_use_exeptions = {
                0: gdal.DontUseExceptions,
                1: gdal.UseExceptions
            }
            if self.use_exceptions in set_use_exeptions:
                set_use_exeptions[self.use_exceptions]()
            gdal.PopErrorHandler()  # Очистка обработчика ошибок gdal

        self.cancel_task()
        self.task_rejected = True
        if not self.has_running_task():
            prepare_to_close()
            # Закрыть окно
            super().reject()

    def custom_btn1_press(self) -> None:
        """
        Перезапуск wizard с 1 страницы при нажатии кнопки "Сконвертировать еще".
        """
        self.restart()

    def custom_btn2_press(self) -> None:
        """
        Отмена потока по нажатию "Прервать"
        """
        self.cancel_task()

    # Task methods

    def cancel_task(self) -> None:
        """
        Послать сигнал на отмену активной задаче. Доп. подтверждение отмены для страницы конвертации.
        Режим полоски прогресса "прерывание".
        """
        page = self.currentPage()
        if hasattr(page, "task"):
            task = page.task
            if task:
                if not task.progress_handler().is_canceled():
                    # Доп. подтверждение для страницы результатов
                    if page == self.results_page:
                        self.m_box_q.exec_()
                        if self.m_box_q.result() == QMessageBox.No:
                            return None
                    # Здесь может быть пауза из-за exec, поэтому еще проверка
                    if self.has_running_task():
                        task.progress_handler().cancel()
                        if hasattr(page, "pbar"):
                            pbar = getattr(page, "pbar")
                            pbar.setRange(0, 100)
                            pbar.setValue(0)
                            pbar.setFormat(tr("Идет прерывание процесса..."))
                            pbar.show()

    def has_running_task(self) -> bool:
        """
        Проверка, есть ли на текущей странице активная задача.
        """
        page = self.currentPage()
        if hasattr(page, "task"):
            task = page.task
            if task and task.progress_handler().is_running():
                return True

        return False

    def starting_task(self) -> None:
        """
        Общие действия при старте задачи в потоке.
        """
        page = self.currentPage()

        layout = [QWizard.Stretch, QWizard.BackButton, QWizard.NextButton, QWizard.CustomButton2,
                  QWizard.CancelButton]
        self.setButtonLayout(layout)

        self.button(QWizard.BackButton).setEnabled(False)
        self.button(QWizard.NextButton).setEnabled(False)
        self.button(QWizard.CancelButton).setEnabled(False)

        page.pbar.setRange(0, 0)
        page.pbar.resetFormat()
        page.pbar.show()

    def finishing_task(self) -> None:
        """
        Общие действия после завершения задачи в потоке.
        """
        page = self.currentPage()
        page.task = None

        page.pbar.hide()

        self.button(QWizard.BackButton).setEnabled(True)
        self.button(QWizard.NextButton).setEnabled(True)
        self.button(QWizard.CancelButton).setEnabled(True)
        # Проверка в самом конце
        if self.task_rejected:
            self.reject()

    # Form functions (gets connected in jsonloader module)

    def dynamic_form_validation(self) -> None:
        """
        Проверяет совместимость выбранных пользователем параметров во время работы программы.
        Совместимость параметров проверяется на форме построенной из json. Формы загружаются на странице выбора
        пар-ов открытия и пар-ов создания. Привязывается только к QComboBox и QCheckBox.
        """

        try:
            type_list = ["open_options", "dataset", "layer"]
            indx_list = [type_list.index(type_) for type_ in type_list]
            type_to_indx = dict(zip(type_list, indx_list))
            indx_to_type = dict(zip(indx_list, type_list))

            drivers = self.drivers

            page = self.currentPage()
            q_scroll_area = page.form_view
            form = q_scroll_area.widget()
            # Все виджеты на форме
            form_widgets = [widget for widget in form.children()
                            if not isinstance(widget, (QFormLayout, QLabel, QPushButton))]
            # Объектные имена всех виджетов на форме
            form_widgets_obj_names = [widget.objectName() for widget in form_widgets]
            # Виджет по объектному имени
            widget_from_obj = dict(zip(form_widgets_obj_names, form_widgets))

            def get_dict_from_obj(obj_name):
                assert obj_name != ""
                option_indx = int(obj_name[:1])
                option_type = indx_to_type[option_indx]
                sender_w_setting_name = obj_name[1:]

                gdal_format = None
                if option_indx in (0,):
                    gdal_format = self.inp_gdal_format
                elif option_indx in (1, 2):
                    gdal_format = self.out_gdal_format
                assert gdal_format is not None

                return option_type, drivers[gdal_format][option_type][sender_w_setting_name]

            def get_current_value_of_form_widget(obj_name):
                type_key_arg, dict_arg = get_dict_from_obj(obj_name)
                widget_type = dict_arg["type"]
                if widget_type == "QCheckBox":
                    check_box_w = widget_from_obj[obj_name]
                    value = check_box_w.checkState()
                    bool_ = helper.chkbox_state_dict[value]
                    w_value = helper.get_key(dict_arg["values"], bool_)
                elif widget_type == "QComboBox":
                    cbox_w = widget_from_obj[obj_name]  # type: QComboBox
                    txt = cbox_w.currentText()
                    w_value = helper.get_key(dict_arg["values"], txt)

                return w_value

            for form_widget_obj_name in form_widgets_obj_names:
                if form_widget_obj_name == "":
                    return None
                type_key, dict_ = get_dict_from_obj(form_widget_obj_name)
                if "disable_if" not in dict_:
                    continue
                disable_dict = dict_["disable_if"]
                bool_list = []
                # Логика OR
                for setting_name, setting_values in disable_dict.items():

                    disable_dict_obj_name = str(type_to_indx[type_key]) + setting_name
                    current_w_value = get_current_value_of_form_widget(disable_dict_obj_name)

                    if current_w_value in setting_values:
                        bool_list.append(True)
                    else:
                        bool_list.append(False)

                widget = widget_from_obj[form_widget_obj_name]

                if any(bool_list):
                    widget.setEnabled(False)
                elif not widget.isEnabled():
                    widget.setEnabled(True)

        except Exception:
            if is_master:
                traceback.print_exc()

    def open_filename_from_options_form(self, lnd: str, filter_str: str) -> None:
        fname = QFileDialog.getOpenFileName(
            parent=self,
            filter=filter_str
        )[0]
        if fname:
            page = self.currentPage()
            page.form_view.findChild(QLineEdit, lnd).setText(fname)
