from axipy.cpp_core_dp import DataObject as ShadowDataObject
from axipy.cpp_core_dp import Table as ShadowTable
from axipy.cpp_core_dp import TransactionalTable, ShadowRaster
from axipy.cpp_core_dp import RasterFacade as CppRasterFacade
from axipy.cs import CoordSystem
from enum import IntFlag
import typing
from typing import Union, Iterator, Optional, List
from PySide2.QtCore import QRectF, QSize, Signal
from PySide2.QtGui import QTransform
from axipy.da.FeatureWrapper import Feature, ShadowFeatureIterator
from axipy.da.Geometry import Geometry
from axipy.da.attribute_schema import Schema
from axipy.utl import Rect
from axipy.da.raster import GCP


__all__ = [
    'DataObject',
    'Table',
    'SupportedOperations',
    'Raster',
    'GCP',
]


class NotRegistered(RuntimeError):
    pass


class NoCoordSystem(RuntimeError):
    pass


class DataObject:
    """Объект данных.

    Открываемые объекты из источников данных представляются объектами
    этого типа. Возможные реализации: таблица, растр, грид, чертеж,
    панорама, и так далее.

    Пример::

        table = provider_manager.openfile('path/to/file.tab')
        ...
        table.close()  # Закрывает таблицу

    Для закрытия объекта данных можно использовать менеджер контекста -
    выражение :any:`with<with>`. В таком случае таблица будет закрыта при выходе
    из блока. См. :meth:`close`.

    Пример::

        with provider_manager.openfile('path/to/file.tab') as raster:
            ...
            # При выходе из блока растр будет закрыт

    """

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap_typed(cls, shadow):
        result = cls.__new__(cls)
        result.shadow = shadow
        cls._assert_wrap(result)
        return result

    @classmethod
    def _assert_wrap(cls, instance):
        pass

    @classmethod
    def _wrap(cls, obj):
        if obj is None:
            return None
        if isinstance(obj.get(), TransactionalTable):
            return Table._wrap_typed(obj, is_editable=True)
        if isinstance(obj.get(), ShadowTable):
            return Table._wrap_typed(obj, is_editable=False)
        if isinstance(obj.get(), ShadowRaster):
            return Raster._wrap_typed(obj)
        if isinstance(obj.get(), ShadowDataObject):
            return DataObject._wrap_typed(obj)
        return None

    @property
    def name(self) -> str:
        """Название объекта данных."""
        return self.shadow.name()

    @name.setter
    def name(self, name: str) -> str:
        return self.shadow.setName(name)

    @property
    def provider(self) -> str:
        """Провайдер изначального источника данных."""
        return self.shadow.provider()

    def __eq__(self, other) -> bool:
        return self.shadow.get() == other.shadow.get()

    def close(self):
        """Пытается закрыть таблицу.

        Raises:
            RuntimeError: Ошибка закрытия таблицы.

        Note:

            Объект данных не всегда может быть сразу закрыт. Например, для
            таблиц используется транзакционная модель редактирования и перед
            закрытием необходимо сохранить или отменить изменения, если они
            есть.
            См. :attr:`Table.is_modified`.
        """
        self.shadow.closeRequest()

    def __enter__(self):
        return self

    def __exit__(self, exception_type, exception_value, traceback):
        self.close()

    @property
    def is_spatial(self) -> bool:
        """Признак того, что объект данных является пространственным."""
        return self.shadow.isSpatial()

    @property
    def properties(self) -> dict:
        """Дополнительные свойства объекта данных."""
        return self.shadow.properties()

    @property
    def destroyed(self) -> Signal:
        """Сигнал оповещения об удалении объекта."""
        return self.shadow.destroyed


class SupportedOperations(IntFlag):
    """Флаги доступных операций.
    
    Реализованы как перечисление :class:`enum.IntFlag` и поддерживают побитовые
    операции.

    .. csv-table::
        :header: Атрибут, Значение
        
        `Empty`, 0
        `Insert`, 1
        `Update`, 2
        `Delete`, 4
        `Write`, `Insert` | `Update` | `Delete` = 7
        `Read`, 8
        `ReadWrite`, `Read` | `Write` = 15

    .. literalinclude:: /../../tests/doc_examples/test_example_supported_operations.py
        :caption: Пример использования
        :pyobject: example_supported_operations
        :lines: 2-
        :dedent: 4
    
    See Also:
        :class:`enum.IntFlag`, :attr:`axipy.da.Table.supported_operations`
    """
    Empty = 0
    Insert = 1
    Update = 2
    Delete = 4
    Write = 7
    Read = 8
    ReadWrite = 15


class Table(DataObject):
    """Таблица.

    Менеджер контекста сохраняет изменения и закрывает таблицу.

    Пример::

        with provider_manager.openfile('path/to/file.tab') as table:
            ...
            # При выходе из блока таблица будет сохранена и закрыта

    See also:
        :meth:`commit`, :meth:`DataObject.close`.
    """

    @classmethod
    def _wrap_typed(cls, shadow, is_editable=False):
        result = cls.__new__(cls)
        result.shadow = shadow
        result.editable = is_editable
        return result


    @property
    def data_changed(self) -> Signal:
        """Сигнал об изменении данных таблицы. Испускается когда были изменены данные таблицы.
                
        .. literalinclude:: /../../tests/doc_examples/test_example_table.py
            :caption: Пример подписки на изменение таблицы.
            :pyobject: test_run_example_table_change_table
            :lines: 3-
            :dedent: 4
        """
        return self.shadow.dataChanged

    @property
    def schema_changed(self) -> Signal:
        """Сигнал об изменении схемы таблицы. Испускается когда была изменена структура таблицы."""
        return self.shadow.schemaChanged

    @property
    def is_editable(self) -> bool:
        """Признак того, что таблица является редактируемой."""
        return self.editable

    @property
    def is_temporary(self) -> bool:
        """Признак того, что таблица является временной."""
        return self.shadow.isTemporary()

    @property
    def is_modified(self) -> bool:
        """Таблица содержит несохраненные изменения."""
        return self.shadow.hasModified() if self.is_editable else False

    @property
    def supported_operations(self) -> SupportedOperations:
        """Доступные операции.
        
        .. literalinclude:: /../../tests/doc_examples/test_example_supported_operations.py
            :caption: Пример использования
            :pyobject: example_supported_operations
            :lines: 2-
            :dedent: 4
        """
        return SupportedOperations(int(self.shadow.supportedOperations()))

    @property
    def schema(self) -> Schema:
        """Возвращает схему таблицы."""
        return Schema._from_dict(self.shadow.schema())

    @property
    def coordsystem(self) -> typing.Union[CoordSystem, None]:
        """Система координат таблицы."""
        return CoordSystem._wrap(self.shadow.coordSystem())

    def ensure_editable(self):
        if not self.is_editable:
            raise RuntimeError('Table is not editable')

    def insert(self, features: Union[Iterator[Feature], Feature]):
        """Вставляет записи в таблицу.

        Args:
            features: Записи для вставки.
        """
        self.ensure_editable()
        if isinstance(features, Feature):
            features = [features]
        self.shadow.insert(f.shadow for f in features)

    def update(self, features: Union[Iterator[Feature], Feature]):
        """Обновляет записи в таблице.

        Args:
            features: Записи для обновления.

        При обновлении проверяется :attr:`Feature.id`. Если запись с таким
        идентификатором не найдена, то она пропускается.

        .. literalinclude:: /../../tests/doc_examples/test_example_table.py
            :caption: Пример использования
            :pyobject: example_table_update
            :lines: 2-
            :dedent: 4

        See also:
            :attr:`Feature.id`, :meth:`commit`, :attr:`is_modified`.
        """
        self.ensure_editable()
        if isinstance(features, Feature):
            features = [features]
        self.shadow.update(f.shadow for f in features)

    def remove(self, ids: Union[int, Iterator[int]]):
        """Удаляет записи из таблицы.

        Args:
            ids: Идентификаторы записей для удаления.
        """
        self.ensure_editable()
        if isinstance(ids, int):
            ids = [ids]
        self.shadow.remove(ids)

    def commit(self):
        """Сохраняет изменения в таблице.

        Если таблица не содержит несохраненные изменения, то команда
        игнорируется.

        Raises:
            RuntimeError: Невозможно сохранить изменения.
        """
        self.ensure_editable()
        if self.is_temporary:
            raise RuntimeError(
                'Невозможно сохранить изменения во временной таблице.')
        self.shadow.commit()

    def restore(self):
        """Отменяет несохраненные изменения в таблице.

        Если таблица не содержит несохраненные изменения, то команда
        игнорируется.
        """
        self.ensure_editable()
        self.shadow.restore()

    @property
    def __is_transaction_table(self):
        return isinstance(self.shadow.get(), TransactionalTable)

    @property
    def can_undo(self) -> bool:
        """Возможен ли откат на один шаг назад."""
        return self.__is_transaction_table and self.shadow.canRollBack()
    
    def undo(self):
        """Производит откат на один шаг назад."""
        if self.can_undo:
            self.shadow.rollBack()

    @property
    def can_redo(self) -> bool:
        """Возможен ли откат на один шаг вперед."""
        return self.__is_transaction_table and self.shadow.canRollForward()
    
    def redo(self):
        """Производит откат на один шаг вперед. При этом возвращается состояние до последней отмены."""
        if self.can_redo:
            self.shadow.rollForward()

    def itemsInRect(self, bbox: Union[Rect, QRectF, tuple]) -> Iterator[Feature]:
        """Запрашивает записи с фильтром по ограничивающему прямоугольнику.

        Args:
            bbox: Ограничивающий прямоугольник.

        Returns:
            Итератор по записям.

        .. literalinclude:: /../../tests/doc_examples/test_example_table.py
            :caption: Пример запроса (таблица в проекции `Робинсона`)
            :pyobject: test_run_example_table_query_mbr
            :lines: 3-
            :dedent: 4

        """
        return ShadowFeatureIterator(self.shadow.itemsInMbr(Rect._rect_value_to_qt(bbox)))

    def itemsInObject(self, obj: Geometry) -> Iterator[Feature]:
        """Запрашивает записи с фильтром по геометрическому объекту.

        Args:
            obj: Геометрия. Если для нее не задана СК, используется СК таблицы.

        Returns:
            Итератор по записям.

        .. literalinclude:: /../../tests/doc_examples/test_example_table.py
            :caption: Пример запроса по полигону
            :pyobject: test_run_example_table_query_obj
            :lines: 3-
            :dedent: 4
        """
        geom = obj
        if isinstance(geom, Geometry) and geom.coordsystem is None:
            geom = obj.clone()
            geom.coordsystem = self.coordsystem
        return ShadowFeatureIterator(self.shadow.itemsInObject(geom.shadow))

    def itemsByIds(self, ids: List[int]) -> Iterator[Feature]:
        """Запрашивает записи по списку :class:`list` с идентификаторами записей,
        либо перечень идентификаторов в виде списка. Идентификаторы несохраненных
        записей имеют отрицательные значения.

        Args:
            ids: Список идентификаторов.

        Returns:
            Итератор по записям.

        .. literalinclude:: /../../tests/doc_examples/test_example_table.py
            :caption: Пример
            :pyobject: test_run_example_table_query_ids
            :lines: 3-
            :dedent: 4
        """
        return ShadowFeatureIterator(self.shadow.itemsById(ids))

    def items(self, bbox: Union[Rect, QRectF, tuple] = None, ids: List[int] = None) -> Iterator[Feature]:
        """Запрашивает записи, удовлетворяющие параметрам. В качестве фильтра может быть указан либо ограничивающий прямоугольник,
        либо перечень идентификаторов в виде списка.

        Args:
            bbox: Ограничивающий прямоугольник.
            ids: Список идентификаторов.

        Returns:
            Итератор по записям.
        """
        if bbox is not None:
            if not self.is_spatial:
                raise RuntimeError('Table is not spatial')
            return self.itemsInRect(bbox)
        if ids is not None:
            return self.itemsByIds(ids)
        return ShadowFeatureIterator(self.shadow.items())

    def count(self, bbox: typing.Union[Rect, QRectF, tuple] = None) -> int:
        """Возвращает количество записей, удовлетворяющих параметрам.

        Данный метод является наиболее предпочтительным для оценки
        количества записей. При этом используется наиболее оптимальный
        вариант выполнения запроса для каждого конкретного провайдера
        данных.

        Args:
            bbox: Ограничивающий прямоугольник.

        Returns:
            Количество записей.
        """
        if bbox is not None:
            if not self.is_spatial:
                raise RuntimeError('Table is not spatial')
            return self.shadow.countInMbr(Rect._rect_value_to_qt(bbox))
        return self.shadow.count()

    def __len__(self) -> int:
        return self.count()

    def __iter__(self):
        return self.items()

    def _commit_if_possible(self):
        if not self.is_editable or self.is_temporary:
            return
        self.commit()

    def __exit__(self, exception_type, exception_value, traceback):
        self._commit_if_possible()
        super().__exit__(exception_type, exception_value, traceback)


class Raster(DataObject):
    """Растровый объект."""
    
    @classmethod
    def _assert_wrap(cls, instance: 'Raster'):
        if not instance.shadow.hasTransform():
            raise NotRegistered()

    @property
    def coordsystem(self) -> typing.Union[CoordSystem, None]:
        """Система координат растра."""
        return CoordSystem._wrap(self.shadow.coordSystem())

    @property
    def size(self) -> QSize:
        """Размер растра в пикселях."""
        return self.shadow.size()

    @property
    def device_to_scene_transform(self) -> QTransform:
        """Матрица преобразования из точек на изображении (пиксели) в точки
        на карте."""
        return CppRasterFacade.getTransform(self.shadow.get())

    @property
    def scene_to_device_transform(self) -> QTransform:
        """Матрица преобразования из точек на карте в точки на
        изображении (пиксели)."""
        return self.device_to_scene_transform.inverted()

    def get_gcps(self) -> List[GCP]:
        """Возвращает точки привязки."""
        cpp_result = CppRasterFacade.getGCPs(self.shadow.get())
        return list(map(GCP._wrap, cpp_result))
