"""
Загружает данные из json.
"""
import json
from datetime import date
from pathlib import Path
import re

from PySide2.QtCore import Qt, QRegExp
from PySide2.QtGui import QRegExpValidator
from PySide2.QtWidgets import (
    QCheckBox, QComboBox, QDateEdit, QFormLayout, QFrame, QHBoxLayout, QLabel, QLineEdit,
    QScrollArea, QSpinBox, QWidget, QWizardPage)
from axipy import tr

# Перевод из индекса в текст(имена типов настроек gdal)
options_types_list = ["open_options", "dataset", "layer"]
# Перевод настроек gdal для отображения в gui
json_keys_to_gui_options = {
    "open_options": tr("Параметры открытия"),
    "dataset": tr("Параметры создания набора данных"),
    "layer": tr("Параметры создания слоя"),
}


def load(json_filename: str, lang: str) -> dict:
    """
    Загружает список драйверов с параметрами из json файла, проверяет на наличие необходимых ключей.

    :param json_filename: Имя файла json с драйверами. Файл .json должен быть в той же папке что и данный скрипт.
    :param lang: Значение языка Аксиомы.
    """
    # Загрузка из json в словарь
    path = Path(__file__).with_name(json_filename)
    with open(path, encoding="utf-8") as f:
        drivers = json.load(f)

    loc = re.compile("[a-z]{2}$")  # Шаблон ключа локализации, например "en", "ru"

    def nested_dict(d: dict):
        """
        Рекурсивный цикл по словарю и поиск словарей с ключами локализаци(словарей локализации).
        Превращает словари локализации в строковое значение, соответствующее текущей локализации.
        """
        for k, v in d.items():
            if type(v) is dict:
                if all(re.match(loc, re_key) for re_key in v.keys()):
                    d[k] = v[lang]
                nested_dict(v)

    nested_dict(drivers)

    # Проверка на наличие необходимых ключей
    for k, v in drivers.items():
        for elem in ["name", "creation"]:
            if elem not in v:
                raise Exception("{} {} {} {}.".format(tr("У формата"), k,
                                                      tr("нет ключа"), elem))

    return drivers


def get_key(d: dict, val) -> str:
    """
    Возвращает ключ по значению в словаре.
    """
    for k, v in d.items():
        if v == val:
            return k


def get_indx_from_type(type_: str) -> int:
    """
    Получает индекс настройки по её типу.

    :param type_: Тип настройки.
    :return: Индекс настроки.
    """
    return options_types_list.index(type_)


def get_form_view(gui_obj: QWizardPage, driver_name: str, options_list: list) -> QScrollArea:
    """
    Создает и возвращает ссылку на созданную разметку формы параметров(в QScrollArea) по имени драйвера и
    списку типов параметров.

    :param QWizardPage gui_obj: Объект QWizardPage.
    :param driver_name: Формат gdal (может быть входной или выходной).
    :param list options_list: Список типов параметров для формы.
    :return: QScrollArea с заполненной формой.
    """
    drivers = gui_obj.parent.drivers  # Получение словаря с параметрами драйверов

    def fill_form(option_name_arg: str, driver: str, form_arg: QFormLayout):
        """
        Функция, заполняющая форму.
        """
        option_indx = get_indx_from_type(option_name_arg)
        str_option_indx = str(option_indx)

        try:
            drivers[driver][option_name_arg]
        except KeyError:  # Если ключа еще нет в json
            label = None
            msg1 = tr("В данный момент программа не поддерживает")
            msg2 = tr("Параметры можно добавить вручную в поле ввода, через точку с запятой \";'\".")
            if option_name_arg == "open_options":
                msg = (msg1, tr("параметры открытия для текущего входного формата."), msg2)
                label = QLabel(" ".join(msg))
            elif option_name_arg == "dataset":
                msg = (msg1, tr("параметры создания набора данных для выбранного выходного формата."), msg2)
                label = QLabel(" ".join(msg))
            elif option_name_arg == "layer":
                msg = (msg1, tr("параметры создания слоя для выбранного выходного формата."), msg2)
                label = QLabel(" ".join(msg))
            if label:
                label.setWordWrap(True)
                line_edit = QLineEdit()
                line_edit.setObjectName(str_option_indx + "semicolon")
                form_arg.addRow(label)
                form_arg.addRow(line_edit)

            return None
        else:
            # Если у ключа точно не будет значения
            if drivers[driver][option_name_arg] is None:
                label = None
                if option_name_arg == "open_options":
                    label = QLabel(tr("Входной формат не имеет параметров открытия."))
                elif option_name_arg == "dataset":
                    label = QLabel(tr("Выходной формат не имеет параметров создания набора данных."))
                elif option_name_arg == "layer":
                    label = QLabel(tr("Выходной формат не имеет параметров создания слоя."))
                if label:
                    form_arg.addRow(label)
                return None
            # Основное заполнение формы
            else:
                params = drivers[driver][option_name_arg]
                params_keys = list(params.keys())
                params_values = list(params.values())
                for i, setting in enumerate(params_values):

                    # Если требуется просто создать QLabel по середине для группировки пар-ов
                    if setting["type"] == "QLabel":
                        hbox = QHBoxLayout()
                        label = QLabel(setting["name"])
                        label.setToolTip(setting["tooltip"])
                        label.setAlignment(Qt.AlignCenter)
                        hbox.addWidget(label)
                        form_arg.addRow(hbox)
                        continue

                    # label
                    label = QLabel(setting["name"])
                    # field
                    field = None

                    # LineEdit
                    if setting["type"] == "QLineEdit":
                        field = QLineEdit()
                        if "validator" in setting:
                            field.setValidator(QRegExpValidator(QRegExp(setting["validator"])))

                    # ComboBox
                    elif setting["type"] == "QComboBox":
                        field = QComboBox()
                        for v in setting["values"].values():
                            field.addItem(v)
                        field.currentIndexChanged.connect(gui_obj.dynamic_form_validation)

                    # CheckBox
                    elif setting["type"] == "QCheckBox":
                        field = QCheckBox()
                        bool_ = list(setting["values"].values())[0]
                        field.setChecked(bool_)
                        field.stateChanged.connect(gui_obj.dynamic_form_validation)

                    # DateEdit
                    elif setting["type"] == "QDateEdit":
                        field = QDateEdit()
                        field.setDate(date.today())

                    # SpinBox
                    elif setting["type"] == "QSpinBox":
                        field = QSpinBox()
                        field.setMinimum(setting["values"]["min"])
                        field.setMaximum(setting["values"]["max"])
                        field.setSingleStep(setting["values"]["single_step"])

                    # Общие настройки поля
                    label.setToolTip(params_keys[i])
                    label.setWordWrap(True)
                    width = gui_obj.parent.wizard_width * 35 // 100
                    label.setFixedWidth(width)

                    field.setToolTip(setting["tooltip"])
                    field.setObjectName(str_option_indx + params_keys[i])

                    if "exec" in setting and field:
                        exec(setting["exec"])

                    # Добавление метки и поля на форму
                    form_arg.addRow(label, field)

    # Создание формы
    form_layout = QFormLayout()  # layout will be added later to parent
    form_layout.setContentsMargins(0, 0, 0, 0)

    # Форма заполняется переданными типами настроек
    for option_name in options_list:
        # Добавление названия параметров
        title_label = QLabel(json_keys_to_gui_options[option_name])
        title_label.setStyleSheet("font-weight: bold")
        form_layout.addRow(title_label)
        # Вызов функции заполняющей форму
        fill_form(option_name, driver_name, form_layout)

    form_widget = QWidget(gui_obj)
    form_widget.setLayout(form_layout)
    scroll_area = QScrollArea(gui_obj)
    scroll_area.setFrameStyle(QFrame.NoFrame)
    scroll_area.setWidget(form_widget)
    scroll_area.setWidgetResizable(True)

    return scroll_area
