import axipy
from axipy.gui import selection_manager
from axipy.da import (Feature, Geometry, Line, LineString, Type,
                      Polygon, GeometryCollection, MultiLineString,
                      MultiPolygon)
from typing import List
from axipy import Pnt
from axipy.interface import AxiomaInterface
from axipy.app import Notifications
from .helper import ensure_editable_table
from axipy.concurrent import AxipyProgressHandler, task_manager, ProgressSpecification, ProgressGuiFlags

supported_types = [
    Type.Line,
    Type.LineString,
    Type.Polygon,
    Type.GeometryCollection]
supported_types_string = ['Линия', 'Полилиния', 'Полигон', 'Коллекция']


def to_reverse_list(pnts: List[Pnt]) -> List[Pnt]:
    return [p for p in reversed(pnts)]


class RedirectLine:
    """ Изменение направления линии для линий, полилиний и полигонов"""

    def __init__(self, title: str, iface: AxiomaInterface) -> None:
        self.title = title
        self.iface = iface

    def is_suitable(self, geometry: Geometry):
        return geometry is not None and geometry.type in supported_types or issubclass(
            type(geometry), GeometryCollection)

    def __revert_line(self, line: Line) -> Line:
        return Line(line.end, line.begin, cs=line.coordsystem)

    def __revert_polyline(self, polyline: LineString) -> LineString:
        return LineString(
            to_reverse_list(
                polyline.points),
            cs=polyline.coordsystem)

    def __revert_polygon(self, polygon: Polygon) -> Polygon:
        points = to_reverse_list(polygon.points)
        result = Polygon(points, cs=polygon.coordsystem)
        if len(polygon.holes) == 0:
            return result
        for hole in polygon.holes:
            hole = to_reverse_list(hole)
            result.holes.append(hole)
        return result

    def __revert_impl(self, geom: Geometry) -> Geometry:
        if geom.type == Type.Line:
            return self.__revert_line(geom)
        elif geom.type == Type.LineString:
            return self.__revert_polyline(geom)
        elif geom.type == Type.Polygon:
            return self.__revert_polygon(geom)
        return None

    def __make_empty_typed_collection(
            self, collection: GeometryCollection) -> GeometryCollection:
        if collection.type == Type.MultiPolygon:
            return MultiPolygon(collection.coordsystem)
        if collection.type == Type.MultiLineString:
            return MultiLineString(collection.coordsystem)
        return GeometryCollection(collection.coordsystem)

    def __revert_geometry_collection(
            self, collection: GeometryCollection) -> GeometryCollection:
        result = self.__make_empty_typed_collection(collection)
        for geometry in collection:
            reverted_geometry = self.__revert_impl(geometry)
            if reverted_geometry is not None:
                result.append(reverted_geometry)
            else:
                # Если не можем поменять направление линии для какого-нибудь элемента
                # коллекции просто добавляем его. Т.к. в итоге мы должны иметь ту же
                # самую коллекцию только с изменёнными направлениями линий
                result.append(geometry)
        return result

    def revert(self, geom: Geometry) -> Geometry:
        if issubclass(type(geom), GeometryCollection):
            return self.__revert_geometry_collection(geom)
        else:
            return self.__revert_impl(geom)

    def __redirect(self, handler: AxipyProgressHandler):
        cursor = selection_manager.get_as_cursor()
        features = [f for f in cursor if f.has_geometry]  # type: List[Feature]
        handler.set_max_progress(len(features))
        updated_features = []
        for f in features:
            handler.raise_if_canceled()
            if not self.is_suitable(f.geometry):
                continue
            f.geometry = self.revert(f.geometry)
            if f.geometry is None:
                continue
            updated_features.append(f)
            handler.add_progress(1)
        editable_table = ensure_editable_table()
        handler.prepare_to_write_changes()
        editable_table.update(updated_features)
        self.__notify_about_finish_process(len(updated_features))

    def on_triggered(self):
        spec = ProgressSpecification(
            description=self.title,
            flags=ProgressGuiFlags.CANCELABLE)
        task_manager.run_and_get(spec, self.__redirect)

    def __notify_about_finish_process(self, feature_count: int):
        if feature_count == 0:
            self.iface.notifications.push(
                self.title, axipy.tr(
                    f'Не удалось обновить ни одну геометрию. Список поддерживаемых типов: '
                    f'{", ".join(supported_types_string)}.'), Notifications.Critical)
        else:
            self.iface.notifications.push(self.title, axipy.tr(
                f"Было обновлено геометрий: {feature_count} "),
                Notifications.Success)
