import os.path
from pathlib import Path
from typing import Any, Dict, List, NoReturn, Optional, Union, cast

from axipy._internal._decorator import _back_compat_params, _experimental
from axipy.cpp_core_dp import ShadowConverter, ShadowServiceOperation
from axipy.cs import CoordSystem
from axipy.da.data_object import Table

from ..schema import Schema
from .data_provider import DataProvider
from .provider_utils import OpenMode
from .source import Destination, Source

__all__: List[str] = [
    "PanoramaSource",
    "PanoramaDestination",
    "PanoramaDataProvider",
]


class PanoramaSource(Source):
    pass


class PanoramaDestination(Destination):
    pass


class PanoramaDataProvider(DataProvider):
    """
    Провайдер для источников ГИС Панорама.

    Note:
        Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.provider_manager.panorama`.

    :any:`export_from_acad_index`

    :any:`export_to_acad_index`
    """

    @staticmethod
    def _identifier() -> str:
        return "PanoramaDataProvider"

    def get_source(self, filename: Union[str, Path], alias: Optional[str] = None, empty_layers: bool = False) -> Source:
        """
        Создает источник данных.

        Args:
          filename: Имя файла.
          alias: Псевдоним для открываемого объекта.
          empty_layers: Показывать пустые слои в общем списке :meth:`ProviderManager.read_contents`
        """
        return PanoramaSource(
            Source._provider(self.id),
            Source._alias(alias),
            Source._table_file(filename),
            {"emptySublayers": empty_layers},
        )

    def open(self, filename: Union[str, Path], alias: Optional[str] = None, empty_layers: bool = False) -> Table:
        """
        Открывает объект данных.

        Args:
          filename: Имя файла. Если это файлы `*.sxf` или `*.txf`, то их необходимо предварительно сконвертировать, используя метод :meth:`convert_file`.
          alias: Псевдоним для открываемого объекта.
          empty_layers: Показывать пустые слои в общем списке`
        """
        filename_str = str(filename)
        if filename_str.lower().endswith((".sxf", ".txf")):
            raise ValueError("File format is not acceptable. Need to convert it with `convert_file`.")
        return cast(Table, self.get_source(filename_str, alias, empty_layers).open())

    @_back_compat_params({"out_filepath": "filepath"})
    def get_destination(
        self,
        filepath: Union[str, Path],
        classificator: str,
        schema: Schema,
        key_field: Optional[str] = None,
        single_object_type: Optional[str] = None,
        coordsystem: Optional[CoordSystem] = None,
        open_mode: OpenMode = OpenMode.Create,
    ) -> Destination:
        """
        Создает назначение объекта данных.

        Args:
            filepath: Путь к результирующему файлу `SIT` или `MAP`.
            schema: Схема таблицы.
            classificator: Путь к классификатору. При проведении конвертации копия классификатора так же сохраняется в папке с выходным файлом.
            key_field: Колонка-атрибут, содержащая ключи объектов, для файла с разнотипными объектами
            single_object_type: Символьный ключ объекта для файла, содержащего однотипные объекты
            coordsystem: Система координат, в которой необходимо получить результат. Если не указана, берется из схемы.
                Если СК не может быть преобразована в СК Панорамы, то вызывается исключение. Проверить предварительно можно
                воспользовавшись функцией :meth:`is_supported_coordsystem`
            open_mode: Режим открытия файла. В случае `OpenMode.Append` будет производится дополнение к существующему файлу.

        Note:
            Обязательно одно из двух полей `key_field` или `single_object_type`
        """
        pars: Dict[str, Any] = {"rscFilename": classificator}
        if coordsystem is not None:
            pars["prj"] = coordsystem.prj
        elif schema.coordsystem is not None:
            pars["prj"] = schema.coordsystem.prj
        if open_mode == OpenMode.Append:
            pars["append"] = True
        else:
            pars["create"] = True
        if key_field is None and single_object_type is None:
            raise ValueError("You must to set 'key_field' or 'single_object_type' parameters.")
        if key_field is not None:
            pars["objectTypeColumn"] = key_field
        if single_object_type is not None:
            pars["singleObjectType"] = single_object_type
        return PanoramaDestination(schema, Source._provider(self.id), Source._table_file(filepath), pars)

    def create_open(self) -> NoReturn:
        """
        Attention:
            Не поддерживается.

        Raises:
            NotImplementedError
        """
        raise NotImplementedError

    def convert_file(
        self, src_filepath: Union[str, Path], dest_filepath: Union[str, Path], classificator: Union[str, Path]
    ) -> None:
        """
        Производит конвертацию исходного файла в формате `SXF` в другой формат этого же
        провайдера (`MAP`).

        Args:
            src_filepath: Путь к исходному файлу `SXF` (имя файла).
            dest_filepath: Путь к выходному файлу (имя файла).
            classificator: Путь к классификатору


        .. code-block:: python

            input_file = 'Podolsk.sxf'
            output_file = 'Podolsk.map'
            classificator = 'Topo100t.rsc'
            provider_manager.panorama.convert_file(input_file, output_file, classificator)
        """
        src_filepath_str = str(src_filepath)
        dest_filepath_str = str(dest_filepath)
        classificator_str = str(classificator)

        if not os.path.exists(src_filepath_str):
            raise FileNotFoundError(f"Файл '{src_filepath_str}' не существует")
        convert_data: Dict[str, Any] = {
            "openWith": self.id,
            "src": src_filepath_str,
            "dest": dest_filepath_str,
            "rscFilename": classificator_str,
        }
        ShadowConverter.convertSxfFile(convert_data)

    @_experimental()
    def _convert_map_to_sxf(self, src_filepath: Union[str, Path], dest_filepath: Union[str, Path]) -> None:
        """Производит конвертацию исходного файла в формате `MAP`(`SIT`) в обменный формат этого же провайдера (`SXF`).

        Args:
            src_filepath: Путь к исходному файлу `MAP` (имя файла).
            dest_filepath: Путь к выходному файлу (имя файла).


        .. code-block:: python

            input_file = 'Podolsk.map'
            output_file = 'Podolsk.sxf'
            provider_manager.panorama._convert_map_to_sxf(input_file, output_file)

        """
        src_filepath_str = str(src_filepath)
        dest_filepath_str = str(dest_filepath)

        if not os.path.exists(src_filepath_str):
            raise FileNotFoundError(f"Файл '{src_filepath_str}' не существует")
        convert_data: Dict[str, Any] = {
            "openWith": self.id,
            "src": src_filepath_str,
            "dest": dest_filepath_str,
        }
        ShadowConverter.convertPanoramaMapToSxf(convert_data)

    @_experimental()
    def _convert_map_to_txf(self, src_filepath: Union[str, Path], dest_filepath: Union[str, Path]) -> None:
        """Производит конвертацию исходного файла в формате `MAP`(`SIT`) в обменный формат этого же провайдера (`TXF`).

        Args:
            src_filepath: Путь к исходному файлу `MAP` (имя файла).
            dest_filepath: Путь к выходному файлу (имя файла).


        .. code-block:: python

            input_file = 'Podolsk.map'
            output_file = 'Podolsk.txf'
            provider_manager.panorama._convert_map_to_txf(input_file, output_file)

        """
        src_filepath_str = str(src_filepath)
        dest_filepath_str = str(dest_filepath)

        if not os.path.exists(src_filepath_str):
            raise FileNotFoundError(f"Файл '{src_filepath_str}' не существует")
        convert_data: Dict[str, Any] = {
            "openWith": self.id,
            "src": src_filepath_str,
            "dest": dest_filepath_str,
        }
        ShadowConverter.convertPanoramaMapToTxf(convert_data)

    def is_supported_coordsystem(self, coordsystem: CoordSystem) -> bool:
        """
        Производится проверка, поддерживается ли СК провайдером.

        Args:
            coordsystem: Система Координат.

        Пример::

            coord_system = CoordSystem.from_prj('1,104')
            print(f'Supported: {provider_manager.panorama.is_supported_coordsystem(coord_system)}')
            coord_system = CoordSystem.from_prj('32, 1020, 7, 42.5, 49.5, 78.5, 30.28813972, 0, 0')
            print(f'Supported: {provider_manager.panorama.is_supported_coordsystem(coord_system)}')
            >>> Supported: False
            >>> Supported: True
        """
        pars: Dict[str, Any] = {"openWith": self.id, "prj": coordsystem.prj}
        return ShadowServiceOperation.panoramaSupportedCoordSystem(pars)

    @_experimental()
    def _get_cs_from_file(
        self, filepath: Union[str, Path], classificator: Optional[str] = None
    ) -> Optional[CoordSystem]:
        """
        Получение СК из файлов формата `MAP`,`SIT`,`SXF`,`TXF`

        Args:
            filepath: путь к файлу
            classificator: Путь к классификатору.
                Если классификатор находится в одном каталоге с исходным файлом (и является единственным), то параметр можно опустить.

        .. code-block:: python

            input_file = 'Podolsk.map'
            classificator = 'Topo100t.rsc'
            provider_manager.panorama._get_cs_from_file(input_file, classificator)
        """
        filepath_str = str(filepath)

        data: Dict[str, Any] = {"openWith": self.id, "src": str(filepath_str), "rscFilename": classificator}
        if not os.path.exists(filepath_str):
            raise FileNotFoundError(f"Файл '{filepath_str}' не существует")
        cs_str = ShadowServiceOperation.getCoordSystemString(data)
        if cs_str is not None:
            return CoordSystem.from_prj(cs_str)
        return None
