from enum import Enum, IntFlag
from functools import cached_property
from pathlib import Path
from types import TracebackType
from typing import (
    TYPE_CHECKING,
    Any,
    Callable,
    Dict,
    Iterable,
    Iterator,
    List,
    Optional,
    Tuple,
    Type,
    Union,
    cast,
    overload,
)

from axipy._internal._check_shadow import _is_valid
from axipy._internal._decorator import (
    _deprecated_by,
    _experimental,
    _experimental_class,
)
from axipy._internal._utils import _NoInitTypeError
from axipy.cpp_core_dp import AccessMode as ShadowAccessMode
from axipy.cpp_core_dp import RasterFacade as ShadowRasterFacade
from axipy.cpp_core_dp import (
    ShadowDatabaseLinkedTable,
    ShadowDatabaseTable,
    ShadowDataObject,
    ShadowQueryTable,
    ShadowRaster,
    ShadowRasteredTable,
    ShadowSelectionTable,
    ShadowTable,
    ShadowTabMetadata,
)
from axipy.cpp_render import ShadowCosmeticTable
from axipy.cs import CoordSystem
from axipy.utl import Rect
from PySide2.QtCore import QCoreApplication, QEventLoop, QMetaObject, QObject, Slot

from .feature import Feature, _FeatureIterator
from .geometry import Geometry
from .raster import GCP
from .schema import Schema

if TYPE_CHECKING:
    from PySide2.QtCore import QRectF, QSize, Signal
    from PySide2.QtGui import QTransform

__all__: List[str] = [
    "DataObject",
    "Table",
    "SupportedOperations",
    "Raster",
    "QueryTable",
    "SelectionTable",
    "_DatabaseTable",
    "_DatabaseLinkedTable",
    "CosmeticTable",
    "RasteredTable",
    "GeometryClass",
    "_TabMetaData",
]


@_experimental_class()
class _TabMetaData:
    """
    Метаданные файла MapInfo TAB.

    .. seealso:: :attr:`DataObject._metadata`

    .. literalinclude:: /../../tests/doc_examples/da/test_example_tabfile.py
        :caption: Пример инициализации метаданных
        :pyobject: test_run_example_metadata
        :start-after: # start
        :end-before: # finish
        :dedent: 4

    .. literalinclude:: /../../tests/doc_examples/da/test_example_tabfile.py
        :caption: Пример запроса метаданных у таблицы
        :pyobject: test_run_example_table_metadata
        :start-after: # start
        :end-before: # finish
        :dedent: 4

    """

    def __init__(self, obj: Optional["DataObject"] = None) -> None:
        """
        Args:
            obj: Объект данных. Если не указана, то создаются метаданные без привязки к конкретному источнику.
        """
        self._shadow = ShadowTabMetadata(obj._shadow if obj is not None else None)

    @staticmethod
    def known_keys() -> Dict[str, Any]:
        """
        Перечень известных ключей метаданных, обрабатываемых ГИС Аксиома. Возвращается в
        виде списка пар псевдоним-значение. Значение должно начинаться с '\\'. При
        установке или запроса значений можно использовать оба значения.

        Получение полного списка::

            for alias, value in _TabMetaData.known_keys().items():
                print(f'Alias="{alias}"; Value="{value}"')

            >>> ...
            >>> Alias="META_IS_READ_ONLY"; Value="\IsReadOnly"
            >>> Alias="META_PROJECTION_CLAUSE"; Value="\Spatial Reference\Geographic\Projection\Clause"
            >>> ...

        Т.е. данные строки идентичны::

            table.metadata['\\IsReadOnly']='TRUE'
            table.metadata['META_IS_READ_ONLY']='TRUE'

            table.metadata['\\Spatial Reference\\Geographic\\Projection\\Clause']='CoordSys Earth Projection 1, 0'
            table.metadata['META_PROJECTION_CLAUSE']='CoordSys Earth Projection 1, 0'
        """  # noqa: W605
        return ShadowTabMetadata.known_keys()

    @property
    def keys(self) -> List[str]:
        """
        Перечень установленных ключей метаданных объекта данных :class:`DataObject`

        Запрос перечня ключей установленных значений::

            for key in t.metadata.keys:
                print(f'Key="{key}"="{t._metadata[key]}"')

        Установка значения. Если по ключу значение уже присутствует, оно будет перезаписано::

            table.metadata['META_PROJECTION_CLAUSE']='CoordSys Earth Projection 1, 0'
        """
        return self._shadow.keys()

    def __setitem__(self, key: str, value: str) -> None:
        self._shadow.set_value(key, value)

    def __getitem__(self, key: str) -> str:
        return self._shadow.get_value(key)

    def __contains__(self, item: str) -> bool:
        if item in self.keys:
            return True
        for alias, value in _TabMetaData.known_keys().items():
            if alias == item:
                return True
        return False

    def __len__(self) -> int:
        return len(self.keys)


class DataObject:
    """
    Объект данных.

    Открываемые объекты из источников данных представляются объектами
    этого типа. Возможные реализации: таблица, растр, грид, чертеж,
    панорама, и так далее.

    Пример::

        table = provider_manager.openfile('path/to/file.tab')
        ...
        table.close()  # Закрывает таблицу

    Для закрытия объекта данных можно использовать менеджер контекста -
    выражение :any:`with<with>`. В таком случае таблица будет закрыта при выходе
    из блока. См. :meth:`close`.

    Пример::

        with provider_manager.openfile('path/to/file.tab') as raster:
            ...
            # При выходе из блока растр будет закрыт
    """

    @classmethod
    def __wrap_typed(cls, shadow: ShadowDataObject) -> "DataObject":
        obj = cls.__new__(cls)
        obj._shadow = shadow
        return obj

    @property
    def _shadow(self) -> ShadowDataObject:
        if self._inner_shadow is None:
            raise RuntimeError("Internal C++ object is already deleted.")
        return self._inner_shadow

    @_shadow.setter
    def _shadow(self, value: ShadowDataObject) -> None:
        self._inner_shadow = value

    @overload
    @classmethod
    def _wrap(cls, shadow: "ShadowQueryTable") -> Optional["QueryTable"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: "ShadowCosmeticTable") -> Optional["CosmeticTable"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: "ShadowSelectionTable") -> Optional["SelectionTable"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: "ShadowDatabaseLinkedTable") -> Optional["_DatabaseLinkedTable"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: "ShadowDatabaseTable") -> Optional["_DatabaseTable"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: "ShadowTable") -> Optional["Table"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: "ShadowRaster") -> Optional["Raster"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: "ShadowRasteredTable") -> Optional["RasteredTable"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: "ShadowDataObject") -> Optional["DataObject"]: ...

    @classmethod
    def _wrap(cls, shadow: "ShadowDataObject") -> Optional["DataObject"]:
        if shadow is None:
            return None

        # Порядок приведения важен!
        obj_type: Type[DataObject]
        if isinstance(shadow, ShadowQueryTable):
            obj_type = QueryTable
        elif isinstance(shadow, ShadowCosmeticTable):
            obj_type = CosmeticTable
        elif isinstance(shadow, ShadowSelectionTable):
            obj_type = SelectionTable
        elif isinstance(shadow, ShadowDatabaseLinkedTable):
            obj_type = _DatabaseLinkedTable
        elif isinstance(shadow, ShadowDatabaseTable):
            obj_type = _DatabaseTable
        elif isinstance(shadow, ShadowTable):
            obj_type = Table
        elif isinstance(shadow, ShadowRaster):
            obj_type = Raster
        elif isinstance(shadow, ShadowRasteredTable):
            obj_type = RasteredTable
        elif isinstance(shadow, ShadowDataObject):
            obj_type = DataObject
        else:
            return None
        return obj_type.__wrap_typed(shadow)

    def __init__(self) -> None:
        raise _NoInitTypeError

    @property
    def name(self) -> str:
        """Название объекта данных."""
        return self._shadow.name()

    @name.setter
    def name(self, name: str) -> None:
        self._shadow.setName(name)

    @property
    def provider(self) -> str:
        """Провайдер изначального источника данных."""
        return self._shadow.provider()

    @property
    def is_hidden(self) -> bool:
        return self._shadow.isHidden()

    @property
    def is_spatial(self) -> bool:
        """Признак того, что объект данных является пространственным."""
        return self._shadow.isSpatial()

    @property
    def properties(self) -> dict:
        """Дополнительные свойства объекта данных."""
        return self._shadow.properties()

    @property
    def file_name(self) -> Optional[Path]:
        """Возвращает путь к файлу таблицы."""
        file_name: Optional[str] = self.properties.get("fileName", None)
        return Path(file_name) if file_name else None

    def close(self) -> None:
        """
        Пытается закрыть таблицу.

        Raises:
            RuntimeError: Ошибка закрытия таблицы.

        Note:

            Объект данных не всегда может быть сразу закрыт. Например, для
            таблиц используется транзакционная модель редактирования и перед
            закрытием необходимо сохранить или отменить изменения, если они
            есть.
            См. :attr:`Table.is_modified`.
        """
        if isinstance(self._shadow, ShadowCosmeticTable):
            raise RuntimeError("It's not allowed to close cosmetic table.")

        if (
            isinstance(self, Table)
            and not isinstance(self, QueryTable)
            and isinstance(self._shadow, ShadowTable)
            and self._shadow.isEditable()
            and self.is_temporary
            and self.is_editable
        ):
            self.rollback()

        self._shadow.closeRequest()
        self._shadow = None  # type: ignore[assignment]
        QCoreApplication.processEvents()

    @property
    def destroyed(self) -> "Signal":
        """Сигнал оповещения об удалении объекта."""
        return self._shadow.innerDestroyed

    def __eq__(self, other: object) -> bool:
        if isinstance(other, DataObject):
            return self._shadow.isEqual(other._shadow)
        return NotImplemented

    def __enter__(self) -> "DataObject":
        return self

    def __exit__(
        self,
        exception_type: Optional[Type[BaseException]],
        exception_value: Optional[BaseException],
        traceback: Optional[TracebackType],
    ) -> Optional[bool]:
        """Закрывает таблицу при выходе из блока кода :obj:`with`."""
        self.close()
        return None

    def __repr__(self) -> str:
        if self._shadow is None:
            return f"<axipy.{self.__class__.__name__} name=None, provider=None>"
        else:
            return f"<axipy.{self.__class__.__name__} name={self.name}, provider={self.provider}>"

    @cached_property
    def __metadata(self) -> _TabMetaData:
        return _TabMetaData(self)

    @property
    @_experimental()
    def _metadata(self) -> _TabMetaData:
        """
        Метаданные в формате TAB MapInfo.

        .. seealso:: Примеры использования см  :class:`_TabMetaData`

        .. seealso:: Для сохранения метаданных в TAB необходимо использовать :meth:`TabFile.generate_tab`

        .. seealso:: Примеры использования см  :meth:`Destination.export`
        """
        return self.__metadata

    @property
    def is_valid(self) -> bool:
        """
        Возвращает признак валидности объекта.
        """
        return _is_valid(self._shadow)


class SupportedOperations(IntFlag):
    """
    Флаги доступных операций.

    Реализованы как перечисление :class:`enum.IntFlag` и поддерживают побитовые
    операции.

    .. csv-table::
        :header: Атрибут, Значение

        `Empty`, 0
        `Insert`, 1
        `Update`, 2
        `Delete`, 4
        `Write`, `Insert` | `Update` | `Delete` = 7
        `Read`, 8
        `ReadWrite`, `Read` | `Write` = 15

    .. literalinclude:: /../../tests/doc_examples/da/test_example_supported_operations.py
        :caption: Пример использования
        :pyobject: example_supported_operations
        :lines: 2-
        :dedent: 4

    See Also:
        :class:`enum.IntFlag`, :attr:`axipy.Table.supported_operations`
    """

    Empty = 0
    Insert = 1
    Update = 2
    Delete = 4
    Write = 7
    Read = 8
    ReadWrite = 15


class GeometryClass(int, Enum):
    """Класс геометрических объектов :attr:`Table.classGeometries`."""

    Unknown = 0
    """Тип неопределен."""
    Points = 1
    """Точечные объекты."""
    Lines = 2
    """Линейные объекты."""
    Polygons = 3
    """Площадные объекты."""
    Texts = 4
    """Текстовые объекты."""


class Table(DataObject):
    """
    Таблица.

    Менеджер контекста сохраняет изменения и закрывает таблицу.

    Пример::

        with provider_manager.openfile('path/to/file.tab') as table:
            ...
            # При выходе из блока таблица будет сохранена и закрыта

    See also:
        :meth:`commit`, :meth:`DataObject.close`.
    """

    if TYPE_CHECKING:
        _shadow: ShadowTable

    @property
    @_experimental()
    def _id(self) -> int:
        """
        Уникальный идентификатор в рамках сессии.

        See also:

            Подробнее об использовании см. :meth:`axipy.DataManager._find_by_id`.
        """
        return self._shadow.id()

    @property
    def data_changed(self) -> "Signal":
        """
        Сигнал об изменении данных таблицы. Испускается когда были изменены данные
        таблицы. В качестве параметра передается дополнительная информация в виде dict.

        .. csv-table:: Доступные параметры
            :header: Наименование, Значение, Описание

            operation, insert, Произведена вставка данных
            operation, update, Данные были обновлены
            operation, remove, Данные были удалены
            ids, list , "Перечень затронутых идентификаторов, если он доступен.
                Если идентификаторы имеют отрицательные значения, то они относятся к новым несохраненным записям."

        .. seealso::  :attr:`DataManager.table_data_changed`

        .. literalinclude:: /../../tests/doc_examples/da/test_example_table.py
            :caption: Пример подписки на изменение таблицы.
            :pyobject: test_run_example_table_change_table
            :lines: 3-6
            :dedent: 4
        """
        return self._shadow.dataChanged

    @property
    def schema_changed(self) -> "Signal":
        """
        Сигнал об изменении схемы таблицы.

        Испускается когда была изменена структура таблицы.
        """
        return self._shadow.schemaChanged

    @property
    def is_editable(self) -> bool:
        """Признак того, что таблица является редактируемой."""
        return self._shadow.accessMode() == ShadowAccessMode.Access_ReadWrite

    @property
    def is_temporary(self) -> bool:
        """
        Признак того, что таблица является временной.

        Это в первую очередь касается таблиц, созданных в памяти. А также таблица
        косметического слоя.
        """
        return self._shadow.isTemporary()

    @property
    def is_modified(self) -> bool:
        """Таблица содержит несохраненные изменения."""
        return (
            self._shadow.isEditable() and cast(ShadowTable, self._shadow).hasModified() if self.is_editable else False
        )

    @property
    def supported_operations(self) -> SupportedOperations:
        """
        Доступные операции.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_supported_operations.py
            :caption: Пример использования
            :pyobject: example_supported_operations
            :lines: 2-
            :dedent: 4
        """
        return SupportedOperations(self._shadow.supportedOperations())

    @property
    def schema(self) -> Schema:
        """
        Схема (структура) таблицы.

        Если требуется изменить схему существующей таблицы, необходимо учесть следующие моменты:

            * Изменение производится напрямую в таблице. Необходимо позаботиться о создании резервной копии.
            * Перед изменением нужно сохранить все данные. В противном случае при попытке изменения таблицы
              с несохраненными данными будет вызвано исключение.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_table.py
            :caption: Пример изменения существующей схемы таблицы
            :pyobject: test_run_example_table_schema
            :lines: 3-
            :dedent: 4

        .. seealso:: :class:`Schema`
        """
        return Schema._from_dict(self._shadow.schema())

    @schema.setter
    def schema(self, new_schema: Schema) -> None:
        self.ensure_editable()
        cast(ShadowTable, self._shadow).setSchema(new_schema._to_dict())

    @property
    def coordsystem(self) -> CoordSystem:
        """Система координат таблицы."""
        return cast(CoordSystem, CoordSystem._wrap(ShadowTable.coordSystem(self._shadow)))

    def ensure_editable(self) -> None:
        if not self.is_editable:
            raise RuntimeError("Table is not editable")

    def insert(self, features: Union[Feature, Iterable[Feature]]) -> None:
        """
        Вставляет записи в таблицу.

        Args:
            features: Записи для вставки.
        """
        self.ensure_editable()
        if isinstance(features, Feature):
            features = [features]
        cast(ShadowTable, self._shadow).insert(f._shadow for f in features)
        QCoreApplication.processEvents()

    def update(self, features: Union[Feature, Iterable[Feature]]) -> None:
        """
        Обновляет записи в таблице.

        Args:
            features: Записи для обновления.

        При обновлении проверяется :attr:`Feature.id`. Если запись с таким
        идентификатором не найдена, то она пропускается.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_table.py
            :caption: Пример использования
            :pyobject: example_table_update
            :lines: 2-
            :dedent: 4

        See also:
            :attr:`Feature.id`, :meth:`commit`, :attr:`is_modified`.
        """
        self.ensure_editable()
        if isinstance(features, Feature):
            features = [features]
        cast(ShadowTable, self._shadow).update(f._shadow for f in features)
        QCoreApplication.processEvents()

    def remove(self, ids: Union[int, List[int]]) -> None:
        """
        Удаляет записи из таблицы.

        Args:
            ids: Идентификаторы записей для удаления.
        """
        self.ensure_editable()
        if isinstance(ids, int):
            ids = [ids]
        cast(ShadowTable, self._shadow).remove(ids)
        QCoreApplication.processEvents()

    def commit(self) -> None:
        """
        Сохраняет изменения в таблице.

        Если таблица не содержит несохраненные изменения, то команда
        игнорируется.

        Raises:
            RuntimeError: Невозможно сохранить изменения.
        """
        self.ensure_editable()
        if self.is_temporary:
            raise RuntimeError("Невозможно сохранить изменения во временной таблице.")
        cast(ShadowTable, self._shadow).commit()
        QCoreApplication.processEvents()

    def rollback(self) -> None:
        """
        Отменяет несохраненные изменения в таблице.

        Если таблица не содержит несохраненные изменения, то команда игнорируется.
        """
        self.ensure_editable()
        cast(ShadowTable, self._shadow).rollback()
        QCoreApplication.processEvents()

    @property
    def can_undo(self) -> bool:
        """Возможен ли откат на один шаг назад."""
        return self._is_transaction_table and cast(ShadowTable, self._shadow).canUndo(1)

    def undo(self, steps: int = 1) -> None:
        """
        Производит откат назад на заданное количество шагов. При этом возвращается
        состояние до последней отмены.

        Args:
            steps: Количество шагов.
        """
        if self.can_undo:
            cast(ShadowTable, self._shadow).undo(steps)

    @property
    def can_redo(self) -> bool:
        """Возможен ли откат на один шаг вперед."""
        return self._is_transaction_table and cast(ShadowTable, self._shadow).canRedo(1)

    def redo(self, steps: int = 1) -> None:
        """
        Производит откат вперед на заданное количество шагов. При этом возвращается
        состояние до последней отмены.

        Args:
            steps: Количество шагов.
        """
        if self.can_redo:
            cast(ShadowTable, self._shadow).redo(steps)

    # noinspection PyPep8Naming
    def itemsByIds(self, ids: List[int]) -> Iterator[Feature]:
        """
        Запрашивает записи по списку :class:`list` с идентификаторами записей, либо
        перечень идентификаторов в виде списка. Идентификаторы несохраненных записей
        имеют отрицательные значения.

        Args:
            ids: Список идентификаторов.

        Returns:
            Итератор по записям.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_table.py
            :caption: Пример
            :pyobject: test_run_example_table_query_ids
            :lines: 3-
            :dedent: 4
        """
        return _FeatureIterator(self._shadow.itemsById(ids))

    # noinspection PyPep8Naming
    def itemsInObject(self, obj: Geometry) -> Iterator[Feature]:
        """
        Запрашивает записи с фильтром по геометрическому объекту.

        Args:
            obj: Геометрия. Если для нее не задана СК, используется СК таблицы.

        Returns:
            Итератор по записям.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_table.py
            :caption: Пример запроса по полигону
            :pyobject: test_run_example_table_query_obj
            :lines: 3-
            :dedent: 4
        """
        geom = obj
        if isinstance(geom, Geometry) and geom.coordsystem is None:
            geom = obj.clone()
            geom.coordsystem = self.coordsystem
        return _FeatureIterator(self._shadow.itemsInObject(geom._shadow))

    # noinspection PyPep8Naming
    def itemsInRect(self, bbox: Union[Rect, "QRectF", tuple]) -> Iterator[Feature]:
        """
        Запрашивает записи с фильтром по ограничивающему прямоугольнику.

        Args:
            bbox: Ограничивающий прямоугольник.

        Returns:
            Итератор по записям.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_table.py
            :caption: Пример запроса (таблица в проекции `Робинсона`)
            :pyobject: test_run_example_table_query_mbr
            :lines: 3-
            :dedent: 4
        """

        return _FeatureIterator(self._shadow.itemsInMbr(Rect._rect_value_to_qt(bbox)))

    @property
    def hotlink(self) -> str:
        """
        Наименование атрибута таблицы для хранения гиперссылки.

        .. csv-table:: Возможны следующие варианты
            :header: Значение, Описание

            axioma://world.tab, Открывает файл или рабочее пространство в аксиоме
            addlayer://world, Добавляет слой world в текущую карту
            exec://gimp,  Запускает на выполнение программу gimp
            https://axioma-gis.ru/, Открывает ссылку в браузере

        Если префикс отсутствует, то производится попытка запустить по ассоциации.

        See also:
            :attr:`axipy.VectorLayer.hotlink`.
        """

        return self._shadow.get_hotlink()

    @hotlink.setter
    def hotlink(self, v: str) -> None:
        self._shadow.set_hotlink(v)

    def items(
        self,
        bbox: Optional[Union[Rect, "QRectF", tuple]] = None,
        ids: Optional[List[int]] = None,
    ) -> Iterator[Feature]:
        """
        Запрашивает записи, удовлетворяющие параметрам. В качестве фильтра может быть
        указан либо ограничивающий прямоугольник, либо перечень идентификаторов в виде
        списка.

        Args:
            bbox: Ограничивающий прямоугольник.
            ids: Список идентификаторов.

        Returns:
            Итератор по записям.
        """
        if bbox is not None:
            if not self.is_spatial:
                raise RuntimeError("Table is not spatial")
            return self.itemsInRect(bbox)
        if ids is not None:
            return self.itemsByIds(ids)
        return _FeatureIterator(self._shadow.selectAll())

    def count(self, bbox: Optional[Union[Rect, "QRectF", tuple]] = None) -> int:
        """
        Возвращает количество записей, удовлетворяющих параметрам.

        Данный метод является наиболее предпочтительным для оценки
        количества записей. При этом используется наиболее оптимальный
        вариант выполнения запроса для каждого конкретного провайдера
        данных.

        Args:
            bbox: Ограничивающий прямоугольник.

        Returns:
            Количество записей.
        """
        if bbox is not None:
            if not self.is_spatial:
                raise RuntimeError("Table is not spatial")
            return self._shadow.countInMbr(Rect._rect_value_to_qt(bbox))
        return self._shadow.count()

    def get_bounds(self) -> Rect:
        """Возвращает область, в которую попадают все данные таблицы."""
        return Rect.from_qt(self._shadow.calculateMbr())

    def class_geometries(self) -> List[GeometryClass]:
        """Возвращает перечень геометрических типов :attr:`GeometryClass`, содержащихся
        в таблице."""
        return [GeometryClass(t) for t in self._shadow.contentTypes()]

    def __len__(self) -> int:
        return self.count()

    def __iter__(self) -> Iterator[Feature]:
        return self.items()

    def __exit__(
        self,
        exception_type: Optional[Type[BaseException]],
        exception_value: Optional[BaseException],
        traceback: Optional[TracebackType],
    ) -> Optional[bool]:
        self._commit_if_possible()
        return super().__exit__(exception_type, exception_value, traceback)

    @property
    def _is_transaction_table(self) -> bool:
        return isinstance(self._shadow, ShadowTable) and self._shadow.isEditable()

    def _commit_if_possible(self) -> None:
        if not self.is_editable or self.is_temporary:
            return None
        self.commit()

    class _WorkerWithIds(QObject):

        def __init__(self) -> None:
            super().__init__()
            self.__marker: object = object()
            self.__result: Union[object, Tuple[int, ...]] = self.__marker

        @cached_property
        def __loop(self) -> QEventLoop:
            return QEventLoop()

        # noinspection PyTypeChecker
        def run(self, table: "Table", func: Callable, *args: Any) -> Tuple[int, ...]:
            table.data_changed.connect(self.slot_data_changed)
            try:
                func(*args)
                if self.__result is self.__marker:
                    self.__loop.exec_()
            finally:
                table.data_changed.disconnect(self.slot_data_changed)

            return self.__result  # type: ignore[return-value]

        @Slot(dict)
        def slot_data_changed(self, params: dict) -> None:
            self.__result = params["ids"]
            QMetaObject.invokeMethod(self.__loop, "quit")

    def _insert_with_ids(self, features: Union[Feature, Iterable[Feature]]) -> Tuple[int, ...]:
        return self._WorkerWithIds().run(self, self.insert, features)

    def _remove_with_ids(self, features: Union[Feature, Iterable[Feature]]) -> Tuple[int, ...]:
        return self._WorkerWithIds().run(self, self.remove, features)

    def _update_with_ids(self, features: Union[Feature, Iterable[Feature]]) -> Tuple[int, ...]:
        return self._WorkerWithIds().run(self, self.update, features)


class SelectionTable(Table):
    """
    Таблица, построенная на основе текущей выборки. Носит временный характер. Создается,
    когда в выборку добавляются объекты и удаляется, когда выборка очищается.

    Пример доступа к выборке с получением идентификаторов::

        for f in data_manager.selection.items():
            print('>>>', f.id)
    """

    if TYPE_CHECKING:
        _shadow: ShadowSelectionTable

    @property
    def base_table(self) -> Table:
        """Таблица, на которой основана данная выборка."""
        # Guaranteed to get ShadowTable (Guaranteed not None)
        return cast(Table, DataObject._wrap(cast(ShadowTable, self._shadow.originalObject())))


@_experimental_class()
class _DatabaseTable(Table):
    """
    Таблица БД
    """

    if TYPE_CHECKING:
        _shadow: ShadowDatabaseTable

    @property
    def sql(self) -> str:
        """
        Текст SQL предложения
        """
        return self._shadow.sqlText()


@_experimental_class()
class _DatabaseLinkedTable(_DatabaseTable):
    """
    Связанная таблица БД
    """

    if TYPE_CHECKING:
        _shadow: ShadowDatabaseLinkedTable

    def refresh(self) -> None:
        """
        Производит обновление с сервера связанной таблицы
        """
        self._shadow.refresh()


class QueryTable(Table):
    """
    Таблица, построенная на основе SQL запроса.

    Пример::

        table = provider_manager.openfile('world.tab')
        query = data_manager.query('select * from world')
    """

    if TYPE_CHECKING:
        _shadow: ShadowQueryTable

    @property
    def sql_text(self) -> str:
        """Текст SQL запроса."""
        return self._shadow.sqlText()


class CosmeticTable(Table):
    """Объект данных косметического слоя :class:`axipy.CosmeticLayer`"""

    if TYPE_CHECKING:
        _shadow: ShadowCosmeticTable

    pass


class Raster(DataObject):
    """Растровый объект."""

    if TYPE_CHECKING:
        _shadow: ShadowRaster

    @property
    def coordsystem(self) -> Union[CoordSystem, None]:
        """Система координат растра."""
        return CoordSystem._wrap(ShadowRaster.coordSystem(self._shadow))

    @property
    def size(self) -> "QSize":
        """Размер растра в пикселях."""
        return self._shadow.size()

    @property
    def device_to_scene_transform(self) -> "QTransform":
        """Матрица преобразования из точек на изображении (пиксели) в точки на карте."""
        return ShadowRasterFacade.getTransform(self._shadow)

    @property
    def scene_to_device_transform(self) -> "QTransform":
        """Матрица преобразования из точек на карте в точки на изображении (пиксели)."""
        return self.device_to_scene_transform.inverted()[0]

    def get_gcps(self) -> List[GCP]:
        """Возвращает точки привязки."""
        cpp_result = ShadowRasterFacade.getGCPs(self._shadow)
        return list(map(GCP._wrap, cpp_result))


class RasteredTable(DataObject):
    """Данные из таких источников, как ГИС Панорама и AutoCAD."""

    if TYPE_CHECKING:
        _shadow: ShadowRasteredTable

    @property
    def coordsystem(self) -> Optional[CoordSystem]:
        """Система координат."""
        return CoordSystem._wrap(ShadowRasteredTable.coordSystem(self._shadow))

    @property
    def schema(self) -> Schema:
        """Схема таблицы."""
        return Schema._from_dict(self._shadow.schema())

    @property
    def layers(self) -> List[str]:
        """Список доступных для запроса данных слоев."""
        return self._shadow.layerList()

    def items(self, layer_name: Optional[str] = None) -> Iterator[Feature]:
        """
        Запрашивает записи из источника.

        Args:
            layer_name: Наименование слоя, по которому будет произведена выборка.
                Если значение пустое, выдаются данные по всем слоям.

        Returns:
            Итератор по записям.
        """
        if layer_name is None or layer_name == "":
            return _FeatureIterator(self._shadow.selectAll())
        return _FeatureIterator(self._shadow.selectForLayer(layer_name))


def _apply_deprecated() -> None:
    @_deprecated_by("rollback")
    def restore(self: Table) -> None:
        self.rollback()

    setattr(Table, "restore", restore)


_apply_deprecated()
