from __future__ import annotations

from collections.abc import Iterator, Iterable
from functools import cached_property
from pathlib import Path
from typing import TYPE_CHECKING
from typing import cast, Generator, Union

import axipy

from .observer import SpatialFilterObserver
from .utils import _init_geom_generator, SUPPORTED_TYPES, _init_feature_geom_generator, NotifyResultsMixIn, run_in_gui

if TYPE_CHECKING:
    from .__init__ import SpatialFilter

POLYGON_LIKE = Union[axipy.Polygon, axipy.MultiPolygon]


class Worker(axipy.DialogTask, NotifyResultsMixIn):

    def __init__(
            self,
            plugin: 'SpatialFilter',
            save_folder: Path,
    ) -> None:
        super().__init__(self.do_work, cancelable=True)

        self._plugin: 'SpatialFilter' = plugin

        self._save_folder: Path = save_folder
        self._sum_count: int = 0

    @cached_property
    def polygon_like_filter(self) -> POLYGON_LIKE:
        return self._get_selection_polygon()

    def do_work(self) -> None:
        self.title = self._plugin.title
        self.message = self.title

        tables: list[axipy.Table] = list(SpatialFilterObserver.get_tables_from_active_map_view())
        all_count: int = len(tables)

        self._sum_count = sum(table.count() for table in tables)

        file_names: list[Path | None] = list(
            self.export_one_table(table)
            for table in tables
        )
        file_names: list[Path] = list(filter(None, file_names))
        successful_count: int = len(file_names)

        if len(file_names) > 0:
            self.create_map_from_result_tables(file_names)

        self.notify_skipped(self._plugin.title)
        self.notify_exceptions(self._plugin.title)
        self.notify_results(self._plugin.title, successful_count, all_count)

    def _get_selection_polygon(self) -> POLYGON_LIKE:
        """Объединение выбранных полигонов в один полигон."""
        selection_table = SpatialFilterObserver.selection_table()

        polygon: POLYGON_LIKE | None = None  # Результирующий полигон

        g_gen = _init_geom_generator(selection_table.items(), selection_table.count(), self)
        g_gen_filtered: Iterator[POLYGON_LIKE] = filter(
            lambda g: isinstance(g, SUPPORTED_TYPES), g_gen)

        for g_filtered in g_gen_filtered:
            if polygon is None:
                polygon = g_filtered
            else:
                polygon = cast(POLYGON_LIKE, polygon.union(g_filtered))

        # At least 1 polygon-like is ensured by observer function 'check_if_only_supported_types_selected', so no None.
        return polygon

    def processed_features_generator(
            self,
            feature_geom_generator: Generator[tuple[axipy.Feature, axipy.Geometry]]
    ) -> Generator[axipy.Feature]:
        polygon = self.polygon_like_filter
        for f, g in feature_geom_generator:
            geometry = g.intersection(polygon)  # Обрезаем геометрию
            if geometry is not None:  # Если в результате обрезки геометрия не пустая
                f.geometry = geometry  # Назначаем новой строке обрезанную геометрию
                yield f

    def export_one_table(
            self,
            table: axipy.Table
    ) -> Path | None:
        try:
            file_name = self._export_one_table_impl(table)
        except Exception as e:
            print(f"Ошибка конвертации таблицы '{table.name}': {e}")
            self.check_exception(e)
            return None
        else:
            return file_name

    def _export_one_table_impl(
            self,
            table: axipy.Table
    ) -> Path | None:
        file_name = (self._save_folder / table.name).with_suffix(".tab")
        if self.check_skipped(file_name):
            return None
        tab_dest = axipy.provider_manager.tab.get_destination(str(file_name), table.schema)
        gen = _init_feature_geom_generator(table.items(), self._sum_count, self)
        items = self.processed_features_generator(gen)

        tab_dest.export(items)
        return file_name

    @run_in_gui
    def create_map_from_result_tables(self, file_names: Iterable[Path]) -> None:
        result_layers = (axipy.Layer.create(axipy.provider_manager.openfile(file_name)) for file_name in file_names)
        # Открытие полученных таблиц на карте
        m = axipy.Map(result_layers)  # todo: fix empty map creation on axipy side?
        axipy.view_manager.create_mapview(m)
