from __future__ import annotations

from dataclasses import dataclass
from typing import TYPE_CHECKING

import axipy
from PySide2.QtCore import Qt, Slot
from PySide2.QtWidgets import QDialog, QPushButton
from .ui import Ui_ConnectionDialog

if TYPE_CHECKING:
    from .__init__ import OpenPostgreRasterPlugin


@dataclass
class ConnectionParams:
    host: str
    port: str
    dbname: str
    user: str
    password: str

    def is_complete(self) -> bool:
        return all((
            self.host,
            self.port,
            self.dbname,
            self.user,
            self.password,
        ))

    def construct_connection_string(self) -> str:
        if not self.is_complete():
            raise RuntimeError

        return (
            f"PG:host={self.host} "
            f"port={self.port} "
            f"dbname='{self.dbname}' "
            f"user='{self.user}' "
            f"password='{self.password}'"
        )

    def construct_connection_string_no_password(self) -> str:
        if not self.is_complete():
            raise RuntimeError

        return (
            f"PG:host={self.host} "
            f"port={self.port} "
            f"dbname='{self.dbname}' "
            f"user='{self.user}'"
        )


class ConnectionDialog(QDialog, Ui_ConnectionDialog):
    @dataclass
    class Result:
        connection_string: str | None = None
        connection_string_no_password: str | None = None
        password: str | None = None

    def __init__(self, plugin: 'OpenPostgreRasterPlugin', result: Result) -> None:
        super().__init__(axipy.view_manager.global_parent)
        self.setupUi(self)
        self.setAttribute(Qt.WA_DeleteOnClose, True)

        self.plugin = plugin
        self._result = result

        self.le_host.textEdited.connect(self.slot_check_ok_enabled)
        self.le_port.textEdited.connect(self.slot_check_ok_enabled)
        self.le_dbname.textEdited.connect(self.slot_check_ok_enabled)
        self.le_user.textEdited.connect(self.slot_check_ok_enabled)
        self.le_password.textEdited.connect(self.slot_check_ok_enabled)

        self._ok_button.setText(self.plugin.tr("Соединиться"))
        self._ok_button.setEnabled(False)

        self.__try_restore_connection_params()

    def __try_restore_connection_params(self) -> None:
        connection_params: ConnectionParams | None = self.plugin.restore_connection_params()
        if connection_params is None:
            return None

        self.le_host.setText(connection_params.host)
        self.le_port.setText(connection_params.port)
        self.le_dbname.setText(connection_params.dbname)
        self.le_user.setText(connection_params.user)
        self.le_password.setText(connection_params.password)

        self.slot_check_ok_enabled()

    def __read_connection_params(self) -> ConnectionParams:
        host = self.le_host.text()
        port = self.le_port.text()
        dbname = self.le_dbname.text()
        user = self.le_user.text()
        password = self.le_password.text()
        return ConnectionParams(host, port, dbname, user, password)

    @Slot()
    def slot_check_ok_enabled(self) -> None:
        enabled = self.__read_connection_params().is_complete()
        self._ok_button.setEnabled(enabled)

    @property
    def _ok_button(self) -> QPushButton:
        return self.buttonBox.button(self.buttonBox.Ok)

    def done(self, return_code: int) -> None:
        if return_code == self.Accepted:
            connection_params = self.__read_connection_params()
            connection_string = connection_params.construct_connection_string()
            connection_string_no_password = connection_params.construct_connection_string_no_password()
            self._result.connection_string = connection_string
            self._result.connection_string_no_password = connection_string_no_password
            self._result.password = connection_params.password
            self.plugin.save_connection_params(connection_params)

        super().done(return_code)
