from itertools import chain
from typing import Any, Generator, Tuple

import axipy
from axipy import (
    GEOMETRY_ATTR,
    CoordTransformer,
    GeometryType,
    Point,
    VectorLayer,
    view_manager,
)
from axipy.gui.map_tool import ViewTool
from PySide2.QtCore import Qt
from PySide2.QtGui import QMouseEvent
from PySide2.QtWidgets import QInputDialog, QMessageBox


# Переопределенный класс инструмента для карты
class MyMapTool(ViewTool):

    def __init__(self, caption: str = "") -> None:
        super().__init__()
        self.caption: str = caption
        self.attr_index: int = 0

    def on_active_changed(self) -> None:
        if self.is_active:
            print(f"Активирован инструмент для окна: {self.view.title}")
        else:
            print(f"Деактивирован инструмент для окна: {self.view.title}")

    def load(self) -> None:
        self.attr_index = 0

    def mousePressEvent(self, event: QMouseEvent) -> bool:
        # Левая кнопка мыши — вывод значений
        if event.button() == Qt.LeftButton:
            self._show_map_info(event)
        # Правая кнопка мыши - выбор атрибута карты
        elif event.button() == Qt.RightButton:
            self._select_attribute_index()
        # Средняя кнопка мыши переключает режим скроллинга при выходе за пределы карты
        elif event.button() == Qt.MiddleButton:
            self._is_scroll_mode = not self._is_scroll_mode
        return self.IGNORE

    def mouseReleaseEvent(self, event: QMouseEvent) -> bool:
        """Blocks right click context menu."""
        return self.ACCEPT

    def _show_map_info(self, event: QMouseEvent) -> None:
        scene_box = self.get_select_rect(event.pos())
        tables = self.get_tables()
        feature_iters = map(lambda table: self.get_features(table, scene_box), tables)
        features = chain.from_iterable(feature_iters)
        # Вывод в формате таблица:значение по всем попавшим в выбор слоям
        text = "\n".join(tab.name + ": " + str(feature[self.attr_index]) for tab, feature in features)
        if text:
            QMessageBox.information(view_manager.global_parent, "Информация", text)

    def _select_attribute_index(self) -> None:
        i, ok = QInputDialog.getInt(
            axipy.mainwindow.widget,
            "Параметры",
            self.caption,
            self.attr_index,
            minValue=0,
        )
        if ok:
            self.attr_index = i

    def get_tables(self) -> Generator[axipy.Table, Any, None]:
        for layer in self.view.map.layers:
            if not isinstance(layer, VectorLayer):
                continue
            yield layer.data_object

    def get_features(self, table: axipy.Table, scene_box) -> Generator[Tuple[axipy.Table, axipy.Feature], Any, None]:
        if table.coordsystem != self.view.coordsystem:
            transformer = CoordTransformer(self.view.coordsystem, table.coordsystem)
            scene_box = transformer.transform(scene_box)
        center = scene_box.center
        scene_point = Point(center.x, center.y, table.coordsystem)
        for feature in table.items(bbox=scene_box):
            geometry = feature[GEOMETRY_ATTR]
            if geometry.type == GeometryType.Point or geometry.contains(scene_point):
                yield table, feature


# Переопределенный класс инструмента для отчета
class MyReportTool(ViewTool):

    def mousePressEvent(self, event: QMouseEvent) -> bool:
        # Левая кнопка мыши - вывод значений
        if event.button() == Qt.LeftButton:
            self._show_report_info(event)
        return self.IGNORE

    def _show_report_info(self, event: QMouseEvent) -> None:
        scene_box = self.get_select_rect(event.pos())
        for item in self.view.report.items:
            if item.rect.intersects(scene_box):
                QMessageBox.information(view_manager.global_parent, "Информация", item._title)
                break
