import time
from pathlib import Path
from typing import Optional

from axipy import sql
from PySide2.QtSql import QSqlQuery

try:
    import sqlite3
except ImportError:
    sqlite3 = None

from .helper import print_, print_exc_

sqlite_db_name = "cad_tools"


def query_exec(query_text: str) -> None:
    print_("query:")
    print_(query_text[:1000])
    query = QSqlQuery(sql.get_database())
    t1 = time.time()
    success = query.exec_(query_text)
    t2 = time.time()
    if success:
        while query.next():
            for i in range(100):
                result = query.value(i)
                if result is None:
                    break
                print_(result)
        print_(f"Completed in {time.strftime('%M:%S', time.localtime(t2 - t1))}.\n")
    else:
        print_("\nError:")
        print_(query.lastError().text() + "\n")


def remove_empty_dir(p: Path) -> None:
    if p.is_dir() and len(list(p.iterdir())) == 0:
        p.rmdir()


def clear_sqlite_tmp(plugin_data_dir: Path) -> None:
    """
    Очистка временной папки sqlite

    :param plugin_data_dir: Папка для временных данных плагина
    """

    try:

        db_dir = plugin_data_dir / "db"

        if not (db_dir.is_dir() and db_dir.exists()):
            return None

        for path_arg in db_dir.iterdir():
            clear_single_sqlite_tmp(path_arg)

        remove_empty_dir(db_dir)

    except (Exception,):
        print_exc_()


def clear_single_sqlite_tmp(uuid_dir: Path) -> None:
    try:
        # Файл базы данных
        tmp_sqlite_path = uuid_dir / "tmp.sqlite"
        if tmp_sqlite_path.is_file():
            tmp_sqlite_path.unlink()
        # Файл журнала базы данных
        tmp_journal_path = uuid_dir / "tmp.sqlite-journal"
        if tmp_journal_path.is_file():
            tmp_journal_path.unlink()
        # Удаление пустой папки
        remove_empty_dir(uuid_dir)
    except OSError:
        print_exc_()


class DbSettings:
    def __init__(self, uid: str, data_dir: Path):
        self.uid = uid  # type: str
        self.data_dir = data_dir  # type: Path


class LocalConnection:

    def __init__(self, db_settings: DbSettings):
        self.uid = db_settings.uid  # type: str
        self.data_dir = db_settings.data_dir  # type: Path

        self.db_path = None  # type: Optional[str]

        self.conn: Optional["sqlite3.Connection"] = None
        self.cursor: Optional["sqlite3.Cursor"] = None

    def init_db(self) -> None:
        data_dir = self.data_dir
        db_dir = data_dir / "db" / self.uid
        if not db_dir.exists():
            db_dir.mkdir(parents=True)

        self.db_path = str(db_dir / "tmp.sqlite")

        if sqlite3 is None:
            raise RuntimeError("sqlite3 is required but not installed or has missing dependencies.")

        self.conn = sqlite3.connect(self.db_path)
        self.conn.enable_load_extension(True)
        cursor = self.conn.cursor()
        cursor.execute("select load_extension('mod_spatialite')")
        cursor.execute("select InitSpatialMetaData(TRUE, 'NONE')")
        self.cursor = cursor

        query_exec(f"attach database '{self.db_path}' as {sqlite_db_name}")

    def get_cursor(self) -> Optional["sqlite3.Cursor"]:
        return self.cursor

    def commit(self) -> None:
        self.conn.commit()

    def close(self) -> None:
        query_exec(f"detach database {sqlite_db_name}")

        self.conn.close()

        data_dir = Path(self.data_dir)
        db_dir = data_dir / "db" / self.uid
        clear_single_sqlite_tmp(db_dir)

    def c_execute(self, query_text: str) -> None:
        print_("c_execute:")
        print_(query_text)

        t1 = time.time()
        self.cursor.execute(query_text)
        t2 = time.time()
        print_(self.cursor.fetchall())
        print_(f"Completed in {time.strftime('%M:%S', time.localtime(t2 - t1))}.\n")
