from typing import Dict

from axipy.da import GeometryType
from PySide2.QtCore import Qt
from PySide2.QtUiTools import QUiLoader
from PySide2.QtWidgets import QCheckBox, QGridLayout, QLayout, QVBoxLayout, QWidget

from ..data_types import (
    NamedSearchPatterns,
    SearchGeometries,
    SearchPatterns,
    SearchSettings,
)
from ..utils import geom_type_to_str
from .style_widget_interface import SelectByStyleWidgetInterface


class SelectByRegionStyleWidget(SelectByStyleWidgetInterface):

    def __init__(self, ui_file: str, parent: QWidget) -> None:
        super().__init__(parent)
        self.__ui = self.load_ui(ui_file)
        self.__patterns_comboboxes = {}  # type: Dict[str, Dict[str, QCheckBox]]
        self.__geom_comboboxes = {}  # type: Dict[GeometryType, QCheckBox]

    def load_ui(self, ui_file: str) -> QWidget:
        ui = QUiLoader().load(ui_file, self)
        grid_layout = QGridLayout()
        grid_layout.setHorizontalSpacing(0)
        grid_layout.setVerticalSpacing(0)
        grid_layout.addWidget(ui)
        self.setLayout(grid_layout)
        return ui

    def result(self) -> SearchSettings:
        return SearchSettings(self.search_patterns, self.search_geometries)

    def set_region_patterns(self, border_patterns: NamedSearchPatterns, brush_patterns: NamedSearchPatterns):
        self.fill_layout(border_patterns, self.__ui.borderVerticalLayout)
        self.fill_layout(brush_patterns, self.__ui.brushVerticalLayout)

    def fill_layout(self, patterns: NamedSearchPatterns, layout: QLayout):
        if patterns.name not in self.__patterns_comboboxes:
            self.__patterns_comboboxes[patterns.name] = {}
        grouped_comboboxes = self.__patterns_comboboxes[patterns.name]
        for pattern_name, is_checked in patterns.patterns.items():
            cb = QCheckBox(pattern_name)
            cb.setCheckState(Qt.Checked if is_checked else Qt.Unchecked)
            layout.addWidget(cb)
            grouped_comboboxes[pattern_name] = cb

    @property
    def search_geometries(self) -> SearchGeometries:
        result = {}  # type: SearchGeometries
        for gtype, cb in self.__geom_comboboxes.items():
            result[gtype] = cb.isChecked()
        return result

    @search_geometries.setter
    def search_geometries(self, geometry_types: Dict[GeometryType, bool]):
        vlayout = QVBoxLayout(self.__ui.searchGeometriesGroupBox)
        for gtype, is_checked in geometry_types.items():
            cb = QCheckBox(geom_type_to_str(gtype))
            cb.setCheckState(Qt.Checked if is_checked else Qt.Unchecked)
            vlayout.addWidget(cb)
            self.__geom_comboboxes[gtype] = cb
        self.__ui.searchGeometriesGroupBox.setLayout(vlayout)

    @property
    def search_patterns(self) -> SearchPatterns:
        result = {}  # type: SearchPatterns
        for group_patterns_name, group_patterns in self.__patterns_comboboxes.items():
            patterns = {}
            for pattern_name, cb in group_patterns.items():
                patterns[pattern_name] = cb.isChecked()
            result[group_patterns_name] = patterns
        return result
