import importlib.util
import inspect
import sys
from pathlib import Path
from types import ModuleType
from typing import Optional, Union

from PySide2.QtCore import QTranslator, QCoreApplication

from axipy._internal._shadow_instance_factory import _shadow_manager
from axipy.axioma_plugin import AxiomaPlugin
from axipy.interface import AxiomaInterface
from axipy.plugin import Plugin


def _load_plugin(absolute_path: str) -> Optional[Union[Plugin, AxiomaPlugin, AxiomaInterface]]:
    plugin_dir = Path(absolute_path)

    def init_translation(path: Path, filename_base: str):
        translator = QTranslator()
        status = translator.load(_shadow_manager.core.translationLocale(), filename_base, "_", str(path))
        if not status:
            return None
        if QCoreApplication.installTranslator(translator):
            translator.setParent(QCoreApplication.instance())

    init_translation(plugin_dir / "i18n", "translation")

    def load_plugin_dir(plugin_dir_arg: Path) -> ModuleType:
        name = plugin_dir_arg.name

        def remove_plugin(name_arg: str):
            if name_arg not in sys.modules:
                return None
            del sys.modules[name_arg]

            def is_submodule(subname: str):
                return subname.startswith(name_arg + '.')

            submodules = list(filter(is_submodule, sys.modules.keys()))
            for submodule in submodules:
                del sys.modules[submodule]

        remove_plugin(name)
        spec = importlib.util.spec_from_file_location(name, str(plugin_dir_arg / "__init__.py"))
        module_arg = importlib.util.module_from_spec(spec)
        sys.modules[name] = module_arg
        spec.loader.exec_module(module_arg)
        return module_arg

    module = load_plugin_dir(plugin_dir)

    for value in module.__dict__.values():
        if inspect.isclass(value) and issubclass(value, Plugin) and value != Plugin:
            return value._Plugin__create(plugin_dir)

    if not hasattr(module, 'Plugin'):
        return None

    plugin_decl = getattr(module, 'Plugin')
    if issubclass(plugin_decl, AxiomaPlugin):
        return plugin_decl._axioma_plugin_create(plugin_dir)

    signature = inspect.signature(plugin_decl.__init__)
    if 'iface' not in signature.parameters:
        return plugin_decl()

    axioma_interface = AxiomaInterface()
    axioma_interface.initialize(plugin_dir)
    return plugin_decl(axioma_interface)
