from typing import TYPE_CHECKING, Iterator, List, Optional, Union, cast

from axipy._internal._shadow_instance_factory import _shadow_manager
from axipy._internal._utils import _AxiRepr, _AxiReprMeta, _Singleton
from axipy.da import DataObject, Feature, Table
from PySide2.QtCore import Signal

if TYPE_CHECKING:
    from axipy.cpp_gui import ShadowSelectionManager

__all__: List[str] = [
    "SelectionManager",
    "selection_manager",
]


class _SelectionManager(_Singleton, _AxiRepr, metaclass=_AxiReprMeta): ...


class SelectionManager(_SelectionManager):
    """
    Класс доступа к выделенным объектам.

    Note:
        Создание :class:`axipy.SelectionManager` не требуется,
        используйте объект :attr:`axipy.selection_manager`.
    """

    @property
    def table(self) -> Optional[Table]:
        """
        Таблица, являющаяся источником текущего выделения.

        Если ничего не выделено, то None.
        """
        return cast(Optional[Table], DataObject._wrap(self._shadow.table()))

    @property
    def ids(self) -> List[int]:
        """Список идентификаторов выделенных записей."""
        return self._shadow.ids()

    @property
    def count(self) -> int:
        """Размер выделения, то есть количество выделенных записей (количество элементов
        в списке идентификаторов)."""
        return self._shadow.count()

    def __len__(self) -> int:
        return self.count

    def clear(self) -> None:
        """Очищает выборку."""
        self._shadow.clear()

    def set(self, table: Table, ids: Union[List[int], int]) -> None:
        """
        Создает выборку из записей таблицы по их идентификаторам.

        Args:
            table: Таблица.
            ids: Идентификаторы записей.

        Пример выбора всех объектов слоя::

            table = data_manager.find('world')
            if table:
                selection_manager.set(table, [f.id for f in table.items()])
        """
        if isinstance(ids, int):
            ids = [ids]
        self._shadow.set(table._shadow, ids)

    def set_all(self, table: Table) -> None:
        """
        Создает выборку по всем записям таблицы.

        Args:
            table: Таблица.

        Пример выбора всех объектов слоя::

            table = data_manager.find('world')
            if table:
                selection_manager.set_all(table)
        """
        self._shadow.set_all(table._shadow)

    def add(self, table: Table, ids: Union[List[int], int]) -> None:
        """
        Добавляет выборку записи таблицы по их идентификаторам.

        Args:
            table: Таблица.
            ids: Идентификаторы записей.

        Пример установки выборки из запроса::

            qry = data_manager.query('select * from world where Столица like "М%"')
            for feature in qry.items():
                selection_manager.add(qry, feature.id)
        """
        if self.count == 0:
            return self.set(table, ids)
        if isinstance(ids, int):
            ids = [ids]
        self._shadow.add(table._shadow, ids)

    def remove(self, table: Table, ids: Union[List[int], int]) -> None:
        """
        Удаляет из выборки записи таблицы по их идентификаторам.

        Args:
            table: Таблица.
            ids: Идентификаторы записей.
        """
        if isinstance(ids, int):
            ids = [ids]
        self._shadow.remove(table._shadow, ids)

    def get_as_cursor(self) -> Iterator[Feature]:
        """
        Возвращает выборку в виде итератора по записям.

        Пример::

            for f in selection_manager.get_as_cursor():
                print('Feature id={}. Страна={}'.format(f.id, f['Страна']))

        Warning:

            .. deprecated:: 3.5
                Используйте :attr:`axipy.DataManager.selection`.
        """
        if self.table is None:
            return iter(())
        return self.table.itemsByIds(self.ids)

    def get_as_table(self) -> Optional[Table]:
        """
        Возвращает выборку в виде таблицы.

        Returns:
            Таблица или :data:`None`, если выборки нет.

        Содержимое таблицы основывается на текущей выборке на момент вызова данного метода.
        При последующем изменении или сбросе выборки контент данной таблицы не меняется.
        Результирующей таблице присваивается наименование в формате `data*`, которое в последствии можно изменить.
        При закрытии базовой таблицы данная таблицы так-же закрывается.

        Пример::

            # Получаем таблицу из выборки.
            tbl = selection_manager.get_as_table()
            # Задаем желаемое имя таблицы (необязательно)
            tbl.name = 'my_table'
            # Регистрация в каталоге (необязательно)
            app.mainwindow.catalog().add(tbl)
            for f in tbl.items():
                print('Feature id={}. Страна={}'.format(f.id, f['Страна']))

        Warning:

            .. deprecated:: 3.5
                Используйте :attr:`axipy.DataManager.selection`.
        """
        return cast(Optional[Table], DataObject._wrap(self._shadow.as_table()))

    @property
    def _shadow(self) -> "ShadowSelectionManager":
        return _shadow_manager.selection_manager

    @property
    def changed(self) -> Signal:
        """
        Выделение было изменено.

        :rtype: Signal[]
        """
        return self._shadow.selectionChanged


selection_manager = SelectionManager()
