from typing import List, NoReturn, Optional

from axipy.da.data_object import DataObject

from .data_provider import DataProvider
from .source import Source

__all__: List[str] = [
    "WmtsSource",
    "WmtsDataProvider",
]


class WmtsSource(Source):
    pass


class WmtsDataProvider(DataProvider):
    """
    Провайдер для тайловых серверов.

    Note:
        Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.provider_manager.wmts`.
    """

    @staticmethod
    def _identifier() -> str:
        return "WmtsDataProvider"

    # noinspection PyPep8Naming
    def get_source(self, capabilitiesUrl: str, dataObject: str, alias: Optional[str] = None) -> Source:
        """
        Создает источник данных.

        Args:
          capabilitiesUrl: URL запроса метаданных.
          dataObject: Наименование слоя.
          alias: Псевдоним для открываемой таблицы.
        """
        return WmtsSource(
            Source._provider(self.id),
            Source._alias(alias),
            {"src": capabilitiesUrl, "dataobject": dataObject},
        )

    # noinspection PyPep8Naming
    def open(self, capabilitiesUrl: str, dataObject: str, alias: Optional[str] = None) -> DataObject:
        """
        Открывает объект данных.

        Args:
          capabilitiesUrl: URL запроса метаданных.
          dataObject: Наименование слоя.
          alias: Псевдоним для открываемого источника данных.
        """
        return self.get_source(capabilitiesUrl, dataObject, alias).open()

    def get_destination(self) -> NoReturn:
        """
        Attention:
            Не поддерживается.

        Raises:
            NotImplementedError
        """
        raise NotImplementedError

    def create_open(self) -> NoReturn:
        """
        Attention:
            Не поддерживается.

        Raises:
            NotImplementedError
        """
        raise NotImplementedError
