from typing import List, Optional, cast

from axipy.da.data_object import Table

from .data_provider import DataProvider
from .source import Destination, Schema, Source

__all__: List[str] = [
    "ExcelSource",
    "ExcelDestination",
    "ExcelDataProvider",
]


class ExcelSource(Source):
    pass


class ExcelDestination(Destination):
    pass


class ExcelDataProvider(DataProvider):
    """
    Провайдер чтения файлов Excel.

    Note:
        Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.provider_manager.excel`.
    """

    @staticmethod
    def _identifier() -> str:
        return "XlsDataProvider"

    def get_source(
        self,
        filepath: str,
        page: Optional[str] = None,
        with_header: bool = False,
        encoding: str = "utf8",
        alias: Optional[str] = None,
    ) -> Source:
        """
        Создает источник данных.

        Args:
            filepath: Путь к файлу.
            page: Имя страницы. Если не указана, то берется первая.
            with_header: Признак того, что в первой строке содержатся имена атрибутов таблицы.
            encoding: Кодировка.
            alias: Псевдоним для открываемой таблицы.
        """
        return ExcelSource(
            Source._provider(self.id),
            Source._table_file(filepath),
            Source._alias(alias),
            {
                "SheetName": page,
                "FirstRowAsFieldName": with_header,
                "charset": encoding,
            },
        )

    def open(
        self,
        filepath: str,
        page: Optional[str] = None,
        with_header: bool = False,
        encoding: str = "utf8",
        alias: Optional[str] = None,
    ) -> Table:
        """
        Открывает объект данных.

        Args:
            filepath: Путь к файлу.
            page: Имя страницы.
            with_header: Признак того, что в первой строке содержатся имена атрибутов таблицы.
            encoding: Кодировка.
            alias: Псевдоним для открываемой таблицы.
        """
        return cast(Table, self.get_source(filepath, page, with_header, encoding, alias).open())

    def get_destination(self, filepath: str, schema: Schema) -> Destination:
        """
        Создает назначение объекта данных.

        Args:
            filepath: Путь к файлу.
            schema: Схема таблицы.
        """
        return ExcelDestination(schema, Source._provider(self.id), Source._table_file(filepath))

    def create_open(self, filepath: str, schema: Schema) -> Table:
        """
        Создает и открывает объект данных.

        Args:
            filepath: Путь к файлу.
            schema: Схема таблицы.
        """
        return cast(Table, self.get_destination(filepath, schema).create_open())
