from abc import abstractmethod
from typing import TYPE_CHECKING, Any, Dict, List

from axipy._internal._utils import _AxiRepr, _AxiReprABCMeta

from .source import Destination, Source

if TYPE_CHECKING:
    from ..data_object import DataObject

__all__: List[str] = [
    "DataProvider",
]


class DataProvider(_AxiRepr, metaclass=_AxiReprABCMeta):
    """Абстрактный провайдер данных."""

    def __init__(self, info: Dict) -> None:
        if type(self) is DataProvider:
            raise NotImplementedError
        self.info = info

    @staticmethod
    @abstractmethod
    def _identifier() -> str:
        raise NotImplementedError

    @property
    def id(self) -> str:
        """Идентификатор провайдера."""
        return self.info["id"]

    def file_extensions(self) -> List[str]:
        """
        Список поддерживаемых расширений файлов.

        Returns:
            Пустой список для не файловых провайдеров.
        """
        return self.info.get("extensions", [])

    def get_source(self, *args: Any, **kwargs: Any) -> Source:
        """
        Создает источник данных.

        Raises:
            NotImplementedError: Если провайдер не поддерживает создание источников.
        """
        raise NotImplementedError

    def get_destination(self, *args: Any, **kwargs: Any) -> Destination:
        """
        Создает назначение объекта данных.

        Raises:
            NotImplementedError: Если провайдер не поддерживает создание назначений.
        """
        raise NotImplementedError

    def open(self, *args: Any, **kwargs: Any) -> "DataObject":
        """
        Открывает объект данных.

        Пример::

            provider.open(...)

        Что эквивалентно::

            provider.get_source(...).open()

        See also:

            :meth:`DataProvider.source`.
        """
        return self.get_source(*args, **kwargs).open()

    def create(self, *args: Any, **kwargs: Any) -> "DataObject":
        return self.create_open(*args, **kwargs)

    def create_open(self, *args: Any, **kwargs: Any) -> "DataObject":
        """
        Создает и открывает объект данных.

        Пример::

            provider.create_open(...)

        Что эквивалентно::

            provider.get_destination(...).create_open()

        See also:

            :meth:`DataProvider.destination`.
        """
        return self.get_destination(*args, **kwargs).create_open()
