from dataclasses import dataclass
from typing import TYPE_CHECKING, List, Optional, Tuple

from axipy._internal._decorator import _experimental_class
from axipy.cpp_core_dp import ShadowMapinfoMapcatalog, TableTypeGeometry
from axipy.cs import CoordSystem
from axipy.da import GeometryClass
from axipy.utl import Rect

__all__: List[str] = [
    "_MapCatalog",
]


@_experimental_class()
class _MapCatalog:
    """
    Работа с таблицей метаданных-каталогом mapinfo_mapcatalog


    See also:
        Получение :meth:`ProviderManager._catalog`.

        Пример использования::

            # Описание соединения
            src_local = provider_manager.postgre.get_source(host='localhost', db_name='esti', user='postgres', password='postgres')
            # Получение ссылки на каталог
            catalog = axipy.provider_manager._mapinfo_mapcatalog(src_local)
            # Создание структуры таблицы mapinfo_mapcatalog
            catalog._create_structure(False)
            # Определение идентификатора в таблице каталога
            tableKey = mapinfo_mapcatalog._MapCatalog.TableKey('public', 'statecap')
            # Определяем Систему Координат
            coordsys = CoordSystem.from_prj('1, 1004')
            # Регистрируем таблицу в каталоге
            catalog.register_table(tableKey, 'geom', GeometryClass.Lines, cs = coordsys)
            # Определяем охват
            r = Rect(-120, -80, 120, 80)
            catalog.set_rect(tableKey, r)
            # Определяем стиль оформления для таблицы
            style = LineStyle(3, Qt.red)
            catalog.set_style_string(tableKey, style.to_mapinfo())
            # Устанавливаем колонку таблицы, которая содержит стиль
            catalog.set_style_attribute_name(tableKey, 'MI_STYLE')
            # Установка наименований колонок для X и Y
            catalog.set_xy_column_names(tableKey, ('x_col', 'y_col'))

            print('Наличие регистрации:', catalog.is_registered_table(tableKey))
            print('Поддерживаемый тип объектов:', catalog.get_type_objects(tableKey))
            print('Охват:', catalog.get_rect(tableKey))
            print('СК:', catalog.get_coordsystem(tableKey))
            print('Стиль:', catalog.get_style_string(tableKey))
            print('Колонка со стилем:', catalog.get_style_attribute_name(tableKey))
            print('Колонки с координатами X и Y:', catalog.get_xy_column_names(tableKey))

            # Убирает регистрацию в таблице каталога
            catalog.unregister_table(tableKey)

            ```>>> Наличие регистрации: True```
            ```>>> Поддерживаемый тип объектов: GeometryClass.Lines```
            ```>>> Охват: (-120.0 -80.0) (120.0 80.0)```
            ```>>> СК: prj:Earth Projection 1, 1004```
            ```>>> Стиль: Pen (1, 3, 16711680)```
            ```>>> Колонка со стилем: MI_STYLE```
            ```>>> Колонки с координатами X и Y: ('x_col', 'y_col')```
    """

    if TYPE_CHECKING:
        _shadow: ShadowMapinfoMapcatalog

    @dataclass
    class TableKey:
        """
        Идентификатор таблицы в каталоге
        """

        owner: str
        """Схема таблицы"""
        table: str
        """Таблица"""

    def __init__(self) -> None:
        raise NotImplementedError

    @classmethod
    def _wrap(cls, shadow: ShadowMapinfoMapcatalog) -> "_MapCatalog":
        result = cls.__new__(cls)
        result._shadow = shadow
        return result

    def _create_structure(self, fill: bool = False) -> None:
        """
        Создает таблицу mapinfo_mapcatalog в БД если она не существует.

        Args:
            fill; Попробовать заполнить данными на основани информации из словаря БД
        """
        self._shadow.create_structure(fill)

    def register_table(
        self, key: TableKey, geom: str, tp: GeometryClass = GeometryClass.Unknown, cs: Optional[CoordSystem] = None
    ) -> None:
        """
        Регистрация таблицы

        Args:
            key: Таблица
            geom: Наименование колонки с геометрией
            tp: Тип поддерживаемых объектов
            cs: Система Кординат
        """
        self._shadow.register_table(key.owner, key.table, geom, TableTypeGeometry(tp))
        if cs is not None:
            self.set_coordsystem(key, cs)

    def unregister_table(self, key: TableKey) -> None:
        """
        Убирает регистрацию в таблице каталога

        Args:
            key: Таблица
        """
        self._shadow.unregister_table(key.owner, key.table)

    def is_registered_table(self, key: TableKey) -> bool:
        """
        Проверка наличия регистрации в таблице каталога

        Args:
            key: Таблица
        """
        return self._shadow.is_registered(key.owner, key.table)

    def get_rect(self, key: TableKey) -> Optional[Rect]:
        """
        Ограничивающий прямоугольник

        Args:
            key: Таблица
        """
        r = self._shadow.rect(key.owner, key.table)
        if r.isValid():
            return Rect.from_qt(r)
        return None

    def set_rect(self, key: TableKey, rect: Rect) -> None:
        """
        Установка ограничивающего прямоугольника

        Args:
            key: Таблица
            rect: Новое значение
        """
        self._shadow.set_rect(key.owner, key.table, Rect._rect_value_to_qt(rect))

    def get_coordsystem(self, key: TableKey) -> Optional[CoordSystem]:
        """
        Система координат таблицы.

        Args:
            key: Таблица
        """
        return CoordSystem._wrap(ShadowMapinfoMapcatalog.coordsystem(self._shadow, key.owner, key.table))

    def set_coordsystem(self, key: TableKey, cs: CoordSystem) -> None:
        """
        Установка системы координат таблицы.

        Args:
            key: Таблица
            cs: Новое значение
        """
        self._shadow.set_coordsystem(key.owner, key.table, cs._shadow)

    def get_style_string(self, key: TableKey) -> Optional[str]:
        """
        Строка со стилем - общего оформления всех записей таблицы.

        Args:
            key: Таблица
        """
        return self._shadow.style_string(key.owner, key.table)

    def set_style_string(self, key: TableKey, style: str) -> None:
        """
        Установка строки оформления записей таблицы.
        При установке значения колонка `renditiontype` устанавливается значение `0`

        Args:
            key: Таблица
            style: Новое значение
        """
        self._shadow.set_style_string(key.owner, key.table, style)

    def get_style_attribute_name(self, key: TableKey) -> Optional[str]:
        """
        Наименование колонки таблицы, которая содержит стиль.

        Args:
            key: Таблица
        """
        return self._shadow.style_attribute_name(key.owner, key.table)

    def set_style_attribute_name(self, key: TableKey, style_attribute: str) -> None:
        """
        Установка наименование колонки таблицы, которая содержит стиль.
        При установке значения колонка `renditiontype` устанавливается значение `1`

        Args:
            key: Таблица
            style_attribute: Новое значение колонки
        """
        self._shadow.set_attribute_name(key.owner, key.table, style_attribute)

    def get_type_objects(self, key: TableKey) -> GeometryClass:
        """
        Возвращает тип поддерживаемой геометрии.

        Args:
            key: Таблица
        """
        return GeometryClass(self._shadow.type_objects(key.owner, key.table))

    # def set_type_objects(self, key: TableKey, type_geometry: GeometryClass) -> None:
    #     """
    #     Установка Типа поддерживаемой геометрии.

    #     Args:
    #         key: Таблица
    #         type_geometry: Тип поддерживаемой геометрии
    #     """
    #     self._shadow.set_type_objects(key.owner, key.table, TableTypeGeometry(type_geometry))

    def get_xy_column_names(self, key: TableKey) -> Tuple[str, str]:
        """
        Возвращает наименования колонок таблицы, которая содержат координаты `X` и `Y`.
        """
        return self._shadow.xy_column_names(key.owner, key.table)

    def set_xy_column_names(self, key: TableKey, columns: Tuple[str, str]) -> None:
        """
        Установка наименований колонок таблицы, которая содержат координаты `X` и `Y`.

        Args:
            key: Таблица
            columns: Новые значения колонок
        """
        self._shadow.set_xy_column_names(key.owner, key.table, columns)
