import collections.abc
import functools
from abc import abstractmethod
from enum import Enum
from functools import cached_property
from typing import (
    TYPE_CHECKING,
    Any,
    Callable,
    Dict,
    Iterable,
    Iterator,
    List,
    Optional,
    Tuple,
    Type,
    Union,
    cast,
    overload,
)

import axipy
from axipy._internal._decorator import _experimental
from axipy.cpp_common import ShadowCalcMode
from axipy.cpp_core_geometry import (
    ShadowCollection,
    ShadowGeometry,
    ShadowLine,
    ShadowLinearRing,
    ShadowLineString,
    ShadowMultiLineString,
    ShadowMultiPoint,
    ShadowMultiPolygon,
    ShadowPoint,
    ShadowPointsLS,
    ShadowPolygon,
)
from axipy.cs import AreaUnit, CoordSystem, LinearUnit
from axipy.utl import CalcMode, Pnt, Rect
from PySide2.QtCore import QPoint, QPointF
from PySide2.QtGui import QTransform

from .._internal._decorator import _deprecated_by

if TYPE_CHECKING:
    from axipy.cpp_core_geometry import (
        ShadowArc,
        ShadowEllipse,
        ShadowRectangle,
        ShadowRoundRectangle,
        ShadowText,
    )
    from axipy.cpp_cs import ShadowAreaUnit, ShadowLinearUnit


__all__: List[str] = [
    "Geometry",
    "Point",
    "Line",
    "ListPoint",
    "LineString",
    "ListHoles",
    "Polygon",
    "GeometryCollection",
    "MultiPoint",
    "MultiLineString",
    "MultiPolygon",
    "GeometryType",
    "LineDirection",
    "LineCapStyle",
    "LineJoinStyle",
]


class LineCapStyle(int, Enum):
    """Стиль окончания линии."""

    Round = 1
    """Конец линии закруглен."""
    Flat = 2
    """Конец линии срезан по ее окончанию."""
    Square = 3
    """Конец линии срезан посередине между ее окончанием и полученным результатом."""


class LineJoinStyle(int, Enum):
    """Стиль соединения линий."""

    Round = 1
    """Соединение линий скруглено."""
    Mitre = 2
    """Соединение линий плоское."""
    Bevel = 3
    """Соединение - точка пересечения края результата"""


class GeometryType(int, Enum):
    """Тип геометрического элемента."""

    Unknown = 0
    """Не определен"""
    Point = 1
    """Точка"""
    Line = 2
    """Линия"""
    LineString = 3
    """Полилиния"""
    Polygon = 4
    """Полигон"""
    MultiPoint = 5
    """Коллекция точек"""
    MultiLineString = 6
    """Коллекция полилиний"""
    MultiPolygon = 7
    """Коллекция полигонов"""
    GeometryCollection = 8
    """Смешанная коллекция"""
    Arc = 9
    """Дуга"""
    Ellipse = 10
    """Эллипс"""
    Rectangle = 11
    """Прямоугольник"""
    RoundedRectangle = 12
    """Скругленный прямоугольник"""
    Text = 13
    """Текст"""


class _FixInputParams:
    """Different kinds of input."""

    @classmethod
    def __check_if_tuple_pnt(cls, target_tuple: tuple) -> bool:
        return len(target_tuple) == 2 and all(isinstance(coord, (int, float)) for coord in target_tuple)

    @classmethod
    def _fix_points_as_param(cls, points: tuple) -> tuple:
        """
        Input arguments tuple is either:
            - tuple of multiple single-geom-like objects
            - tuple of a single container with single-geom-like objects
            - tuple of 2 floats or ints (single point)
        Will be returned as tuple of multiple single-geom-like objects, even if its tuple of singular object.
        """
        if len(points) <= 0:
            raise ValueError("points argument can't be empty.")
        # tuple of Pnt, as expected
        elif isinstance(points[0], Pnt):
            return points
        # if first arg is tuple, can be 2 cases:
        elif isinstance(points[0], tuple):
            # first arg is tuple[float, float]
            if cls.__check_if_tuple_pnt(points[0]):
                return points
            # first arg is tuple of arbitrary collection, assume its of supported types
            return tuple(points[0])
        # one point as float, float
        elif cls.__check_if_tuple_pnt(points):
            # wrap two coordinates as tuple[tuple[x, y],]
            return (points,)
        # any other Iterable passed as first argument
        elif isinstance(points[0], Iterable):
            return tuple(points[0])
        raise ValueError("Unexpected parameter:{}".format(type(points)))

    @classmethod
    def _fix_points_and_cs(cls, points: tuple, cs: Optional[CoordSystem]) -> Tuple[tuple, Optional[CoordSystem]]:
        if cs is not None:
            return points, cs
        elif len(points) > 0:
            # (pnt1, pnt2, cs)
            if isinstance(points[-1], CoordSystem):
                return points[:-1], points[-1]
            # ([pnt1, pnt2], cs)
            elif len(points) == 2 and isinstance(points[1], CoordSystem):
                return points[0], points[1]
        return points, cs


class Geometry:
    """
    Абстрактный класс геометрического объекта (геометрии).

    Note:
        Для получения краткого текстового представления геометрии можно воспользоваться функцией :class:`str`.
        Для более полного - :func:`repr`.
    """

    _shadow: ShadowGeometry

    @abstractmethod
    def __init__(self) -> None:
        raise NotImplementedError

    def _set_shadow(self, shadow: ShadowGeometry) -> None:
        self._shadow: ShadowGeometry = shadow

    @classmethod
    def __wrap_typed(cls, shadow: ShadowGeometry) -> "Geometry":
        obj = cls.__new__(cls)
        obj._shadow = shadow
        return obj

    @overload
    @classmethod
    def _wrap(cls, shadow: ShadowPoint) -> Optional["Point"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: ShadowLine) -> Optional["Line"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: ShadowLineString) -> Optional["LineString"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: ShadowPolygon) -> Optional["Polygon"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: ShadowMultiPoint) -> Optional["MultiPoint"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: ShadowMultiLineString) -> Optional["MultiLineString"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: ShadowMultiPolygon) -> Optional["MultiPolygon"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: ShadowCollection) -> Optional["GeometryCollection"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: "ShadowRoundRectangle") -> Optional["axipy.mi.RoundRectangle"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: "ShadowRectangle") -> Optional["axipy.mi.Rectangle"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: "ShadowEllipse") -> Optional["axipy.mi.Ellipse"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: "ShadowArc") -> Optional["axipy.mi.Arc"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: "ShadowText") -> Optional["axipy.mi.Text"]: ...

    @overload
    @classmethod
    def _wrap(cls, shadow: ShadowGeometry) -> Optional["Geometry"]: ...

    @classmethod
    def _wrap(cls, shadow: ShadowGeometry) -> Optional["Geometry"]:
        if shadow is None:
            return None

        types_geometry: Dict[str, Type["Geometry"]] = {
            "Point": Point,
            "Line": Line,
            "LineString": LineString,
            "Polygon": Polygon,
            "GeometryCollection": GeometryCollection,
            "MultiPoint": MultiPoint,
            "MultiLineString": MultiLineString,
            "MultiPolygon": MultiPolygon,
            "Rectangle": axipy.mi.Rectangle,
            "RoundRectangle": axipy.mi.RoundRectangle,
            "Ellipse": axipy.mi.Ellipse,
            "Arc": axipy.mi.Arc,
            "Text": axipy.mi.Text,
        }
        t = types_geometry.get(shadow.id(), Geometry)
        return t.__wrap_typed(shadow)

    def _try_to_point_tuple(self, obj: object) -> Optional[Tuple[float, float]]:
        if isinstance(obj, Tuple):  # type: ignore[arg-type]
            obj_t: Tuple = cast(Tuple, obj)
            if len(obj_t) == 2:
                return cast(Tuple[float, float], obj)
        return None

    def __shadow_calc_mode(self, calc_mode: Optional[CalcMode]) -> ShadowCalcMode:
        return ShadowCalcMode(calc_mode) if calc_mode is not None else ShadowCalcMode.CalcModeDefault

    @overload
    def __shadow_unit(self, unit: "AreaUnit") -> "ShadowAreaUnit": ...

    @overload
    def __shadow_unit(self, unit: "LinearUnit") -> "ShadowLinearUnit": ...

    @overload
    def __shadow_unit(self, unit: None) -> None: ...

    def __shadow_unit(
        self, unit: Union[None, "AreaUnit", "LinearUnit"]
    ) -> Union[None, "ShadowLinearUnit", "ShadowAreaUnit"]:
        return unit._shadow if unit is not None else None

    def __ensure_result(self, v: float, what: str) -> float:
        if v == 0:
            raise ValueError(f"Could not to calculate {what}.")
        return v

    def __ensure_cs(self) -> None:
        if self.coordsystem is None:
            raise RuntimeWarning("CoordSystem is not defined.")

    def get_length(self, u: Optional[LinearUnit] = None, calc_mode: Optional[CalcMode] = None) -> float:
        """
        Рассчитывает длину геометрии. Метод применим только для линейных объектов. В
        случае, если СК задана как Широта/Долгота, то расчет производится на сфере в
        метрах.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_length
            :lines: 2-
            :dedent: 4

        Args:
            u: Единица измерения, в которой необходимо получить результат.
                Если не задана, то используется единица измерения для СК.
                Если и она не задана, то производится расчет на плоскости.
            calc_mode: Режим расчета. Если не задан, то используется значение по умолчанию для
                 СК геометрии :attr:`CoordSystem.supported_calc_modes`.

        Raises:
            ValueError, RuntimeError, RuntimeWarning
        """
        if u is not None:
            self.__ensure_cs()

        return self.__ensure_result(
            self._shadow.length(self.__shadow_calc_mode(calc_mode), self.__shadow_unit(u)), "length"
        )

    def get_perimeter(self, u: Optional[LinearUnit] = None, calc_mode: Optional[CalcMode] = None) -> float:
        """
        Рассчитывает периметр геометрии. Метод применим только для площадных объектов. В
        случае, если СК задана как Широта/Долгота, то расчет производится на сфере в
        метрах.

        Пример см. :meth:`get_area`

        Args:
            u: Единица измерения, в которой необходимо получить результат.
                Если не задана, то используется единица измерения для СК.
                Если и она не задана, то производится расчет на плоскости.
            calc_mode: Режим расчета. Если не задан, то используется значение по умолчанию
                для СК геометрии :attr:`CoordSystem.supported_calc_modes`.

        Raises:
            ValueError, RuntimeError, RuntimeWarning
        """
        if u is not None:
            self.__ensure_cs()

        return self.__ensure_result(
            self._shadow.perimeter(self.__shadow_calc_mode(calc_mode), self.__shadow_unit(u)), "perimeter"
        )

    def get_area(self, area_unit: Optional[AreaUnit] = None, calc_mode: Optional[CalcMode] = None) -> float:
        """
        Рассчитывает площадь, если объект площадной. В случае, если СК задана как
        Широта/Долгота, то расчет производится на сфере или эллипсоиде по заданному
        значению `calc_mode` в квадратных метрах.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_area
            :lines: 2-
            :dedent: 4

        Args:
            area_unit: Единица измерения, в которой необходимо получить результат.
                Если не задана, то используется единица измерения для СК.
                Если и она не задана, то производится расчет на плоскости.
            calc_mode: Режим расчета. Если не задан, то используется значение по умолчанию
                для СК геометрии :attr:`CoordSystem.supported_calc_modes`.

        Raises:
            ValueError, RuntimeError, RuntimeWarning
        """
        if area_unit is None:
            if self.coordsystem:
                area_unit = AreaUnit.from_linear_unit(self.coordsystem.unit)

        if isinstance(area_unit, AreaUnit):
            self.__ensure_cs()

        return self.__ensure_result(
            self._shadow.area(self.__shadow_calc_mode(calc_mode), self.__shadow_unit(area_unit)), "area"
        )

    @property
    def bounds(self) -> Rect:
        """Возвращает минимальный ограничивающий прямоугольник."""
        return Rect.from_qt(self._shadow.bounds())

    @property
    def type(self) -> GeometryType:
        """
        Возвращает тип геометрического элемента.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :pyobject: test_run_example_geometry_type
            :lines: 2-
            :dedent: 4
        """
        return GeometryType(self._shadow.type())

    @property
    def name(self) -> str:
        """Возвращает наименование геометрического объекта."""
        return self._shadow.name()

    def get_distance(self, other: "Geometry", u: Optional[LinearUnit] = None) -> float:
        """
        Производит расчет расстояния до объекта `other`. Результат возвращает в СК
        текущего объекта.

        Args:
            other: Анализируемый объект.
            u: Единицы измерения, в которых требуется получить результат.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_distance
            :lines: 2-
            :dedent: 4
        """
        if u is not None:
            if not isinstance(u, LinearUnit):
                raise ValueError("Unit must be of LinearUnit type.")
            if self.coordsystem is None:
                raise RuntimeWarning("CoordSystem is not defined")
        return self._shadow.distance(other._shadow, self.__shadow_unit(u))

    @classmethod
    def from_wkt(cls, wkt: str, coordsystem: Optional[CoordSystem] = None) -> "Geometry":
        """
        Создает геометрический объект из строки формата
        `WKT <https://en.wikipedia.org/wiki/Well-known_text_representation_of_geometry>`_.

        Args:
            wkt: Строка WKT. Допустимо задание строки в формате с указанием SRID (EWKT).
                В данном случае система координат
                для создаваемой геометрии будет установлено исходя их этого значения.
            coordsystem: Система координат, которая будет установлена для геометрии.
                Если строка задана в виде EWKT и указано значение SRID, игнорируется.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_wkt
            :lines: 2-
            :dedent: 4
        """
        return cast(
            Geometry, cls._wrap(ShadowGeometry.from_wkt(wkt, coordsystem._shadow if coordsystem is not None else None))
        )

    def to_wkt(self) -> str:
        """Возвращает WKT строку для геометрии."""
        return self._shadow.to_wkt()

    @classmethod
    def from_wkb(cls, wkb: bytes, coordsystem: Optional[CoordSystem] = None) -> "Geometry":
        """
        Создает геометрический объект из строки формата
        `WKB <https://en.wikipedia.org/wiki/Well-known_text_representation_of_geometry>`_.

        Args:
            wkb: Строка WKB
            coordsystem: Система координат, которая будет установлена для геометрии.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_wkb
            :lines: 2-
            :dedent: 4

        """
        return cast(
            Geometry, cls._wrap(ShadowGeometry.from_wkb(wkb, coordsystem._shadow if coordsystem is not None else None))
        )

    def to_wkb(self) -> bytes:
        """Возвращает WKB строку для геометрии."""
        return bytes(self._shadow.to_wkb())

    @property
    def coordsystem(self) -> Optional[CoordSystem]:
        """Система Координат (СК) геометрии."""
        return CoordSystem._wrap(self._shadow.cs())

    @coordsystem.setter
    def coordsystem(self, cs: Optional[CoordSystem]) -> None:
        self._shadow.set_cs(cs._shadow if cs is not None else None)

    # Relations
    def equals(self, other: "Geometry") -> bool:
        """
        Производит сравнение с другой геометрией.

        Args:
            other: Сравниваемая геометрия.

        Returns:
            Возвращает True, если геометрии равны.
        """
        return self._shadow.equals(other._shadow)

    def __eq__(self, other: object) -> bool:
        if isinstance(other, Geometry):
            return self._shadow.equals(other._shadow)
        return NotImplemented

    def almost_equals(self, other: "Geometry", tolerance: float) -> bool:
        """
        Производит примерное сравнения с другой геометрией в пределах заданной точности.

        Args:
            other: Сравниваемый объект.
            tolerance: Точность сравнения.

        Returns:
            Возвращает True, если геометрии равны в пределах заданного отклонения.
        """
        return self._shadow.almost_equals(other._shadow, tolerance)

    def contains(self, other: "Geometry") -> bool:
        """Возвращает True, если геометрия полностью содержит передаваемую в качестве
        параметра геометрию."""
        return self._shadow.contains(other._shadow)

    def crosses(self, other: "Geometry") -> bool:
        """Возвращает True, если при пересечении геометрий объекты частично
        пересекаются."""
        return self._shadow.crosses(other._shadow)

    def disjoint(self, other: "Geometry") -> bool:
        """Возвращает True, если геометрии не пересекаются и не соприкасаются."""
        return self._shadow.disjoint(other._shadow)

    def intersects(self, other: "Geometry") -> bool:
        """Возвращает True, если геометрии пересекаются."""
        return self._shadow.intersects(other._shadow)

    def overlaps(self, other: "Geometry") -> bool:
        """Возвращает True, если пересечение геометрий отличается от обеих геометрий."""
        return self._shadow.overlaps(other._shadow)

    def touches(self, other: "Geometry") -> bool:
        """Возвращает True, если геометрии соприкасаются."""
        return self._shadow.touches(other._shadow)

    def within(self, other: "Geometry") -> bool:
        """Возвращает True, если геометрия находится полностью внутри геометрии
        `other`."""
        return self._shadow.within(other._shadow)

    def covers(self, other: "Geometry") -> bool:
        """Возвращает True, если геометрия охватывает геометрию `other`."""
        return self._shadow.covers(other._shadow)

    # de-9im
    def relate(self, other: "Geometry") -> str:
        """
        Проверяет отношения между объектами.

        Подробнее см. `DE-9IM
        <https://en.wikipedia.org/wiki/DE-9IM>`_
        """
        return self._shadow.relate(other._shadow)

    # Analysis
    def boundary(self) -> Optional["Geometry"]:
        """Возвращает границы геометрии в виде полилинии."""
        return Geometry._wrap(ShadowGeometry.boundary(self._shadow))

    def centroid(self) -> "Point":
        """Возвращает центроид геометрии."""
        return cast(Point, Geometry._wrap(ShadowGeometry.centroid(self._shadow)))

    def difference(self, other: "Geometry") -> Optional["Geometry"]:
        """Возвращает область первой геометрии, которая не пересечена второй геометрией."""
        return Geometry._wrap(ShadowGeometry.difference(self._shadow, other._shadow))

    def intersection(self, other: "Geometry") -> Optional["Geometry"]:
        """Возвращает область пересечения с другой геометрией."""
        return Geometry._wrap(ShadowGeometry.intersection(self._shadow, other._shadow))

    def symmetric_difference(self, other: "Geometry") -> "Geometry":
        """
        Возвращает логический XOR областей геометрий (объединение разниц).

        Args:
            other: Геометрия для анализа.
        """
        return cast(Geometry, Geometry._wrap(ShadowGeometry.symmetric_difference(self._shadow, other._shadow)))

    def union(self, other: "Geometry") -> "Geometry":
        """Возвращает результат объединения двух геометрий."""
        return cast(Geometry, Geometry._wrap(ShadowGeometry.union(self._shadow, other._shadow)))

    # noinspection PyPep8Naming
    def buffer(
        self,
        distance: float,
        resolution: int = 16,
        capStyle: LineCapStyle = LineCapStyle.Round,
        join_style: LineJoinStyle = LineJoinStyle.Round,
        mitreLimit: float = 5.0,
    ) -> "Geometry":
        """
        Производит построение буфера вокруг объекта.

        Args:
            distance: Ширина буфера.
            resolution: Количество сегментов на квадрант.
            capStyle: Стиль окончания.
            join_style: Стиль соединения.
            mitreLimit: Предел среза.
        """
        return cast(
            Geometry,
            Geometry._wrap(ShadowGeometry.buffer(self._shadow, distance, resolution, capStyle, join_style, mitreLimit)),
        )

    def convex_hull(self) -> "Geometry":
        """Возвращает минимальный окаймляющий полигон со всеми выпуклыми углами."""
        return cast(Geometry, Geometry._wrap(ShadowGeometry.convex_hull(self._shadow)))

    def envelope(self) -> "Geometry":
        """Возвращает полигон, описывающий заданную геометрию."""
        return cast(Geometry, Geometry._wrap(ShadowGeometry.envelope(self._shadow)))

    @_experimental()
    def _snap(self, target: "Geometry", tolerance: float) -> "Geometry":
        """
        Подбивает текущую геометрию к геометрии target с заданной точностью.

        Args:
            target: Геометрия, к которой производится привязка
            tolerance: Точность
        """
        return cast(Geometry, Geometry._wrap(ShadowGeometry.snap(self._shadow, target._shadow, tolerance)))

    # Transform
    def affine_transform(self, trans: QTransform) -> "Geometry":
        """
        Трансформирует объект исходя из заданных параметров трансформации.

        Args:
            trans: Матрица трансформации.

        See also:
            Для простых операций типа сдвиг, масштабирование  и поворот,
            рекомендуется использовать :meth:`shift`, :meth:`scale`, :meth:`rotate` соответственно.

        Для выполнения преобразования необходимо сформировать матрицу :class:`PySide2.QtGui.QTransform`.
        Сделать это можно или последовательно дополняя преобразование или же сразу задав необходимые коэффициенты.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример последовательного задания матрицы.
            :pyobject: test_run_example_geometry_affine
            :lines: 2-23
            :dedent: 4

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример задания матрицы через коэффициенты.
            :pyobject: test_run_example_geometry_affine
            :lines: 2-12,24-
            :dedent: 4
        """
        return cast(Geometry, Geometry._wrap(ShadowGeometry.affine_transform(self._shadow, trans)))

    @_experimental()
    def _round(self, ndigits: int = 0) -> "Geometry":
        """
        Округление координат геометрии.

        Args:
            ndigits: Количество знаков после запятой. Если значение отрицательное, округление по количеству разрядов до запятой


        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример округления координат.
            :pyobject: test_run_example_geometry_round
            :start-after: # start
            :end-before: # finish
            :dedent: 4
        """
        return cast(Geometry, Geometry._wrap(ShadowGeometry.round(self._shadow, ndigits)))

    def shift(self, dx: float, dy: float) -> "Geometry":
        """
        Смещает объект на заданную величину. Значения задаются в текущей проекции.

        Args:
            dx: Сдвиг по координате X.
            dy: Сдвиг по координате Y.
        """
        return self.affine_transform(QTransform.fromTranslate(dx, dy))

    def scale(self, kx: float, ky: float) -> "Geometry":
        """
        Масштабирует объект по заданным коэффициентам масштабирования. После выполнения
        операции центр результирующего объекта остается прежним.

        Args:
            kx: Масштабирование по координате X.
            ky: Масштабирование по координате Y.
        """
        c = self.bounds.center
        tr_shift = QTransform.fromTranslate(-c.x, -c.y)
        tr_scale = QTransform.fromScale(kx, ky)
        tr = tr_shift * tr_scale * tr_shift.inverted()[0]
        return self.affine_transform(tr)

    def reproject(self, cs: CoordSystem) -> "Geometry":
        """
        Перепроецирует геометрию в другую систему координат.

        Args:
            cs: СК, в которой требуется получить объект.
        """
        return cast(Geometry, Geometry._wrap(ShadowGeometry.reproject(self._shadow, cs._shadow)))

    def rotate(self, point: Union[Pnt, Tuple[float, float]], angle: float) -> "Geometry":
        """
        Поворот геометрии относительно заданной точки. Поворот производится против
        часовой стрелки.

        Args:
            point: Точка, вокруг которой необходимо произвести поворот.
            angle: Угол поворота в градусах.
        """
        return cast(Geometry, Geometry._wrap(ShadowGeometry.rotate(self._shadow, Pnt._fix_pnt(point).to_qt(), angle)))

    @property
    def is_valid(self) -> bool:
        """Проверяет геометрию на валидность."""
        return self._shadow.is_valid()

    @property
    def is_valid_reason(self) -> str:
        """Если геометрия неправильная, возвращает краткую аннотацию причины."""
        return self._shadow.is_valid_reason()

    def normalize(self) -> "Geometry":
        return cast(Geometry, Geometry._wrap(ShadowGeometry.normalize(self._shadow)))

    def clone(self) -> "Geometry":
        """Создает копию объекта."""
        return cast(Geometry, self._wrap(ShadowGeometry.clone(self._shadow)))

    @staticmethod
    def point_by_azimuth(
        point: Union[Pnt, Tuple[float, float]],
        azimuth: float,
        distance: float,
        cs: Optional[CoordSystem] = None,
    ) -> Pnt:
        """
        Производит расчет координат точки относительно заданной, и находящейся на
        расстоянии `distance` по направлению `azimuth`.

        .. seealso::
            Обратная функция :meth:`distance_by_points`

        Args:
            point: Точка, относительно которой производится расчет. Если задана СК, то в ней.
            azimuth: Азимут в градусах, указывающий направление.
            distance: Расстояние по азимуту. Задается в метрах.
            cs: СК, на базе эллипсоида которой производится расчет. Если не задана, то расчет производится на плоскости.

        Returns:
            Возвращает результирующую точку.
        """
        return Pnt.from_qt(
            ShadowGeometry.point_by_azimuth(
                Pnt._fix_pnt(point).to_qt(),
                azimuth,
                distance,
                None if cs is None else cs._shadow,
            )
        )

    @staticmethod
    def distance_by_points(
        start: Union[Pnt, Tuple[float, float]],
        end: Union[Pnt, Tuple[float, float]],
        cs: Optional[CoordSystem] = None,
    ) -> Tuple:
        """
        Производит расчет расстояния между двумя точками и азимут от первой до второй
        точки.

        .. seealso::
            Обратная функция :meth:`point_by_azimuth`

        Args:
            start: Начальная точка. Если задана СК, то координаты в ней.
            end: Конечная точка. Если задана СК, то координаты в ней.
            cs: СК, на базе эллипсоида которой производится расчет. Если не задана, то расчет производится на плоскости.

        Returns:
            Возвращается пара значений (расстояние в метрах, азимут в градусах)
        """
        return ShadowGeometry.distance_by_points(
            Pnt._fix_pnt(start).to_qt(),
            Pnt._fix_pnt(end).to_qt(),
            None if cs is None else cs._shadow,
        )

    def __repr__(self) -> str:
        """Представление геометрии в виде строки."""
        return self.to_wkt() + self._cs_str

    @property
    def _cs_str(self) -> str:
        return "; " + str(self.coordsystem) if self.coordsystem else ""

    @property
    def _class_name(self) -> str:
        return self.__class__.__name__

    def to_linestring(self) -> Optional["Geometry"]:
        """
        Пробует геометрию преобразовать в линейный объект.

        В случае неудачи возвращает None.
        """
        return Geometry._wrap(ShadowGeometry.convertToMultiLineString(self._shadow))

    def to_polygon(self) -> Optional["Geometry"]:
        """
        Пробует геометрию преобразовать в площадной объект.

        В случае неудачи возвращает None.
        """
        return Geometry._wrap(ShadowGeometry.convertFromLineStringsToPolygons(self._shadow))

    def to_geojson(self) -> str:
        """Преобразует геометрию в формат 'GeoJSON'."""
        return ShadowGeometry.convertGeometryToJson(self._shadow)

    @staticmethod
    def from_geojson(json: str, cs: Optional[CoordSystem] = None) -> "Geometry":
        """
        Возвращает геометрию из ее 'GeoJSON' представления.

        Args:
            json: json представление в виде строки.
            cs: Система координат.
        """
        return cast(Geometry, Geometry._wrap(ShadowGeometry.geometryFromJson(json, None if cs is None else cs._shadow)))

    def to_mif(self) -> str:
        """
        Преобразует геометрию в формат `MIF`.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :pyobject: test_run_example_geometry_mif
            :lines: 2-3
            :dedent: 4
        """
        return ShadowGeometry.geometryToMif(self._shadow)

    @staticmethod
    def from_mif(mif: str) -> "Geometry":
        """
        Возвращает геометрию из ее 'MIF' представления.

        Args:
            mif: mif представление в виде строки.
        """
        return cast(Geometry, Geometry._wrap(ShadowGeometry.geometryFromMif(mif)))

    def split_by_polyline(self, splitter: "Geometry") -> Optional["Geometry"]:
        """
        Разрезает объект линейным объектом.

        Args:
            splitter: Геометрический объект, которым будет производиться разрезание объекта.

        Return:
            Возвращает полученный результат

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :pyobject: test_run_example_geometry_split
            :lines: 2-
            :dedent: 4
        """
        if splitter is None:
            raise ValueError()
        return Geometry._wrap(ShadowGeometry.polyline_split(self._shadow, splitter._shadow))

    @classmethod
    def _iterable_pnt_to_qt(cls, points: Iterable[Pnt]) -> List[QPointF]:
        return [p.to_qt() for p in points]


class Point(Geometry):
    """
    Геометрический объект типа точка.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_point
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, x: float, y: float, cs: Optional[CoordSystem] = None) -> None:
        """
        Конструктор класса.

        Args:
            x: X координата.
            y: Y координата.
            cs: Система Координат, в которой создается геометрия.
        """
        self._set_shadow(ShadowPoint(x, y, None if cs is None else cs._shadow))

    @property
    def x(self) -> float:
        """X Координата."""
        return self._shadow.get_pos().x()

    @x.setter
    def x(self, x: float) -> None:
        self._shadow.set_pos(QPointF(x, self.y))

    @property
    def y(self) -> float:
        """Y Координата."""
        return self._shadow.get_pos().y()

    @y.setter
    def y(self, y: float) -> None:
        self._shadow.set_pos(QPointF(self.x, y))

    def __check_geometry(self, other: object) -> Optional[Geometry]:
        if isinstance(other, Geometry):
            return cast(Geometry, other)
        result: Optional[Tuple[float, float]] = self._try_to_point_tuple(other)
        if result:
            other_t: Tuple[float, float] = cast(Tuple[float, float], result)
            return Point(other_t[0], other_t[1])  # Без СК!!!
        return None

    def __eq__(self, other: object) -> bool:
        g: Optional[Geometry] = self.__check_geometry(other)
        if g is not None:
            return self._shadow.equals(g._shadow)
        return NotImplemented

    def __str__(self) -> str:
        return "{} pos={}".format(self._class_name, Pnt.from_qt(self._shadow.get_pos())) + self._cs_str

    _shadow: ShadowPoint


class Line(Geometry):
    """
    Геометрический объект типа линия.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_line
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, begin: Pnt, end: Pnt, cs: Optional[CoordSystem] = None) -> None:
        """
        Конструктор класса.

        Args:
            begin: Начальная точка линии.
            end: Конечная точка линии.
            cs: Система Координат, в которой создается геометрия.
        """
        self._set_shadow(
            ShadowLine(
                Pnt._fix_pnt(begin).to_qt(),
                Pnt._fix_pnt(end).to_qt(),
                None if cs is None else cs._shadow,
            )
        )

    @property
    def begin(self) -> Pnt:
        """
        Начальная точка линии.

        Точки допустимо создавать как экземпляр :class:`Pnt` либо в виде пары `float` значений tuple
        """
        return Pnt.from_qt(self._shadow.get_begin())

    @begin.setter
    def begin(self, p: Union[Pnt, Tuple[float, float]]) -> None:
        self._shadow.set_begin(Pnt._fix_pnt(p).to_qt())

    @property
    def end(self) -> Pnt:
        """Конечная точка линии."""
        return Pnt.from_qt(self._shadow.get_end())

    @end.setter
    def end(self, p: Union[Pnt, Tuple[float, float]]) -> None:
        self._shadow.set_end(Pnt._fix_pnt(p).to_qt())

    def __repr__(self) -> str:
        return "{} begin={} end={}".format(self._class_name, self.begin, self.end) + self._cs_str

    _shadow: ShadowLine


class LineDirection(int, Enum):
    """Направление линии."""

    CounterClockwise = 0
    """Против часовой стрелки."""
    Clockwise = 1
    """По часовой стрелке."""


class ListPoint:

    def __init__(self, ls: Union[ShadowLineString, ShadowLinearRing]) -> None:
        self._shadow: ShadowPointsLS = ShadowPointsLS(ls)

    def __len__(self) -> int:
        return self._shadow.count()

    def __getitem__(self, idx: int) -> Pnt:
        if idx >= self._shadow.count() or idx < 0:
            raise IndexError()
        return Pnt.from_qt(self._shadow.at(idx))

    def __setitem__(self, idx: int, p: Union[Pnt, Tuple[float, float]]) -> None:
        self._shadow.set(idx, Pnt._fix_pnt(p).to_qt())

    if TYPE_CHECKING:

        def __iter__(self) -> Iterator[Pnt]: ...

        def __contains__(self, item: Pnt) -> bool: ...

    def insert(self, idx: int, p: Union[Pnt, Tuple[float, float]]) -> None:
        """Вставляет точку по индексу."""
        self._shadow.insert(idx, Pnt._fix_pnt(p).to_qt())

    def append(self, p: Union[Pnt, Tuple[float, float]]) -> None:
        """Добавляет точку в конец."""
        self._shadow.append(Pnt._fix_pnt(p).to_qt())

    def remove(self, idx: int) -> None:
        """Удаляет точку по индексу."""
        self._shadow.remove(idx)

    def pop(self, idx: int) -> Pnt:
        """Удаляет и возвращает точку."""
        return Pnt.from_qt(self._shadow.pop(idx))

    def __str__(self) -> str:
        return "".join(str(e) for e in self)

    def get_line_direction(self) -> LineDirection:
        """Возвращает направление точек."""
        return LineDirection(self._shadow.get_clockwise())

    def set_line_direction(self, direction: LineDirection) -> None:
        """Устанавливает направление точек."""
        self._shadow.set_clockwise(direction == LineDirection.Clockwise)

    @_experimental()
    def _get_determinant(self) -> float:
        return self._shadow.get_determinant()

    _shadow: ShadowPointsLS


# noinspection PyUnresolvedReferences
collections.abc.Collection.register(ListPoint)


class LineString(Geometry):
    """
    Геометрический объект типа полилиния.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_ls
        :lines: 2-
        :dedent: 4
    """

    @overload
    def __init__(
        self, points: Union[Iterable[Pnt], Iterable[Tuple[float, float]]], /, *, cs: Optional[CoordSystem] = None
    ) -> None:
        pass

    @overload
    def __init__(self, *points: Union[Pnt, Tuple[float, float]], cs: Optional[CoordSystem] = None) -> None:
        pass

    def __init__(self, *points, cs: Optional[CoordSystem] = None) -> None:  # type: ignore[no-untyped-def]
        """
        Конструктор класса.

        Args:
            points: Точки.
                Могут задаваться следующим образом:

                * В виде списка состоящего из объектов типа :class:`axipy.Pnt` или :class:`~typing.Tuple[float, float]`.
                * В виде перечня точек(:class:`axipy.Pnt` или :class:`~typing.Tuple[float, float]`) через запятую.

            cs: Система Координат, в которой создается геометрия.
                При задании, необходимо явно указать наименование параметра, cs=.

        """
        pars = _FixInputParams._fix_points_and_cs(points, cs)
        self._set_shadow(
            ShadowLineString(
                [Pnt._fix_pnt(p).to_qt() for p in _FixInputParams._fix_points_as_param(pars[0])],
                None if pars[1] is None else pars[1]._shadow,
            )
        )

    @classmethod
    @_experimental()
    def _from_iterable_pnt(cls, points: Iterable[Pnt], cs: Optional[CoordSystem] = None) -> "LineString":
        shadow: ShadowLineString = ShadowLineString(cls._iterable_pnt_to_qt(points), None if cs is None else cs._shadow)
        return cast(LineString, cls._wrap(shadow))

    @classmethod
    @_experimental()
    def _from_iterable_q_point(
        cls, points: Iterable[Union[QPoint, QPointF]], cs: Optional[CoordSystem] = None
    ) -> "LineString":
        shadow: ShadowLineString = ShadowLineString(list(points), None if cs is None else cs._shadow)
        return cast(LineString, cls._wrap(shadow))

    @cached_property
    def _points(self) -> ListPoint:
        return ListPoint(self._shadow)

    @property
    def points(self) -> ListPoint:
        """
        Точки полилинии. Реализован как список python :class:`list` точек :class:`Pnt`.
        Также поддерживаются список пар :class:`tuple`.

        Note:
            При обновлении значения точки допустимо только изменение ее заменой.
        """
        return self._points

    def __len__(self) -> int:
        return len(self.points)

    def __str__(self) -> str:
        return "{} (points={})".format(self._class_name, self.__len__()) + self._cs_str

    def get_line_direction(self) -> LineDirection:
        """Направление линии."""
        return LineDirection(self._shadow.get_clockwise())

    def set_line_direction(self, direction: LineDirection) -> None:
        """
        Задание направления линии.

        Args:
            direction: Желаемое значение
        """
        self._shadow.set_clockwise(direction == LineDirection.Clockwise)

    @_experimental()
    def _is_closed(self) -> bool:
        return self._shadow.is_closed()

    _shadow: ShadowLineString


class ListHoles:

    def __init__(self, poly: ShadowPolygon) -> None:
        self.m: ShadowPolygon = poly

    def __len__(self) -> int:
        return self.m.interiorCount()

    def __getitem__(self, idx: int) -> ListPoint:
        if idx >= self.m.interiorCount() or idx < 0:
            raise IndexError()
        return ListPoint(self.m.interior(idx))

    if TYPE_CHECKING:

        def __iter__(self) -> Iterator[ListPoint]: ...

        def __contains__(self, item: ListPoint) -> bool: ...

    def append(
        self,
        *points: Union[
            Pnt,
            Tuple[float, float],
            Iterable[Pnt],
            Iterable[Tuple[float, float]],
        ],
    ) -> None:
        """Добавляет точки как новую дырку."""
        self.m.addInterior([Pnt._fix_pnt(p).to_qt() for p in _FixInputParams._fix_points_as_param(points)])

    def remove(self, idx: int) -> None:
        """Удаляет дырку по индексу."""
        self.m.removeInterior(idx)


# noinspection PyUnresolvedReferences
collections.abc.Collection.register(ListHoles)


class Polygon(Geometry):
    """
    Геометрический объект типа полигон. Представляет собой часть плоскости, ограниченной
    замкнутой полилинией. Кроме внешней границы, полигон может иметь одну или несколько
    внутренних (дырок).

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_poly
        :lines: 2-
        :dedent: 4
    """

    @overload
    def __init__(
        self, points: Union[Iterable[Pnt], Iterable[Tuple[float, float]]], /, *, cs: Optional[CoordSystem] = None
    ) -> None:
        pass

    @overload
    def __init__(self, *points: Union[Pnt, Tuple[float, float]], cs: Optional[CoordSystem] = None) -> None:
        pass

    def __init__(self, *points, cs: Optional[CoordSystem] = None) -> None:  # type: ignore[no-untyped-def]
        """
        Конструктор класса.

        Args:
            points: Точки.
                Могут задаваться следующим образом:

                * В виде списка состоящего из объектов типа :class:`axipy.Pnt` или :class:`~typing.Tuple[float, float]`.
                * В виде перечня точек(:class:`axipy.Pnt` или :class:`~typing.Tuple[float, float]`) через запятую.

            cs: Система Координат, в которой создается геометрия.
                При задании, необходимо явно указать наименование параметра, cs=.

        """
        pars = _FixInputParams._fix_points_and_cs(points, cs)
        self._set_shadow(
            ShadowPolygon(
                [Pnt._point_value_to_qt(p) for p in _FixInputParams._fix_points_as_param(pars[0])],
                None if pars[1] is None else pars[1]._shadow,
            )
        )

    @classmethod
    @_experimental()
    def _from_iterable_pnt(cls, points: Iterable[Pnt], cs: Optional[CoordSystem] = None) -> "Polygon":
        shadow: ShadowPolygon = ShadowPolygon(cls._iterable_pnt_to_qt(points), None if cs is None else cs._shadow)
        return cast(Polygon, cls._wrap(shadow))

    @classmethod
    @_experimental()
    def _from_iterable_q_point(
        cls, points: Iterable[Union[QPoint, QPointF]], cs: Optional[CoordSystem] = None
    ) -> "Polygon":
        shadow: ShadowPolygon = ShadowPolygon(list(points), None if cs is None else cs._shadow)
        return cast(Polygon, cls._wrap(shadow))

    @staticmethod
    def from_rect(rect: Rect, cs: Optional[CoordSystem] = None) -> "Polygon":
        """
        Создает полигон на базе прямоугольника.

        Args:
            rect: Прямоугольник, на основе которого формируются координаты.
            cs: Система Координат, в которой создается геометрия.
        """
        return Polygon(
            (rect.xmin, rect.ymin),
            (rect.xmin, rect.ymax),
            (rect.xmax, rect.ymax),
            (rect.xmax, rect.ymin),
            cs=cs,
        )

    @cached_property
    def _points(self) -> ListPoint:
        return ListPoint(self._shadow.exterior())

    @cached_property
    def _holes(self) -> ListHoles:
        return ListHoles(self._shadow)

    @property
    def points(self) -> ListPoint:
        """
        Точки полигона.

        Реализован как список python :class:`list` точек :class:`Pnt`.
        В каждом контуре, количество точек на 1 больше, чем количество узлов, так как контур - это замкнутая линия,
        и первая и последняя точка совпадают.
        """
        return self._points

    @property
    def holes(self) -> ListHoles:
        """
        Дырки полигона. Реализован в виде списка списков точек.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_poly_holes
            :lines: 2-
            :dedent: 4
        """
        return self._holes

    def get_line_direction(self) -> LineDirection:
        """Направление линии."""
        return LineDirection(self._shadow.get_clockwise())

    def set_line_direction(self, direction: LineDirection) -> None:
        """
        Задание направления линии.

        Args:
            direction: Желаемое значение
        """
        self._shadow.set_clockwise(direction == LineDirection.Clockwise)

    def __len__(self) -> int:
        return len(self.points)

    def __str__(self) -> str:
        return (
            "{} (points={}, holes={})".format(self._class_name, self.__len__(), len(self.holes) if self.holes else 0)
            + self._cs_str
        )

    _shadow: ShadowPolygon


class GeometryCollection(Geometry):
    """
    Коллекция разнотипных геометрических объектов. Допустимо хранение геометрических
    объектов различного типа, за исключением коллекций. Доступ к элементам производится
    по аналогии работы со списком :class:`list`.

    Args:
        cs: Система Координат, в которой создается геометрия.


    Для получения размера коллекции используйте функцию len::

        cnt = len(coll)

    Доступ к элементам производится по индексу. Нумерация начинается с 0.

    В качестве примера получим первый элемент::

        coll[1]

    Обновление геометрии в коллекции так же производится по ее индексу.
    Также допустимо изменение некоторых свойств геометрии в зависимости от ее типа.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :pyobject: test_run_example_geometry_coll
        :lines: 9-
        :dedent: 4
    """

    def __init__(self, cs: Optional[CoordSystem] = None) -> None:
        self._set_shadow(ShadowCollection(None if cs is None else cs._shadow))

    def __len__(self) -> int:
        return self._shadow.count()

    def _fix_tuple(self, t: tuple) -> Union[Point, LineString, Polygon]:
        if len(t) == 1:
            return Point(t[0][0], t[0][1], cs=self.coordsystem)
        return LineString(*t, cs=self.coordsystem)

    def __fix_args(self, g: tuple) -> Geometry:
        """
        Input arguments tuple is either:
            - tuple of multiple single-geom-like objects
            - tuple of a single container with single-geom-like objects
            - tuple of 2 floats or ints (single point)
        """
        if len(g) == 1:
            gg = g[0]
        else:
            gg = self._fix_tuple(_FixInputParams._fix_points_as_param(g))

        if isinstance(gg, GeometryCollection):
            raise TypeError("Allow only simple Geometry type.")
        elif isinstance(gg, Geometry):
            return gg
        elif isinstance(gg, tuple) and len(gg) == 2:
            return Point(gg[0], gg[1], self.coordsystem)
        elif isinstance(gg, list) and len(gg) >= 2:
            # TODO: refactor
            return self._fix_tuple(gg)  # type: ignore[arg-type]
        elif isinstance(gg, Pnt):
            return Point(gg.x, gg.y, self.coordsystem)
        raise TypeError("Allow only Geometry type.")

    def _fix_args_append(self: Optional["GeometryCollection"] = None) -> Callable:
        def decorator(func: Callable) -> Callable:
            @functools.wraps(func)
            def wrapper(inst: "GeometryCollection", value: Union[Point, LineString, Polygon], *args: Any) -> Any:
                args_fixed = tuple((value, *args))
                return func(inst, args_fixed)

            return wrapper

        return decorator

    @_fix_args_append()
    def append(self, value: Union[Point, LineString, Polygon]) -> None:
        """
        Добавление геометрии в коллекцию. Задание параметров аналогично указанию их при
        создании объектов конкретного типа. Если опустить указание класса, то для одной
        точки или пары значений float,

        будет создан точечный объект :class:`Point`,
        если точек больше, - объект класса :class:`LineString`.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_coll
            :lines: 2-8
            :dedent: 4
        """
        # Will always be tuple, because of the backwards compatibility decorator
        g = self.__fix_args(value)  # type: ignore[arg-type]
        self._shadow.append(g._shadow)

    def remove(self, idx: int) -> None:
        """
        Удаление геометрии из коллекции.

        Args:
            idx: Индекс геометрии в коллекции.
        """
        self._shadow.remove(idx)

    def __getitem__(self, idx: int) -> Union[Point, LineString, Polygon]:
        if idx >= len(self) or idx < 0:
            raise IndexError()
        # type enforced by logic
        return Geometry._wrap(ShadowCollection.at(self._shadow, idx))  # type: ignore[return-value]

    if TYPE_CHECKING:

        def __iter__(self) -> Iterator[Union[Point, LineString, Polygon]]: ...

        def __contains__(self, item: Union[Point, LineString, Polygon]) -> bool: ...

    def _fix_args_set_item(self: Optional["GeometryCollection"] = None) -> Callable:
        def decorator(func: Callable) -> Callable:
            @functools.wraps(func)
            def wrapper(
                inst: "GeometryCollection", idx: int, value: Union[Point, LineString, Polygon], *args: Any
            ) -> Any:
                args_fixed = tuple((value, *args))
                return func(inst, idx, args_fixed)

            return wrapper

        return decorator

    @_fix_args_set_item()
    def __setitem__(self, idx: int, value: Union[Point, LineString, Polygon]) -> None:
        if idx >= len(self) or idx < 0:
            raise IndexError()
        # Will always be tuple, because of the backwards compatibility decorator
        g = self.__fix_args(value)  # type: ignore[arg-type]
        self._shadow.set(idx, g._shadow)

    def __repr__(self) -> str:
        try:
            return self.to_wkt()
        except Exception:
            return ", ".join(g.__repr__() for g in self)

    def __str__(self) -> str:
        return "{} (objects={})".format(self._class_name, self.__len__()) + self._cs_str

    def try_to_simplified(
        self,
    ) -> Union["MultiPoint", "MultiLineString", "MultiPolygon", "GeometryCollection"]:
        """
        Создает копию коллекции при этом пробует упростить ее тип.

        К примеру, если в коллекции только полигоны, то вернет объект типа :class:`MultiPolygon`.
        Если исходная коллекция разнородна, возвращается копия исходной.
        """
        polygons: int = 0
        points: int = 0
        lines: int = 0

        for o in self:
            polygons += isinstance(o, Polygon)
            points += isinstance(o, Point)
            lines += isinstance(o, (Line, LineString))

        res: Union["MultiPoint", "MultiLineString", "MultiPolygon", "GeometryCollection"]
        if polygons > 0 and points == 0 and lines == 0:
            res = MultiPolygon(cs=self.coordsystem)
        elif polygons == 0 and points > 0 and lines == 0:
            res = MultiPoint(cs=self.coordsystem)
        elif polygons == 0 and points == 0 and lines > 0:
            res = MultiLineString(cs=self.coordsystem)
        else:
            # type enforced by logic
            return self.clone()  # type: ignore[return-value]

        for o in self:
            res.append(o)
        return res

    _shadow: ShadowCollection


# noinspection PyUnresolvedReferences
collections.abc.Collection.register(GeometryCollection)


class MultiPoint(GeometryCollection):
    """
    Коллекция точечных объектов. Может содержать только объекты типа точка.

    Args:
        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_coll_point
        :lines: 2-
        :dedent: 4
    """

    # noinspection PyMissingConstructor
    def __init__(self, cs: Optional[CoordSystem] = None) -> None:
        self._set_shadow(ShadowMultiPoint(None if cs is None else cs._shadow))

    if TYPE_CHECKING:

        def __iter__(self) -> Iterator[Point]: ...

    _shadow: ShadowMultiPoint


class MultiLineString(GeometryCollection):
    """
    Коллекция полилиний. Может содержать только объекты типа полилиния.

    Args:
        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_coll_ls
        :lines: 2-
        :dedent: 4
    """

    # noinspection PyMissingConstructor
    def __init__(self, cs: Optional[CoordSystem] = None) -> None:
        self._set_shadow(ShadowMultiLineString(None if cs is None else cs._shadow))

    if TYPE_CHECKING:

        def __iter__(self) -> Iterator[LineString]: ...

    _shadow: ShadowMultiLineString


class MultiPolygon(GeometryCollection):
    """
    Коллекция полигонов. Может содержать только объекты типа полигон.

    Args:
        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_coll_poly
        :lines: 2-
        :dedent: 4
    """

    # noinspection PyMissingConstructor
    def __init__(self, cs: Optional[CoordSystem] = None) -> None:
        self._set_shadow(ShadowMultiPolygon(None if cs is None else cs._shadow))

    if TYPE_CHECKING:

        def __iter__(self) -> Iterator[Polygon]: ...

    # TODO: refactor
    def _fix_tuple(self, g) -> Union[Point, LineString, Polygon]:  # type: ignore[no-untyped-def]
        return Polygon(g, cs=self.coordsystem)

    _shadow: ShadowMultiPolygon


def _apply_deprecated() -> None:
    # Using getattr to hide deprecated objects from axipy namespace on IDE inspections
    getattr(__all__, "extend")(("Type",))

    @_deprecated_by("axipy.Geometry.to_wkt")
    def wkt(self: Geometry) -> str:
        return self._shadow.to_wkt()

    @_deprecated_by("axipy.Geometry.to_wkb")
    def wkb(self: Geometry) -> bytes:
        return bytes(self._shadow.to_wkb())

    setattr(Geometry, "wkt", property(wkt))
    setattr(Geometry, "wkb", property(wkb))

    globals().update(
        Type=GeometryType,
    )


_apply_deprecated()
