from typing import Iterable, List, Tuple, Union, overload

from axipy.cpp_cs import ShadowCoordinateTransformer
from axipy.utl import Pnt, Rect
from PySide2.QtCore import QPointF, QRectF

from .coord_system import CoordSystem, _internal_transform

__all__: List[str] = [
    "CoordTransformer",
]


class CoordTransformer:
    """
    Класс для преобразования координат из одной СК в другую. При создании объекта
    трансформации в него передается исходная и целевая СК. После этого данный объект
    может использоваться для преобразования данных между этими СК.

    .. literalinclude:: /../../tests/doc_examples/cs/test_example_cs.py
        :caption: Пример преобразования точки
        :pyobject: test_run_example_doc_common_example
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, cs_from: CoordSystem, cs_to: CoordSystem) -> None:
        """
        Конструктор класса.

        Args:
            cs_from: Исходная СК.
            cs_to: Целевая СК.
        """
        cs_from = self.__ensure_cs(cs_from)
        cs_to = self.__ensure_cs(cs_to)

        if not isinstance(cs_from, CoordSystem):
            raise Exception("Invalid parameter cs_from")
        if not isinstance(cs_to, CoordSystem):
            raise Exception("Invalid parameter cs_to")
        self._shadow = ShadowCoordinateTransformer(cs_from._shadow, cs_to._shadow)

    def __ensure_cs(self, cs: CoordSystem) -> CoordSystem:
        # backwards compatibility for Union[CoordSystem, str]
        if isinstance(cs, str):
            return CoordSystem.from_string(cs)
        return cs

    @overload
    def transform(self, value: Union[QRectF, Rect]) -> Rect: ...

    @overload
    def transform(self, value: Union[Tuple[float, float], Pnt, QPointF]) -> Pnt: ...

    @overload
    def transform(self, value: Iterable[Union[Tuple[float, float], Pnt, QPointF]]) -> List[Pnt]: ...

    def transform(
        self,
        value: Union[
            QRectF, Rect, Tuple[float, float], Pnt, QPointF, Iterable[Union[Tuple[float, float], Pnt, QPointF]]
        ],
    ) -> Union[Rect, Pnt, List[Pnt]]:
        """
        Преобразовывает точки из исходной СК в целевую СК.

        Args:
            value: Входное значение. Может быть точкой, массивом точек :class:`axipy.Pnt`
                или :class:`axipy.Rect`.

        Returns:
            Выходное значение. Тип зависит от входного и аналогичен ему.

        Raises:
            RuntimeError: Ошибка выполнения преобразования.
        """
        return _internal_transform(value, self._shadow.transform)

    @classmethod
    def proj_transform_definition(cls, cs_from: str, cs_to: str) -> str:
        """
        Возвращает строку трансформации (pipeline) для преобразования между двумя СК,
        заданными в формате proj.

        Args:
            cs_from: Строка с определением исходной СК в формате proj
            cs_to: Строка с определением исходной СК в формате proj

        Returns:
            Строка с определением трансформации между двумя этими СК в формате proj

        .. literalinclude:: /../../tests/doc_examples/cs/test_example_cs.py
            :caption: Пример получения строки трансформации
            :pyobject: test_run_example_doc_proj_transform
            :lines: 2-
            :dedent: 4
        """
        return ShadowCoordinateTransformer.proj_transform_definition(cs_from, cs_to)
