import functools
import inspect
import os.path
from typing import Any, Callable, Iterable, List, Optional, Union

from axipy.da import Observer
from axipy.gui import MapTool, gui_instance, view_manager
from axipy.menubar import (
    ActionButton,
    Button,
    Position,
    Separator,
    SystemActionButton,
    ToolButton,
)
from PySide2.QtCore import QCoreApplication
from PySide2.QtGui import QIcon

from .interface import AxiomaInterface

__all__: List[str] = [
    "AxiomaPlugin",
    "tr",
    "local_file",
]


@functools.lru_cache(maxsize=None)
def tr(text: str):
    """
    Ищет перевод строки строки.

    Производит поиск строки в загруженных файлах перевода.

    Args:
        text:
            Строка для перевода.

    Returns:
        Перевод стоки, если строка найдена. Иначе - сама переданная строка.

    Пример::

        button_name = tr('My button')
    """
    try:
        frm = inspect.stack()[1]
        mod = inspect.getmodule(frm[0])
        module_name = mod.__name__
        base_module_name = module_name.split(".", 1)[0]
        return QCoreApplication.translate(base_module_name, text)
    except (Exception,):
        return text


@functools.lru_cache(maxsize=None)
def local_file(*paths: Iterable[str]) -> str:
    try:
        frm = inspect.stack()[1]
        mod = inspect.getmodule(frm[0])
        root_dir = os.path.dirname(mod.__file__)
        return os.path.join(root_dir, *paths)
    except (Exception,):
        return ""


class AxiomaPlugin(AxiomaInterface):
    """
    Модуль для ГИС Аксиома.

    Содержит вспомогательные функции и свойства, которые могут быть использованы
    при реализации пользовательского модуля.

    Note:
        Не переопределяйте конструктор. Переопределяйте метод :meth:`load`.

    See also:
        Подробнее в главе :ref:`to_modules`.
    """

    @classmethod
    def _axioma_plugin_create(cls, plugin_dir):
        result = cls()
        result.initialize(plugin_dir)
        result._axioma_plugin_load()
        view_manager.mainwindow_activated.connect(result.load_mainwindow)
        return result

    def _axioma_plugin_load(self):
        self.load()

    def _axioma_plugin_unload(self):
        self.unload()

    def load(self):
        """
        Загружает модуль.

        Переопределяйте этот метод для задания логики модуля.
        """
        pass

    def load_mainwindow(self):
        """
        Вызывается при инициализации объекта главного окна приложения.

        Переопределяйте это метод если есть необходимость доступаться к главному окну
        приложения при автоматической загрузке плагина.
        """
        pass

    def unload(self):
        """
        Выгружает модуль.

        Переопределяйте этот метод для очистки ресурсов.
        """
        pass

    def create_tool(
        self,
        title: str,
        on_click: Callable[[], MapTool],
        icon: Union[str, QIcon] = "",
        enable_on: Optional[Union[str, Observer]] = None,
        tooltip: Optional[str] = None,
        doc_file: Optional[str] = None,
    ) -> ToolButton:
        """
        Создает кнопку с инструментом.

        Args:
            title: Текст.
            on_click: Класс инструмента.
            icon: Иконка. Может быть путем к файлу или адресом ресурса.
            enable_on: Идентификатор наблюдателя для определения доступности кнопки.
            tooltip: Строка с дополнительной короткой информацией по данному действию.
            doc_file: Относительная ссылка на файл документации. Расположение рассматривается по отношению к
              каталогу `documentation`.
        Return:
            Кнопка с инструментом.

        See also:
            :class:`axipy.ObserverManager`.

        Note:
            То же, что и :class:`ToolButton`, но дополнительно
            делает идентификатор кнопки уникальным для данного модуля.
        """
        result = ToolButton(title, on_click, icon, enable_on, tooltip)
        if doc_file is not None:
            result.action.setWhatsThis(self.__full_doc_file(doc_file))
        self._disambiguate(result)
        return result

    def __full_doc_file(self, filename) -> str:
        return "file:///{}".format(self.local_file("documentation", filename).replace("\\", "/"))

    def create_separator(self) -> Separator:
        """Создает разделитель."""
        return Separator()

    def create_action(
        self,
        title: str,
        on_click: Callable[[], Any],
        icon: Union[str, QIcon] = "",
        enable_on: Optional[Observer] = None,
        tooltip: Optional[str] = None,
        doc_file: Optional[str] = None,
    ) -> ActionButton:
        """
        Создает кнопку с действием.

        Args:
            title: Текст.
            on_click: Действие на нажатие.
            icon: Иконка. Может быть путем к файлу или адресом ресурса.
            enable_on: Идентификатор наблюдателя для определения доступности кнопки.
            tooltip: Строка с дополнительной короткой информацией по данному действию.
            doc_file: Относительная ссылка на файл документации. Расположение рассматривается по отношению к
              каталогу `documentation`.
        Return:
            Кнопка с действием.

        See also:
            :class:`axipy.ObserverManager`.

        Note:
            То же, что и :class:`ActionButton`, но дополнительно
            делает идентификатор кнопки уникальным для данного модуля.
        """
        result = ActionButton(title, on_click, icon, enable_on, tooltip=tooltip)
        if doc_file is not None:
            result.action.setWhatsThis(self.__full_doc_file(doc_file))
        self._disambiguate(result)
        return result

    def create_system_action(self, id: str) -> SystemActionButton:
        """
        Создает кнопку с действием, основанном на уже зарегистрированном в системе
        действии.

        Args:
            id: Идентификатор действия. Подробнее см. :class:`axipy.ActionManager`
        """
        result = SystemActionButton(id)
        return result

    def _disambiguate(self, button: Button):
        button.action.setObjectName(self._make_action_id(button.action.text()))

    def get_position(self, tab: str, group: str) -> Position:
        """
        Возвращает положение в меню. Может заранее не существовать.

        Args:
            tab: Название вкладки.
            group: Название группы.

        Return:
            Положение для кнопки.

        Note:
            Дублирует :class:`axipy.menubar.Position`.
        """
        return self.menubar.get_position(tab, group)

    @property
    def __module_name(self):
        try:
            frm = inspect.stack()[1]
            mod = inspect.getmodule(frm[0])
            return mod.__name__
        except (Exception,):
            return ""

    def user_plugin_data_dir(self, file_name: str = "") -> str:
        """
        Возвращает каталог, в котором находится изменяемые данные модуля. Расположение
        определяется в подкаталоге `installed_plugins_data`, находящемся на один уровень
        вверх по отношению к каталогу самого модуля :meth:`plugin_dir`.

        Args:
            file_name: Если параметр указан, возвращается полный путь файла в данном каталоге.

        Note:

            Если необходимо создать данный каталог::

                from pathlib import Path
                Path(self.user_plugin_data_dir()).mkdir(parents=True, exist_ok=True)
        """
        from pathlib import Path

        res = Path(self.user_plugin_dir())
        res = os.path.join(
            os.path.join(res.parent.absolute(), "installed_plugins_data"),
            os.path.basename(self._plugin_dir),
        )
        if file_name:
            return os.path.join(res, file_name)
        return res

    def user_plugin_dir(self, file_name: str = "") -> str:
        """
        Возвращает каталог данного модуля.

        Args:
            file_name: Если параметр указан, возвращается полный путь файла в каталоге модуля.
        """
        res = os.path.join(
            gui_instance._shadow.installedPluginsPath(),
            os.path.basename(self._plugin_dir),
        )
        if file_name:
            return os.path.join(res, file_name)
        return res
