import functools
from typing import Callable, Any

import axipy
from osgeo import gdal


def _gdal_exceptions_decor(use_exceptions: bool = True) -> Callable:
    """
    Decorator to set desired GDAL exception state and restore it after function ran.

    Args:
        use_exceptions (bool): If True, enable exceptions; if False, disable exceptions.
    """

    def decorator(func: Callable) -> Callable:
        @functools.wraps(func)
        def inner(*args, **kwargs) -> Any:
            # Capture the initial state of GDAL exceptions
            initial_state = gdal.GetUseExceptions()
            if initial_state:  # If UseExceptions is already set, returning.
                return func(*args, **kwargs)

            # Set the desired exception state
            if use_exceptions:
                gdal.UseExceptions()
            else:
                gdal.DontUseExceptions()

            try:
                # Run the decorated function
                return func(*args, **kwargs)
            finally:
                # Restore the initial state
                if initial_state:
                    gdal.UseExceptions()
                else:
                    gdal.DontUseExceptions()

        return inner

    return decorator


def _run_in_gui_decor(func: Callable) -> Callable:
    @functools.wraps(func)
    def inner(*args, **kwargs) -> Any:
        return axipy.run_in_gui(func, *args, **kwargs)

    return inner
